"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.appendToTimer = appendToTimer;
exports.removeFromTimer = removeFromTimer;
/**
 * @module node-opcua-server
 */
const node_opcua_assert_1 = require("node-opcua-assert");
const node_opcua_debug_1 = require("node-opcua-debug");
const node_opcua_types_1 = require("node-opcua-types");
const node_opcua_utils_1 = require("node-opcua-utils");
const debugLog = (0, node_opcua_debug_1.make_debugLog)(__filename);
const doDebug = (0, node_opcua_debug_1.checkDebugFlag)(__filename);
const timers = {};
const NS_PER_SEC = 1e9;
function sampleMonitoredItem(monitoredItem) {
    const _monitoredItem = monitoredItem;
    if (monitoredItem.monitoringMode === node_opcua_types_1.MonitoringMode.Disabled) {
        return;
    }
    setImmediate(() => {
        _monitoredItem._on_sampling_timer();
    });
}
function appendToTimer(monitoredItem) {
    const samplingInterval = monitoredItem.samplingInterval;
    const key = samplingInterval.toString();
    (0, node_opcua_assert_1.assert)(samplingInterval > 0);
    let _t = timers[key];
    if (!_t) {
        _t = {
            _samplingId: false,
            monitoredItems: {},
            monitoredItemsCount: 0
        };
        _t._samplingId = setInterval(() => {
            const start = doDebug ? (0, node_opcua_utils_1.hrtime)() : undefined;
            let counter = 0;
            for (const m in _t.monitoredItems) {
                if (Object.prototype.hasOwnProperty.call(_t.monitoredItems, m)) {
                    sampleMonitoredItem(_t.monitoredItems[m]);
                    counter++;
                }
            }
            // istanbul ignore next
            if (doDebug) {
                const elapsed = (0, node_opcua_utils_1.hrtime)(start);
                debugLog(`Sampler ${samplingInterval}  ms : Benchmark took ${((elapsed[0] * NS_PER_SEC + elapsed[1]) /
                    1000 /
                    1000.0).toFixed(3)} milliseconds for ${counter} elements`);
            }
        }, samplingInterval);
        timers[key] = _t;
    }
    (0, node_opcua_assert_1.assert)(!_t.monitoredItems[monitoredItem.monitoredItemId]);
    _t.monitoredItems[monitoredItem.monitoredItemId] = monitoredItem;
    _t.monitoredItemsCount++;
    return key;
}
function removeFromTimer(monitoredItem) {
    const samplingInterval = monitoredItem.samplingInterval;
    (0, node_opcua_assert_1.assert)(samplingInterval > 0);
    (0, node_opcua_assert_1.assert)(typeof monitoredItem._samplingId === "string");
    const key = monitoredItem._samplingId;
    const _t = timers[key];
    if (!_t) {
        debugLog("cannot find common timer for samplingInterval", key);
        return;
    }
    (0, node_opcua_assert_1.assert)(_t);
    (0, node_opcua_assert_1.assert)(_t.monitoredItems[monitoredItem.monitoredItemId]);
    delete _t.monitoredItems[monitoredItem.monitoredItemId];
    _t.monitoredItemsCount--;
    (0, node_opcua_assert_1.assert)(_t.monitoredItemsCount >= 0);
    if (_t.monitoredItemsCount === 0) {
        if (_t._samplingId !== false) {
            clearInterval(_t._samplingId);
        }
        delete timers[key];
    }
}
//# sourceMappingURL=node_sampler.js.map