"use strict";
var __importDefault = (this && this.__importDefault) || function (mod) {
    return (mod && mod.__esModule) ? mod : { "default": mod };
};
Object.defineProperty(exports, "__esModule", { value: true });
exports.TokenStack = void 0;
const node_opcua_assert_1 = require("node-opcua-assert");
const node_opcua_debug_1 = require("node-opcua-debug");
const chalk_1 = __importDefault(require("chalk"));
const debugLog = (0, node_opcua_debug_1.make_debugLog)("TOKEN");
const doDebug = (0, node_opcua_debug_1.checkDebugFlag)("TOKEN");
const warningLog = (0, node_opcua_debug_1.make_warningLog)("TOKEN");
function hasTokenReallyExpired(token) {
    const now = new Date();
    const age = now.getTime() - token.createdAt.getTime();
    return age > token.revisedLifetime * 1.25;
}
class TokenStack {
    #tokenStack = [];
    #clientKeyProvider;
    #serverKeyProvider;
    id = 0;
    constructor(channelId) {
        this.id = channelId;
        this.#clientKeyProvider = {
            getDerivedKey: (tokenId) => {
                const d = this.getTokenDerivedKeys(tokenId);
                if (!d)
                    return null;
                return d.derivedClientKeys;
            }
        };
        this.#serverKeyProvider = {
            getDerivedKey: (tokenId) => {
                const d = this.getTokenDerivedKeys(tokenId);
                if (!d)
                    return null;
                return d.derivedServerKeys;
            }
        };
    }
    serverKeyProvider() { return this.#serverKeyProvider; }
    clientKeyProvider() { return this.#clientKeyProvider; }
    pushNewToken(securityToken, derivedKeys) {
        this.removeOldTokens();
        // TODO: make sure this list doesn't grow indefinitely
        const _tokenStack = this.#tokenStack;
        (0, node_opcua_assert_1.assert)(_tokenStack.length === 0 || _tokenStack[0].securityToken.tokenId !== securityToken.tokenId);
        _tokenStack.push({
            derivedKeys,
            securityToken
        });
        /* istanbul ignore next */
        if (doDebug) {
            debugLog("id=", this.id, chalk_1.default.cyan("Pushing new token with id "), securityToken.tokenId, this.tokenIds());
        }
    }
    tokenIds() {
        return this.#tokenStack.map((a) => a.securityToken.tokenId);
    }
    getToken(tokenId) {
        const token = this.#tokenStack.find((a) => a.securityToken.tokenId === tokenId);
        if (!token)
            return null;
        return token.securityToken;
    }
    getTokenDerivedKeys(tokenId) {
        const token = this.#tokenStack.find((a) => a.securityToken.tokenId === tokenId);
        if (!token)
            return null;
        if (hasTokenReallyExpired(token.securityToken)) {
            return null;
        }
        return token.derivedKeys;
    }
    removeOldTokens() {
        // remove all expired tokens
        this.#tokenStack = this.#tokenStack.filter((token) => !hasTokenReallyExpired(token.securityToken));
    }
}
exports.TokenStack = TokenStack;
//# sourceMappingURL=token_stack.js.map