"use strict";
/**
 * @module node-opcua-secure-channel
 */
// tslint:disable:max-line-length
Object.defineProperty(exports, "__esModule", { value: true });
exports.MessageChunker = void 0;
const node_opcua_assert_1 = require("node-opcua-assert");
const node_opcua_basic_types_1 = require("node-opcua-basic-types");
const node_opcua_status_code_1 = require("node-opcua-status-code");
const node_opcua_binary_stream_1 = require("node-opcua-binary-stream");
const node_opcua_service_secure_channel_1 = require("node-opcua-service-secure-channel");
const node_opcua_utils_1 = require("node-opcua-utils");
const node_opcua_debug_1 = require("node-opcua-debug");
const node_opcua_chunkmanager_1 = require("node-opcua-chunkmanager");
const secure_message_chunk_manager_1 = require("./secure_message_chunk_manager");
const sequence_number_generator_1 = require("./sequence_number_generator");
const doTraceChunk = process.env.NODEOPCUADEBUG && process.env.NODEOPCUADEBUG.indexOf("CHUNK") >= 0;
const errorLog = (0, node_opcua_debug_1.make_errorLog)("secure_channel");
const warningLog = (0, node_opcua_debug_1.make_warningLog)("secure_channel");
class MessageChunker {
    static defaultMaxMessageSize = 16 * 1024 * 1024;
    static defaultChunkCount = 0; // 0 => no limits
    maxMessageSize;
    maxChunkCount;
    securityMode;
    #sequenceNumberGenerator = new sequence_number_generator_1.SequenceNumberGenerator();
    constructor(options) {
        options = options || { securityMode: node_opcua_service_secure_channel_1.MessageSecurityMode.Invalid };
        this.securityMode = options.securityMode || node_opcua_service_secure_channel_1.MessageSecurityMode.None;
        this.maxMessageSize = options.maxMessageSize || MessageChunker.defaultMaxMessageSize;
        this.maxChunkCount = options.maxChunkCount === undefined ? MessageChunker.defaultChunkCount : options.maxChunkCount;
    }
    dispose() { }
    #makeAbandonChunk(params) {
        const finalC = "A";
        const msgType = "MSG";
        const buffer = Buffer.alloc(
        // MSGA
        4 +
            // length
            4 +
            // secureChannelId
            4 +
            // tokenId
            4 +
            2 * 4);
        const stream = new node_opcua_binary_stream_1.BinaryStream(buffer);
        // message header --------------------------
        // ---------------------------------------------------------------
        // OPC UA Secure Conversation Message Header : Part 6 page 36
        // MessageType     Byte[3]
        // IsFinal         Byte[1]  C : intermediate, F: Final , A: Final with Error
        // MessageSize     UInt32   The length of the MessageChunk, in bytes. This value includes size of the message header.
        // SecureChannelId UInt32   A unique identifier for the ClientSecureChannelLayer assigned by the server.
        stream.writeUInt8(msgType.charCodeAt(0));
        stream.writeUInt8(msgType.charCodeAt(1));
        stream.writeUInt8(msgType.charCodeAt(2));
        stream.writeUInt8(finalC.charCodeAt(0));
        stream.writeUInt32(0); // will be written later
        stream.writeUInt32(params.channelId || 0); // secure channel id
        const securityHeader = params.securityHeader ||
            new node_opcua_service_secure_channel_1.SymmetricAlgorithmSecurityHeader({
                tokenId: 0
            });
        securityHeader.encode(stream);
        const sequenceHeader = new node_opcua_chunkmanager_1.SequenceHeader({
            sequenceNumber: this.#sequenceNumberGenerator.next(),
            requestId: params.securityOptions.requestId /// fix me
        });
        sequenceHeader.encode(stream);
        // write chunk length
        const length = stream.length;
        stream.length = 4;
        stream.writeUInt32(length);
        stream.length = length;
        return buffer;
    }
    #_build_chunk_manager(msgType, params) {
        let securityHeader = params.securityHeader;
        if (msgType === "OPN") {
            (0, node_opcua_assert_1.assert)(securityHeader instanceof node_opcua_service_secure_channel_1.AsymmetricAlgorithmSecurityHeader);
        }
        else if (msgType === "MSG") {
            (0, node_opcua_assert_1.assert)(securityHeader instanceof node_opcua_service_secure_channel_1.SymmetricAlgorithmSecurityHeader);
        }
        const channelId = params.channelId;
        const mode = this.securityMode;
        const chunkManager = new secure_message_chunk_manager_1.SecureMessageChunkManager(mode, msgType, channelId, params.securityOptions, securityHeader, this.#sequenceNumberGenerator);
        return chunkManager;
    }
    prepareChunk(msgType, params, messageLength) {
        // calculate message size ( with its  encodingDefaultBinary)
        try {
            const chunkManager = this.#_build_chunk_manager(msgType, params);
            const { chunkCount, totalLength } = chunkManager.evaluateTotalLengthAndChunks(messageLength);
            if (this.maxChunkCount > 0 && chunkCount > this.maxChunkCount) {
                errorLog(`[NODE-OPCUA-E10] message chunkCount ${chunkCount} exceeds the negotiated maximum chunk count ${this.maxChunkCount}, message current size is ${totalLength}`);
                errorLog(`[NODE-OPCUA-E10] ${messageLength} totalLength = ${totalLength} chunkManager.maxBodySize = ${this.maxMessageSize}`);
                return { statusCode: node_opcua_status_code_1.StatusCodes.BadTcpMessageTooLarge, chunkManager: null };
            }
            if (this.maxMessageSize > 0 && totalLength > this.maxMessageSize) {
                errorLog(`[NODE-OPCUA-E11] message size ${totalLength} exceeds the negotiated message size ${this.maxMessageSize} nb chunks ${chunkCount}`);
                return { statusCode: node_opcua_status_code_1.StatusCodes.BadTcpMessageTooLarge, chunkManager: null };
            }
            return { statusCode: node_opcua_status_code_1.StatusCodes.Good, chunkManager: chunkManager };
        }
        catch (err) {
            return { statusCode: node_opcua_status_code_1.StatusCodes.BadTcpInternalError, chunkManager: null };
        }
    }
    chunkSecureMessage(msgType, params, message, messageChunkCallback) {
        const calculateMessageLength = (message) => {
            const stream = new node_opcua_binary_stream_1.BinaryStreamSizeCalculator();
            (0, node_opcua_basic_types_1.encodeExpandedNodeId)(message.schema.encodingDefaultBinary, stream);
            message.encode(stream);
            return stream.length;
        };
        // evaluate the message size
        const messageLength = calculateMessageLength(message);
        const { statusCode, chunkManager } = this.prepareChunk(msgType, params, messageLength);
        if (statusCode !== node_opcua_status_code_1.StatusCodes.Good) {
            return statusCode;
        }
        if (!chunkManager) {
            return node_opcua_status_code_1.StatusCodes.BadInternalError;
        }
        let nbChunks = 0;
        let totalSize = 0;
        chunkManager.on("chunk", (messageChunk) => {
            nbChunks++;
            totalSize += messageChunk.length;
            messageChunkCallback(messageChunk);
        })
            .on("finished", () => {
            if (doTraceChunk) {
                console.log((0, node_opcua_utils_1.timestamp)(), "   <$$ ", msgType, "nbChunk = " + nbChunks.toString().padStart(3), "totalLength = " + totalSize.toString().padStart(8), "l=", messageLength.toString().padStart(6), "maxChunkCount=", this.maxChunkCount, "maxMessageSize=", this.maxMessageSize);
            }
            messageChunkCallback(null);
        });
        // create buffer to stream 
        const stream = new node_opcua_binary_stream_1.BinaryStream(messageLength);
        (0, node_opcua_basic_types_1.encodeExpandedNodeId)(message.schema.encodingDefaultBinary, stream);
        message.encode(stream);
        // inject buffer to chunk manager
        chunkManager.write(stream.buffer, stream.buffer.length);
        chunkManager.end();
        return node_opcua_status_code_1.StatusCodes.Good;
    }
}
exports.MessageChunker = MessageChunker;
//# sourceMappingURL=message_chunker.js.map