"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.findActiveConditions = findActiveConditions;
exports.acknowledgeAllConditions = acknowledgeAllConditions;
exports.confirmAllConditions = confirmAllConditions;
const node_opcua_nodeid_1 = require("node-opcua-nodeid");
const node_opcua_service_filter_1 = require("node-opcua-service-filter");
const node_opcua_service_read_1 = require("node-opcua-service-read");
const node_opcua_status_code_1 = require("node-opcua-status-code");
const node_opcua_variant_1 = require("node-opcua-variant");
const node_opcua_debug_1 = require("node-opcua-debug");
const event_stuff_1 = require("./event_stuff");
const extract_condition_fields_1 = require("./extract_condition_fields");
const call_method_condition_1 = require("./call_method_condition");
const call_condition_refresh_1 = require("./call_condition_refresh");
const doDebug = (0, node_opcua_debug_1.checkDebugFlag)(__filename);
const debugLog = (0, node_opcua_debug_1.make_debugLog)(__filename);
const errorLog = (0, node_opcua_debug_1.make_errorLog)(__filename);
/**
 *
 * @param session
 * @param eventStuff
 * @param comment
 */
async function acknowledgeConditionEV(session, eventStuff, comment) {
    const conditionId = eventStuff.conditionId?.value;
    const eventId = eventStuff.eventId?.value;
    try {
        return await (0, call_method_condition_1.acknowledgeCondition)(session, conditionId, eventId, comment);
    }
    catch (err) {
        errorLog(`conditionId: ${conditionId?.toString()}`);
        errorLog("Acknowledging Condition has failed !", err);
        return node_opcua_status_code_1.StatusCodes.BadInternalError;
    }
}
async function confirmConditionEV(session, eventStuff, comment) {
    const conditionId = eventStuff.conditionId?.value;
    const eventId = eventStuff.eventId?.value;
    try {
        return await (0, call_method_condition_1.confirmCondition)(session, conditionId, eventId, comment);
    }
    catch (err) {
        errorLog(`conditionId: ${conditionId?.toString()}`);
        errorLog("Confirming Condition has failed !", err);
        return node_opcua_status_code_1.StatusCodes.BadInternalError;
    }
}
/**
 * Enumerate all events
 * @param session
 */
async function findActiveConditions(session) {
    const request = {
        maxNotificationsPerPublish: 10000,
        priority: 6,
        publishingEnabled: true,
        requestedLifetimeCount: 1000,
        requestedMaxKeepAliveCount: 100,
        requestedPublishingInterval: 100
    };
    const subscription = await session.createSubscription2(request);
    const itemToMonitor = {
        attributeId: node_opcua_service_read_1.AttributeIds.EventNotifier,
        nodeId: (0, node_opcua_nodeid_1.resolveNodeId)("Server") // i=2253session
    };
    const fields = await (0, extract_condition_fields_1.extractConditionFields)(session, "AcknowledgeableConditionType");
    // note: we may want to have this select clause
    //  Or(OfType("AcknowledgeableConditionType"), OfType("RefreshStartEventType"), OfType("RefreshEndEventType"))
    const eventFilter = (0, node_opcua_service_filter_1.constructEventFilter)(fields);
    const monitoringParameters = {
        discardOldest: false,
        filter: eventFilter,
        queueSize: 100,
        samplingInterval: 0
    };
    const event_monitoringItem = await subscription.monitor(itemToMonitor, monitoringParameters, node_opcua_service_read_1.TimestampsToReturn.Both);
    const acknowledgeableConditions = [];
    let refreshStartEventHasBeenReceived = false;
    let RefreshEndEventHasBeenReceived = false;
    const RefreshStartEventType = (0, node_opcua_nodeid_1.resolveNodeId)("RefreshStartEventType").toString();
    const RefreshEndEventType = (0, node_opcua_nodeid_1.resolveNodeId)("RefreshEndEventType").toString();
    await new Promise((resolve, reject) => {
        // now create a event monitored Item
        event_monitoringItem.on("changed", (_eventFields) => {
            const eventFields = _eventFields;
            try {
                if (RefreshEndEventHasBeenReceived) {
                    return;
                }
                // dumpEvent(session, fields, eventFields);
                const pojo = (0, event_stuff_1.fieldsToJson)(fields, eventFields);
                // make sure we only start recording event after the RefreshStartEvent has been received
                if (!refreshStartEventHasBeenReceived) {
                    if (pojo.eventType.value.toString() === RefreshStartEventType) {
                        refreshStartEventHasBeenReceived = true;
                    }
                    return;
                }
                if (pojo.eventType.value.toString() === RefreshEndEventType) {
                    RefreshEndEventHasBeenReceived = true;
                    resolve();
                    return;
                }
                if (!pojo.conditionId.value) {
                    // not a Acknowledgeable condition
                    return;
                }
                if (pojo.ackedState.id.dataType === node_opcua_variant_1.DataType.Boolean) {
                    acknowledgeableConditions.push(pojo);
                }
            }
            catch (err) {
                errorLog("Error !!", err);
            }
        });
        // async call without waiting !
        try {
            (0, call_condition_refresh_1.callConditionRefresh)(session, subscription.subscriptionId);
        }
        catch (err) {
            // it is possible that server do not implement conditionRefresh ...
            errorLog("Server may not implement conditionRefresh", err.message);
        }
    });
    // now shut down subscription
    await subscription.terminate();
    return acknowledgeableConditions;
}
// let conditions = await findActiveConditions(session);
// if (conditions.length === 0) {
//     debugLog("Warning: cannot find conditions ");
// }
async function acknowledgeAllConditions(session, message) {
    try {
        let conditions = await findActiveConditions(session);
        // filter acknowledgeable conditions (no acked yet)
        conditions = conditions.filter((pojo) => pojo.ackedState.id.value === false);
        const promises = [];
        for (const eventStuff of conditions) {
            promises.push(acknowledgeConditionEV(session, eventStuff, message));
        }
        const result = await Promise.all(promises);
        // istanbul ignore next
        if (doDebug) {
            debugLog("Acked all results: ", result.map((e) => e.toString()).join(" "));
        }
    }
    catch (err) {
        errorLog("Error", err);
    }
}
async function confirmAllConditions(session, message) {
    try {
        let conditions = await findActiveConditions(session);
        // filter acknowledgeable conditions (no acked yet)
        conditions = conditions.filter((pojo) => pojo.confirmedState.id.value === false);
        const promises = [];
        for (const eventStuff of conditions) {
            promises.push(confirmConditionEV(session, eventStuff, message));
        }
        const result = await Promise.all(promises);
        // istanbul ignore next
        if (doDebug) {
            debugLog("Confirm all results: ", result.map((e) => e.toString()).join(" "));
        }
    }
    catch (err) {
        errorLog("Error", err);
    }
}
//# sourceMappingURL=acknowledge_all_conditions.js.map