"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.getCountData = getCountData;
const node_opcua_data_value_1 = require("node-opcua-data-value");
const node_opcua_status_code_1 = require("node-opcua-status-code");
const node_opcua_variant_1 = require("node-opcua-variant");
const common_1 = require("./common");
const interval_1 = require("./interval");
/**
 * The Count Aggregate retrieves a count of all the raw values within an interval.
 *  If one or more raw values are non-Good, they are not included in the count, and the Aggregate
 *  StatusCode is determined using the StatusCode Calculation for non-time based Aggregates.
 * If no Good data exists for an interval, the count is zero.
 */
function calculateCountValue(interval, options) {
    const indexStart = interval.index;
    let statusCode = node_opcua_status_code_1.StatusCodes.Good;
    let isPartial = interval.isPartial;
    let nbBad = 0;
    let nbGood = 0;
    let nbUncertain = 0;
    let badDuration = 0;
    let uncertainDuration = 0;
    let goodDuration = 0;
    for (let i = indexStart; i < indexStart + interval.count; i++) {
        const dataValue = interval.dataValues[i];
        if (dataValue.statusCode.equals(node_opcua_status_code_1.StatusCodes.BadNoData)) {
            isPartial = true;
            continue;
        }
        const regionDuration = interval.regionDuration(i);
        if (dataValue.statusCode.isBad()) {
            nbBad++;
            badDuration += regionDuration;
        }
        else if ((0, interval_1.isUncertain)(dataValue.statusCode)) {
            nbUncertain++;
            uncertainDuration += regionDuration;
        }
        else if (dataValue.statusCode.isGoodish()) {
            nbGood++;
            goodDuration += regionDuration;
        }
    }
    const partialFlag = isPartial ? node_opcua_status_code_1.extraStatusCodeBits.HistorianPartial : 0;
    // debugLog(" ", goodDuration, uncertainDuration, badDuration);
    if (nbBad > 0) {
        const duration = interval.duration();
        if (options.treatUncertainAsBad) {
            badDuration += uncertainDuration;
        }
        const actualPercentDataBad = (badDuration / duration) * 100.0;
        const percentDataBad = options.percentDataBad === undefined ? 100 : options.percentDataBad;
        if (actualPercentDataBad >= percentDataBad) {
            return new node_opcua_data_value_1.DataValue({
                sourceTimestamp: interval.startTime,
                statusCode: node_opcua_status_code_1.StatusCodes.Bad
            });
        }
    }
    if (nbUncertain > 0 || nbBad > 0) {
        statusCode = node_opcua_status_code_1.StatusCode.makeStatusCode(node_opcua_status_code_1.StatusCodes.UncertainDataSubNormal, node_opcua_status_code_1.extraStatusCodeBits.HistorianCalculated | partialFlag);
    }
    else {
        statusCode = node_opcua_status_code_1.StatusCode.makeStatusCode(node_opcua_status_code_1.StatusCodes.Good, node_opcua_status_code_1.extraStatusCodeBits.HistorianCalculated | partialFlag);
    }
    if (nbUncertain === 0 && nbGood === 0) {
        statusCode = node_opcua_status_code_1.StatusCodes.BadNoData;
        return new node_opcua_data_value_1.DataValue({
            sourceTimestamp: interval.startTime,
            statusCode: statusCode
        });
    }
    return new node_opcua_data_value_1.DataValue({
        sourceTimestamp: interval.startTime,
        statusCode: statusCode,
        value: {
            dataType: node_opcua_variant_1.DataType.UInt32,
            value: nbGood
        }
    });
}
function getCountData(node, processingInterval, startDate, endDate, callback) {
    (0, common_1.getAggregateData)(node, processingInterval, startDate, endDate, calculateCountValue, callback);
}
//# sourceMappingURL=count.js.map