"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.getAggregateData = getAggregateData;
exports.interpolateValue = interpolateValue;
/**
 * @module node-opca-aggregates
 */
const node_opcua_address_space_1 = require("node-opcua-address-space");
const node_opcua_data_model_1 = require("node-opcua-data-model");
const node_opcua_data_value_1 = require("node-opcua-data-value");
const node_opcua_service_history_1 = require("node-opcua-service-history");
const node_opcua_status_code_1 = require("node-opcua-status-code");
const node_opcua_nodeid_1 = require("node-opcua-nodeid");
const aggregates_1 = require("./aggregates");
const interval_1 = require("./interval");
/**
 * @internal
 */
function processAggregateData(node, processingInterval, startDate, endDate, dataValues, lambda, callback) {
    const aggregateConfiguration = (0, aggregates_1.getAggregateConfiguration)(node);
    const results = [];
    const startTime = startDate.getTime();
    const endTime = endDate.getTime();
    const indexHint = 0;
    for (let t = startTime; t < endTime; t += processingInterval) {
        const sourceTimestamp = new Date();
        sourceTimestamp.setTime(t);
        const interval = (0, interval_1.getInterval)(sourceTimestamp, processingInterval, indexHint, dataValues);
        const dataValue = lambda(interval, aggregateConfiguration);
        /* istanbul ignore next */
        if (!dataValue || !dataValue.sourceTimestamp) {
            // const dataValue = interval.interpolatedValue(aggregateConfiguration);
            throw Error("invalid DataValue");
        }
        results.push(dataValue);
        // debugLog(" => ", dataValue.sourceTimestamp.toISOString(), dataValue.statusCode.toString(), dataValue.value.value);
    }
    setImmediate(() => {
        callback(null, results);
    });
}
function getAggregateData(node, processingInterval, startDate, endDate, lambda, callback) {
    /* istanbul ignore next */
    if (node.nodeClass !== node_opcua_data_model_1.NodeClass.Variable) {
        return callback(new Error("node must be UAVariable"));
    }
    /* istanbul ignore next */
    if (processingInterval <= 0) {
        return callback(new Error("Invalid processing interval, shall be greater than 0"));
    }
    const continuationPointManager = new node_opcua_address_space_1.ContinuationPointManager();
    const context = new node_opcua_address_space_1.SessionContext({
        session: {
            continuationPointManager,
            getSessionId: () => (0, node_opcua_nodeid_1.coerceNodeId)("i=0")
        }
    });
    const historyReadDetails = new node_opcua_service_history_1.ReadRawModifiedDetails({
        endTime: endDate,
        startTime: startDate,
        isReadModified: false,
        numValuesPerNode: 0
        // returnBounds: true,
    });
    const indexRange = null;
    const dataEncoding = null;
    const continuationPoint = null;
    node.historyRead(context, historyReadDetails, indexRange, dataEncoding, { continuationPoint }, (err, result) => {
        /* istanbul ignore next */
        if (err) {
            return callback(err);
        }
        const historyData = result.historyData;
        const dataValues = historyData.dataValues || [];
        processAggregateData(node, processingInterval, startDate, endDate, dataValues, lambda, callback);
    });
}
function interpolateValue(dataValue1, dataValue2, date) {
    const t0 = dataValue1.sourceTimestamp.getTime();
    const t = date.getTime();
    const t1 = dataValue2.sourceTimestamp.getTime();
    const coef1 = (t - t0) / (t1 - t0);
    const coef2 = (t1 - t) / (t1 - t0);
    const value = dataValue1.value.clone();
    value.value = coef2 * dataValue1.value.value + coef1 * dataValue2.value.value;
    const statusCode = node_opcua_status_code_1.StatusCode.makeStatusCode(dataValue1.statusCode, "HistorianInterpolated");
    return new node_opcua_data_value_1.DataValue({
        sourceTimestamp: date,
        statusCode,
        value
    });
}
//# sourceMappingURL=common.js.map