"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.createBoilerType = createBoilerType;
exports.makeBoiler = makeBoiler;
/* eslint-disable max-statements */
/**
 * @module node-opcua-address-space
 */
const node_opcua_assert_1 = require("node-opcua-assert");
const node_opcua_data_model_1 = require("node-opcua-data-model");
const node_opcua_status_code_1 = require("node-opcua-status-code");
const node_opcua_utils_1 = require("node-opcua-utils");
const node_opcua_address_space_base_1 = require("node-opcua-address-space-base");
const __1 = require("..");
function myGetExecutableFlag(method, toState, methodName) {
    const stateMachineW = (0, __1.promoteToStateMachine)(method.parent);
    return stateMachineW.isValidTransition(toState);
}
function implementProgramStateMachine(programStateMachine) {
    function installMethod(methodName, toState) {
        let method = programStateMachine.getMethodByName(methodName);
        if (!method) {
            // 'method' has ModellingRule=OptionalPlaceholder and should be created from the type definition
            let methodToClone = programStateMachine.typeDefinitionObj.getMethodByName(methodName);
            if (!methodToClone) {
                methodToClone = programStateMachine.typeDefinitionObj.subtypeOfObj.getMethodByName(methodName);
            }
            methodToClone.clone({
                namespace: programStateMachine.namespace,
                componentOf: programStateMachine
            });
            method = programStateMachine.getMethodByName(methodName);
            (0, node_opcua_assert_1.assert)(method !== null, "Method clone should cause parent object to be extended");
        }
        (0, node_opcua_assert_1.assert)(method.nodeClass === node_opcua_data_model_1.NodeClass.Method);
        method._getExecutableFlag = function ( /* sessionContext: SessionContext */) {
            // must use  a function here to capture 'this'
            return myGetExecutableFlag(this, toState, methodName);
        };
        method.bindMethod(function (inputArguments, context, callback) {
            const stateMachineW = this.parent;
            stateMachineW.setState(toState);
            callback(null, {
                outputArguments: [],
                statusCode: node_opcua_status_code_1.StatusCodes.Good
            });
        });
        (0, node_opcua_assert_1.assert)(programStateMachine.getMethodByName(methodName) !== null, "Method " + methodName + " should be added to parent object (checked with getMethodByName)");
        const lc_name = (0, node_opcua_utils_1.lowerFirstLetter)(methodName);
    }
    installMethod("Halt", "Halted");
    installMethod("Reset", "Ready");
    installMethod("Start", "Running");
    installMethod("Suspend", "Suspended");
    installMethod("Resume", "Running");
}
function addRelation(srcNode, referenceType, targetNode) {
    (0, node_opcua_assert_1.assert)(srcNode, "expecting srcNode !== null");
    (0, node_opcua_assert_1.assert)(targetNode, "expecting targetNode !== null");
    if (typeof referenceType === "string") {
        const nodes = srcNode.findReferencesAsObject(referenceType, true);
        (0, node_opcua_assert_1.assert)(nodes.length === 1);
        referenceType = nodes[0];
    }
    srcNode.addReference({ referenceType: referenceType.nodeId, nodeId: targetNode });
}
function createBoilerType(namespace) {
    // istanbul ignore next
    if (namespace.findObjectType("BoilerType")) {
        console.warn("createBoilerType has already been called");
        return namespace.findObjectType("BoilerType");
    }
    // --------------------------------------------------------
    // referenceTypes
    // --------------------------------------------------------
    // create new reference Type FlowTo HotFlowTo & SignalTo
    const flowTo = namespace.addReferenceType({
        browseName: "FlowTo",
        description: "a reference that indicates a flow between two objects",
        inverseName: "FlowFrom",
        subtypeOf: "NonHierarchicalReferences"
    });
    const hotFlowTo = namespace.addReferenceType({
        browseName: "HotFlowTo",
        description: "a reference that indicates a high temperature flow between two objects",
        inverseName: "HotFlowFrom",
        subtypeOf: flowTo
    });
    const signalTo = namespace.addReferenceType({
        browseName: "SignalTo",
        description: "a reference that indicates an electrical signal between two variables",
        inverseName: "SignalFrom",
        subtypeOf: "NonHierarchicalReferences"
    });
    const addressSpace = namespace.addressSpace;
    flowTo.isSubtypeOf(addressSpace.findReferenceType("References"));
    flowTo.isSubtypeOf(addressSpace.findReferenceType("NonHierarchicalReferences"));
    hotFlowTo.isSubtypeOf(addressSpace.findReferenceType("References"));
    hotFlowTo.isSubtypeOf(addressSpace.findReferenceType("NonHierarchicalReferences"));
    hotFlowTo.isSubtypeOf(addressSpace.findReferenceType("FlowTo", namespace.index));
    const NonHierarchicalReferences = addressSpace.findReferenceType("NonHierarchicalReferences");
    NonHierarchicalReferences;
    // --------------------------------------------------------
    // EventTypes
    // --------------------------------------------------------
    const boilerHaltedEventType = namespace.addEventType({
        browseName: "BoilerHaltedEventType",
        subtypeOf: "TransitionEventType"
    });
    boilerHaltedEventType;
    // --------------------------------------------------------
    // CustomControllerType
    // --------------------------------------------------------
    const customControllerType = namespace.addObjectType({
        browseName: "CustomControllerType",
        description: "a custom PID controller with 3 inputs"
    });
    const input1 = namespace.addVariable({
        browseName: "Input1",
        dataType: "Double",
        description: "a reference that indicates an electrical signal between two variables",
        modellingRule: "Mandatory",
        propertyOf: customControllerType
    });
    input1;
    const input2 = namespace.addVariable({
        browseName: "Input2",
        dataType: "Double",
        modellingRule: "Mandatory",
        propertyOf: customControllerType
    });
    input2;
    const input3 = namespace.addVariable({
        browseName: "Input3",
        dataType: "Double",
        modellingRule: "Mandatory",
        propertyOf: customControllerType
    });
    input3;
    const controlOut = namespace.addVariable({
        browseName: "ControlOut",
        dataType: "Double",
        modellingRule: "Mandatory",
        propertyOf: customControllerType
    });
    controlOut;
    const description = namespace.addVariable({
        browseName: "Description",
        dataType: "LocalizedText",
        modellingRule: "Mandatory",
        propertyOf: customControllerType
    });
    description;
    // --------------------------------------------------------
    // GenericSensorType
    // --------------------------------------------------------
    const genericSensorType = namespace.addObjectType({
        browseName: "GenericSensorType"
    });
    namespace.addAnalogDataItem({
        browseName: "Output",
        componentOf: genericSensorType,
        dataType: "Double",
        engineeringUnitsRange: { low: -100, high: 200 },
        modellingRule: "Mandatory"
    });
    genericSensorType.install_extra_properties();
    genericSensorType.getComponentByName("Output");
    (0, node_opcua_assert_1.assert)(genericSensorType.getComponentByName("Output").modellingRule === "Mandatory");
    // --------------------------------------------------------
    // GenericSensorType  <---- GenericControllerType
    // --------------------------------------------------------
    const genericControllerType = namespace.addObjectType({
        browseName: "GenericControllerType"
    });
    namespace.addVariable({
        browseName: "ControlOut",
        dataType: "Double",
        modellingRule: "Mandatory",
        propertyOf: genericControllerType
    });
    namespace.addVariable({
        browseName: "Measurement",
        dataType: "Double",
        modellingRule: "Mandatory",
        propertyOf: genericControllerType
    });
    namespace.addVariable({
        browseName: "SetPoint",
        dataType: "Double",
        modellingRule: "Mandatory",
        propertyOf: genericControllerType
    });
    // --------------------------------------------------------------------------------
    // GenericSensorType  <---- GenericControllerType <--- FlowControllerType
    // --------------------------------------------------------------------------------
    const flowControllerType = namespace.addObjectType({
        browseName: "FlowControllerType",
        subtypeOf: genericControllerType
    });
    // --------------------------------------------------------------------------------
    // GenericSensorType  <---- GenericControllerType <--- LevelControllerType
    // --------------------------------------------------------------------------------
    const levelControllerType = namespace.addObjectType({
        browseName: "LevelControllerType",
        subtypeOf: genericControllerType
    });
    // --------------------------------------------------------------------------------
    // GenericSensorType  <---- FlowTransmitterType
    // --------------------------------------------------------------------------------
    const flowTransmitterType = namespace.addObjectType({
        browseName: "FlowTransmitterType",
        subtypeOf: genericSensorType
    });
    // --------------------------------------------------------------------------------
    // GenericSensorType  <---- LevelIndicatorType
    // --------------------------------------------------------------------------------
    const levelIndicatorType = namespace.addObjectType({
        browseName: "LevelIndicatorType",
        subtypeOf: genericSensorType
    });
    // --------------------------------------------------------------------------------
    // GenericActuatorType
    // --------------------------------------------------------------------------------
    const genericActuatorType = namespace.addObjectType({
        browseName: "GenericActuatorType"
    });
    namespace.addAnalogDataItem({
        browseName: "Input",
        componentOf: genericActuatorType,
        dataType: "Double",
        engineeringUnitsRange: { low: -100, high: 200 },
        modellingRule: "Mandatory"
    });
    // --------------------------------------------------------------------------------
    // GenericActuatorType  <---- ValveType
    // --------------------------------------------------------------------------------
    const valveType = namespace.addObjectType({
        browseName: "ValveType",
        subtypeOf: genericActuatorType
    });
    // --------------------------------------------------------------------------------
    // FolderType  <---- BoilerInputPipeType
    // --------------------------------------------------------------------------------
    const boilerInputPipeType = namespace.addObjectType({
        browseName: "BoilerInputPipeType",
        subtypeOf: "FolderType"
    });
    const ftx1 = flowTransmitterType.instantiate({
        browseName: "FlowTransmitter",
        componentOf: boilerInputPipeType,
        modellingRule: "Mandatory",
        notifierOf: boilerInputPipeType,
        eventNotifier: node_opcua_address_space_base_1.EventNotifierFlags.SubscribeToEvents
    });
    (0, node_opcua_assert_1.assert)(ftx1.eventNotifier === node_opcua_address_space_base_1.EventNotifierFlags.SubscribeToEvents);
    (0, node_opcua_assert_1.assert)(ftx1.output.browseName.toString() === `${namespace.index}:Output`);
    const valve1 = valveType.instantiate({
        browseName: "Valve",
        componentOf: boilerInputPipeType,
        modellingRule: "Mandatory"
    });
    // --------------------------------------------------------------------------------
    // FolderType  <---- BoilerOutputPipeType
    // --------------------------------------------------------------------------------
    const boilerOutputPipeType = namespace.addObjectType({
        browseName: "BoilerOutputPipeType",
        subtypeOf: "FolderType"
    });
    const ftx2 = flowTransmitterType.instantiate({
        browseName: "FlowTransmitter",
        componentOf: boilerOutputPipeType,
        modellingRule: "Mandatory",
        notifierOf: boilerOutputPipeType,
        eventNotifier: node_opcua_address_space_base_1.EventNotifierFlags.SubscribeToEvents
    });
    ftx2.getComponentByName("Output").browseName.toString();
    // --------------------------------)------------------------------------------------
    // FolderType  <---- BoilerDrumType
    // --------------------------------------------------------------------------------
    const boilerDrumType = namespace.addObjectType({
        browseName: "BoilerDrumType",
        subtypeOf: "FolderType"
    });
    const levelIndicator = levelIndicatorType.instantiate({
        browseName: "LevelIndicator",
        componentOf: boilerDrumType,
        modellingRule: "Mandatory",
        notifierOf: boilerDrumType,
        eventNotifier: node_opcua_address_space_base_1.EventNotifierFlags.SubscribeToEvents
    });
    (0, node_opcua_assert_1.assert)(levelIndicator.eventNotifier === node_opcua_address_space_base_1.EventNotifierFlags.SubscribeToEvents);
    const programFiniteStateMachineType = addressSpace.findObjectType("ProgramStateMachineType");
    // --------------------------------------------------------
    // define boiler State Machine
    // --------------------------------------------------------
    const boilerStateMachineType = namespace.addObjectType({
        browseName: "BoilerStateMachineType",
        postInstantiateFunc: implementProgramStateMachine,
        subtypeOf: programFiniteStateMachineType
    });
    // programStateMachineType has Optional placeHolder for method "Halt", "Reset","Start","Suspend","Resume")
    function addMethod(baseType, objectType, methodName) {
        (0, node_opcua_assert_1.assert)(!objectType.getMethodByName(methodName));
        const method = baseType.getMethodByName(methodName);
        const m = method.clone({
            namespace,
            componentOf: objectType,
            modellingRule: "Mandatory"
        });
        (0, node_opcua_assert_1.assert)(objectType.getMethodByName(methodName));
        (0, node_opcua_assert_1.assert)(objectType.getMethodByName(methodName).modellingRule === "Mandatory");
    }
    addMethod(programFiniteStateMachineType, boilerStateMachineType, "Halt");
    addMethod(programFiniteStateMachineType, boilerStateMachineType, "Reset");
    addMethod(programFiniteStateMachineType, boilerStateMachineType, "Start");
    addMethod(programFiniteStateMachineType, boilerStateMachineType, "Suspend");
    addMethod(programFiniteStateMachineType, boilerStateMachineType, "Resume");
    // --------------------------------------------------------------------------------
    // BoilerType
    // --------------------------------------------------------------------------------
    const boilerType = namespace.addObjectType({
        browseName: "BoilerType",
        eventNotifier: 0x1
    });
    // BoilerType.CustomController (CustomControllerType)
    const customController = customControllerType.instantiate({
        browseName: "CustomController",
        componentOf: boilerType,
        modellingRule: "Mandatory"
    });
    customController;
    // BoilerType.FlowController (FlowController)
    const flowController = flowControllerType.instantiate({
        browseName: "FlowController",
        componentOf: boilerType,
        modellingRule: "Mandatory"
    });
    flowController;
    // BoilerType.LevelController (LevelControllerType)
    const levelController = levelControllerType.instantiate({
        browseName: "LevelController",
        componentOf: boilerType,
        modellingRule: "Mandatory"
    });
    levelController;
    // BoilerType.LevelIndicator (BoilerInputPipeType)
    const inputPipe = boilerInputPipeType.instantiate({
        browseName: "InputPipe",
        componentOf: boilerType,
        modellingRule: "Mandatory",
        notifierOf: boilerType,
        eventNotifier: node_opcua_address_space_base_1.EventNotifierFlags.SubscribeToEvents
    });
    (0, node_opcua_assert_1.assert)(inputPipe.eventNotifier === node_opcua_address_space_base_1.EventNotifierFlags.SubscribeToEvents);
    // BoilerType.BoilerDrum (BoilerDrumType)
    const boilerDrum = boilerDrumType.instantiate({
        browseName: "BoilerDrum",
        componentOf: boilerType,
        modellingRule: "Mandatory",
        notifierOf: boilerType,
        eventNotifier: node_opcua_address_space_base_1.EventNotifierFlags.SubscribeToEvents
    });
    // BoilerType.OutputPipe (BoilerOutputPipeType)
    const outputPipe = boilerOutputPipeType.instantiate({
        browseName: "OutputPipe",
        componentOf: boilerType,
        modellingRule: "Mandatory",
        notifierOf: boilerType,
        eventNotifier: node_opcua_address_space_base_1.EventNotifierFlags.SubscribeToEvents
    });
    // BoilerType.Simulation (BoilerStateMachineType)
    const simulation = boilerStateMachineType.instantiate({
        browseName: "Simulation",
        componentOf: boilerType,
        eventSourceOf: boilerType,
        modellingRule: "Mandatory"
    });
    simulation;
    addRelation(inputPipe, flowTo, boilerDrum);
    addRelation(boilerDrum, hotFlowTo, outputPipe);
    (0, node_opcua_assert_1.assert)(boilerType.inputPipe.flowTransmitter);
    (0, node_opcua_assert_1.assert)(boilerType.inputPipe.flowTransmitter.output);
    (0, node_opcua_assert_1.assert)(boilerType.flowController.measurement);
    addRelation(boilerType.inputPipe.flowTransmitter.output, signalTo, boilerType.flowController.measurement);
    addRelation(boilerType.inputPipe.flowTransmitter.output, signalTo, boilerType.customController.input2);
    addRelation(boilerType.flowController.controlOut, signalTo, boilerType.inputPipe.valve.input);
    // indicates that the level controller gets its measurement from the drum's level indicator.
    addRelation(boilerType.boilerDrum.levelIndicator.output, signalTo, boilerType.levelController.measurement);
    addRelation(boilerType.outputPipe.flowTransmitter.output, signalTo, boilerType.customController.input3);
    addRelation(boilerType.levelController.controlOut, signalTo, boilerType.customController.input1);
    addRelation(boilerType.customController.controlOut, signalTo, boilerType.flowController.setPoint);
    return boilerType;
}
function makeBoiler(addressSpace, options) {
    const namespace = addressSpace.getOwnNamespace();
    (0, node_opcua_assert_1.assert)(options);
    let boilerType;
    boilerType = namespace.findObjectType("BoilerType");
    // istanbul ignore next
    if (!boilerType) {
        createBoilerType(namespace);
        boilerType = namespace.findObjectType("BoilerType");
    }
    // now instantiate boiler
    const boiler1 = boilerType.instantiate({
        browseName: options.browseName,
        organizedBy: addressSpace.rootFolder.objects
    });
    (0, __1.promoteToStateMachine)(boiler1.simulation);
    const boilerStateMachine = boiler1.simulation;
    const readyState = boilerStateMachine.getStateByName("Ready");
    boilerStateMachine.setState(readyState);
    return boiler1;
}
//# sourceMappingURL=boiler_system.js.map