"use strict";
var __importDefault = (this && this.__importDefault) || function (mod) {
    return (mod && mod.__esModule) ? mod : { "default": mod };
};
Object.defineProperty(exports, "__esModule", { value: true });
exports.UAVariableImplT = exports.UAVariableImpl = void 0;
exports.adjust_accessLevel = adjust_accessLevel;
exports.adjust_userAccessLevel = adjust_userAccessLevel;
/* eslint-disable max-statements */
/**
 * @module node-opcua-address-space
 */
const util_1 = require("util");
const chalk_1 = __importDefault(require("chalk"));
const node_opcua_address_space_base_1 = require("node-opcua-address-space-base");
const node_opcua_assert_1 = require("node-opcua-assert");
const node_opcua_data_model_1 = require("node-opcua-data-model");
const node_opcua_data_value_1 = require("node-opcua-data-value");
const node_opcua_date_time_1 = require("node-opcua-date-time");
const node_opcua_debug_1 = require("node-opcua-debug");
const node_opcua_extension_object_1 = require("node-opcua-extension-object");
const node_opcua_nodeid_1 = require("node-opcua-nodeid");
const node_opcua_numeric_range_1 = require("node-opcua-numeric-range");
const node_opcua_service_write_1 = require("node-opcua-service-write");
const node_opcua_status_code_1 = require("node-opcua-status-code");
const node_opcua_types_1 = require("node-opcua-types");
const node_opcua_utils_1 = require("node-opcua-utils");
const node_opcua_variant_1 = require("node-opcua-variant");
const session_context_1 = require("../source/session_context");
const multiform_func_1 = require("../source/helpers/multiform_func");
const base_node_impl_1 = require("./base_node_impl");
const base_node_private_1 = require("./base_node_private");
const ua_data_type_impl_1 = require("./ua_data_type_impl");
const apply_condition_refresh_1 = require("./apply_condition_refresh");
const ua_variable_impl_ext_obj_1 = require("./ua_variable_impl_ext_obj");
const adjust_datavalue_status_code_1 = require("./data_access/adjust_datavalue_status_code");
const get_basic_datatype_1 = require("./get_basic_datatype");
const validate_data_type_correctness_1 = require("./validate_data_type_correctness");
const debugLog = (0, node_opcua_debug_1.make_debugLog)(__filename);
const warningLog = (0, node_opcua_debug_1.make_warningLog)(__filename);
const doDebug = (0, node_opcua_debug_1.checkDebugFlag)(__filename);
const errorLog = (0, node_opcua_debug_1.make_errorLog)(__filename);
function adjust_accessLevel(accessLevel) {
    accessLevel = (0, node_opcua_utils_1.isNullOrUndefined)(accessLevel) ? "CurrentRead | CurrentWrite" : accessLevel;
    accessLevel = (0, node_opcua_data_model_1.makeAccessLevelFlag)(accessLevel);
    (0, node_opcua_assert_1.assert)(isFinite(accessLevel));
    return accessLevel;
}
function adjust_userAccessLevel(userAccessLevel, accessLevel) {
    if (userAccessLevel === undefined) {
        return undefined;
    }
    userAccessLevel = adjust_accessLevel(userAccessLevel);
    accessLevel = adjust_accessLevel(accessLevel);
    return (0, node_opcua_data_model_1.makeAccessLevelFlag)(accessLevel & userAccessLevel);
}
function adjust_samplingInterval(minimumSamplingInterval) {
    (0, node_opcua_assert_1.assert)(isFinite(minimumSamplingInterval));
    if (minimumSamplingInterval < 0) {
        return -1; // only -1 is a valid negative value for samplingInterval and means "unspecified"
    }
    return minimumSamplingInterval;
}
function is_Variant(v) {
    return v instanceof node_opcua_variant_1.Variant;
}
function is_StatusCode(v) {
    return (v &&
        v.constructor &&
        (v instanceof node_opcua_status_code_1.StatusCode ||
            v.constructor.name === "ConstantStatusCode" ||
            v.constructor.name === "StatusCode" ||
            v.constructor.name === "ModifiableStatusCode"));
}
function is_Variant_or_StatusCode(v) {
    if (is_Variant(v)) {
        // /@@assert(v.isValid());
    }
    return is_Variant(v) || is_StatusCode(v);
}
function default_func(dataValue1, callback1) {
    return _default_writable_timestamped_set_func.call(this, dataValue1, callback1);
}
/**
 * A OPCUA Variable Node
 *
 * @class UAVariable
 * @constructor
 * @extends  BaseNode
 *  The AccessLevel Attribute is used to indicate how the Value of a Variable can be accessed (read/write) and if it
 *  contains current and/or historic data. The AccessLevel does not take any user access rights into account,
 *  i.e. although the Variable is writable this may be restricted to a certain user / user group.
 *  The AccessLevel is an 8-bit unsigned integer with the structure defined in the following table:
 *
 *  Field            Bit    Description
 *  CurrentRead      0      Indicates if the current value is readable
 *                          (0 means not readable, 1 means readable).
 *  CurrentWrite     1      Indicates if the current value is writable
 *                          (0 means not writable, 1 means writable).
 *  HistoryRead      2      Indicates if the history of the value is readable
 *                          (0 means not readable, 1 means readable).
 *  HistoryWrite     3      Indicates if the history of the value is writable (0 means not writable, 1 means writable).
 *  SemanticChange   4      Indicates if the Variable used as Property generates SemanticChangeEvents (see 9.31).
 *  Reserved         5:7    Reserved for future use. Shall always be zero.
 *
 *  The first two bits also indicate if a current value of this Variable is available and the second two bits
 *  indicates if the history of the Variable is available via the OPC UA server.
 *
 */
class UAVariableImpl extends base_node_impl_1.BaseNodeImpl {
    nodeClass = node_opcua_data_model_1.NodeClass.Variable;
    dataType;
    _basicDataType;
    $extensionObject;
    $set_ExtensionObject;
    $historicalDataConfiguration;
    varHistorian;
    /**
     * @internal @private
     */
    $dataValue;
    accessLevel;
    userAccessLevel;
    valueRank;
    minimumSamplingInterval;
    historizing;
    semantic_version;
    arrayDimensions;
    _timestamped_get_func;
    _timestamped_set_func;
    _get_func;
    _set_func;
    refreshFunc;
    __waiting_callbacks;
    get typeDefinitionObj() {
        // istanbul ignore next
        if (super.typeDefinitionObj && super.typeDefinitionObj.nodeClass !== node_opcua_data_model_1.NodeClass.VariableType) {
            // this could happen in faulty external nodeset and has been seen once
            // in an nano server
            warningLog(super.typeDefinitionObj.toString());
            return this.addressSpace.findVariableType("BaseVariableType");
        }
        return super.typeDefinitionObj;
    }
    get typeDefinition() {
        return super.typeDefinition;
    }
    constructor(options) {
        super(options);
        (0, node_opcua_variant_1.verifyRankAndDimensions)(options);
        this.valueRank = options.valueRank;
        this.arrayDimensions = options.arrayDimensions;
        this.dataType = this.resolveNodeId(options.dataType); // DataType (NodeId)
        this.accessLevel = adjust_accessLevel(options.accessLevel);
        this.userAccessLevel = adjust_userAccessLevel(options.userAccessLevel, this.accessLevel);
        this.minimumSamplingInterval = adjust_samplingInterval(options.minimumSamplingInterval || 0);
        this.historizing = !!options.historizing; // coerced to boolean"
        this.$dataValue = new node_opcua_data_value_1.DataValue({ statusCode: node_opcua_status_code_1.StatusCodes.UncertainInitialValue, value: { dataType: node_opcua_variant_1.DataType.Null } });
        if (options.value) {
            this.bindVariable(options.value);
        }
        this.setMaxListeners(5000);
        this.semantic_version = 0;
    }
    checkAccessLevelPrivate(_context, accessLevel) {
        if (this.userAccessLevel === undefined) {
            return true;
        }
        return (this.userAccessLevel & accessLevel) === accessLevel;
    }
    checkPermissionPrivate(context, permission) {
        if (!context)
            return true;
        (0, node_opcua_assert_1.assert)(context instanceof session_context_1.SessionContext);
        if (context.checkPermission) {
            if (!(context.checkPermission instanceof Function)) {
                errorLog("context checkPermission is not a function");
                return false;
            }
            if (!context.checkPermission(this, permission)) {
                return false;
            }
        }
        return true;
    }
    checkPermissionAndAccessLevelPrivate(context, permission, accessLevel) {
        if (!this.checkPermissionPrivate(context, permission)) {
            return false;
        }
        return this.checkAccessLevelPrivate(context, accessLevel);
    }
    isReadable(context) {
        return (this.accessLevel & node_opcua_data_model_1.AccessLevelFlag.CurrentRead) === node_opcua_data_model_1.AccessLevelFlag.CurrentRead;
    }
    isUserReadable(context) {
        if (!this.isReadable(context)) {
            return false;
        }
        if (!this.checkPermissionPrivate(context, node_opcua_types_1.PermissionType.Read)) {
            return false;
        }
        return this.checkAccessLevelPrivate(context, node_opcua_data_model_1.AccessLevelFlag.CurrentRead);
    }
    isWritable(context) {
        return (this.accessLevel & node_opcua_data_model_1.AccessLevelFlag.CurrentWrite) === node_opcua_data_model_1.AccessLevelFlag.CurrentWrite;
    }
    isUserWritable(context) {
        if (!this.isWritable(context)) {
            return false;
        }
        return this.checkPermissionAndAccessLevelPrivate(context, node_opcua_types_1.PermissionType.Write, node_opcua_data_model_1.AccessLevelFlag.CurrentWrite);
    }
    canUserReadHistory(context) {
        return this.checkPermissionAndAccessLevelPrivate(context, node_opcua_types_1.PermissionType.ReadHistory, node_opcua_data_model_1.AccessLevelFlag.HistoryRead);
    }
    canUserWriteHistorizingAttribute(context) {
        if (context && !context.checkPermission(this, node_opcua_types_1.PermissionType.WriteHistorizing)) {
            return false;
        }
        return true;
    }
    canUserInsertHistory(context) {
        return this.checkPermissionAndAccessLevelPrivate(context, node_opcua_types_1.PermissionType.InsertHistory, node_opcua_data_model_1.AccessLevelFlag.HistoryWrite);
    }
    canUserModifyHistory(context) {
        return this.checkPermissionAndAccessLevelPrivate(context, node_opcua_types_1.PermissionType.ModifyHistory, node_opcua_data_model_1.AccessLevelFlag.HistoryWrite);
    }
    canUserDeleteHistory(context) {
        return this.checkPermissionAndAccessLevelPrivate(context, node_opcua_types_1.PermissionType.DeleteHistory, node_opcua_data_model_1.AccessLevelFlag.HistoryWrite);
    }
    /**
     *
     *
     * from OPC.UA.Spec 1.02 part 4
     *  5.10.2.4 StatusCodes
     *  Table 51 defines values for the operation level statusCode contained in the DataValue structure of
     *  each values element. Common StatusCodes are defined in Table 166.
     *
     * Table 51 Read Operation Level Result Codes
     *
     * | Symbolic Id                 | Description
     * |-----------------------------|---------------------------------------------------------------------------------------------|
     * |BadNodeIdInvalid             | The syntax of the node id is not valid.|
     * |BadNodeIdUnknown            |The node id refers to a node that does not exist in the server address space.|
     * |BadAttributeIdInvalid      | BadAttributeIdInvalid The attribute is not supported for the specified node.|
     * |BadIndexRangeInvalid       | The syntax of the index range parameter is invalid.|
     * |BadIndexRangeNoData        | No data exists within the range of indexes specified.|
     * |BadDataEncodingInvalid     | The data encoding is invalid.|
     * |                           | This result is used if no dataEncoding can be applied because an Attribute other|
     * |                           | than Value was requested or the DataType of the Value Attribute is not a subtype|
     * |                           | of the Structure DataType.|
     * |BadDataEncodingUnsupported | The server does not support the requested data encoding for the node. |
     * |                           | This result is used if a dataEncoding can be applied but the passed data encoding |
     * |                           | is not known to the Server. |
     * |BadNotReadable             | The access level does not allow reading or subscribing to the Node.|
     * |BadUserAccessDenied        | User does not have permission to perform the requested operation. (table 165)|
     */
    readValue(context, indexRange, dataEncoding) {
        if (!context) {
            context = session_context_1.SessionContext.defaultContext;
        }
        if (context.isAccessRestricted(this)) {
            return new node_opcua_data_value_1.DataValue({ statusCode: node_opcua_status_code_1.StatusCodes.BadSecurityModeInsufficient });
        }
        if (!this.isReadable(context)) {
            return new node_opcua_data_value_1.DataValue({ statusCode: node_opcua_status_code_1.StatusCodes.BadNotReadable });
        }
        if (!this.checkPermissionPrivate(context, node_opcua_types_1.PermissionType.Read)) {
            return new node_opcua_data_value_1.DataValue({ statusCode: node_opcua_status_code_1.StatusCodes.BadUserAccessDenied });
        }
        if (!this.isUserReadable(context)) {
            return new node_opcua_data_value_1.DataValue({ statusCode: node_opcua_status_code_1.StatusCodes.BadNotReadable });
        }
        if (!(0, node_opcua_data_model_1.isValidDataEncoding)(dataEncoding)) {
            return new node_opcua_data_value_1.DataValue({ statusCode: node_opcua_status_code_1.StatusCodes.BadDataEncodingInvalid });
        }
        if (this._timestamped_get_func) {
            if (this._timestamped_get_func.length === 0) {
                const dataValueOrPromise = this._timestamped_get_func();
                if (!Object.prototype.hasOwnProperty.call(dataValueOrPromise.constructor.prototype, "then")) {
                    if (dataValueOrPromise !== this.$dataValue) {
                        // we may have a problem here if we use a getter that returns a dataValue that is a ExtensionObject
                        if (dataValueOrPromise.value?.dataType === node_opcua_variant_1.DataType.ExtensionObject) {
                            // eslint-disable-next-line max-depth
                            if (this.$extensionObject || this.$$extensionObjectArray) {
                                // we have an extension object already bound to this node
                                // the client is asking us to replace the object entirely by a new one
                                // const ext = dataValue.value.value;
                                this._internal_set_dataValue(dataValueOrPromise);
                                return dataValueOrPromise;
                            }
                        }
                        // TO DO : is this necessary ? this may interfere with current use of $dataValue
                        this.$dataValue = dataValueOrPromise;
                        if (this.$dataValue.statusCode.isGoodish()) {
                            this.verifyVariantCompatibility(this.$dataValue.value);
                        }
                    }
                }
                else {
                    errorLog("[NODE-OPCUA-E28] Unsupported: _timestamped_get_func returns a Promise ! , when the uaVariable has an async getter. Fix your application code.");
                }
            }
        }
        let dataValue = this.$dataValue;
        if (dataValue.statusCode.isGoodish()) {
            // note : extractRange will clone the dataValue
            dataValue = (0, node_opcua_data_value_1.extractRange)(dataValue, indexRange);
        }
        /* istanbul ignore next */
        if (dataValue.statusCode.equals(node_opcua_status_code_1.StatusCodes.BadWaitingForInitialData) ||
            dataValue.statusCode.equals(node_opcua_status_code_1.StatusCodes.UncertainInitialValue)) {
            debugLog(chalk_1.default.red(" Warning:  UAVariable#readValue ") +
                chalk_1.default.cyan(this.browseName.toString()) +
                " (" +
                chalk_1.default.yellow(this.nodeId.toString()) +
                ") exists but dataValue has not been defined");
        }
        return dataValue;
    }
    isEnumeration() {
        return this.addressSpacePrivate.isEnumeration(this.dataType);
    }
    /**
     * return true if the DataType is of type Extension object
     * this is not taking into account the valueRank of the variable
     */
    isExtensionObject() {
        // DataType must be one of Structure
        if (this.dataType.isEmpty())
            return false;
        const dataTypeNode = this.addressSpace.findDataType(this.dataType);
        if (!dataTypeNode) {
            throw new Error(" Cannot find  DataType  " + this.dataType.toString() + " in standard address Space");
        }
        const structureNode = this.addressSpace.findDataType("Structure");
        if (!structureNode) {
            throw new Error(" Cannot find 'Structure' DataType in standard address Space");
        }
        return dataTypeNode.isSubtypeOf(structureNode);
    }
    _getEnumerationInfo() {
        // DataType must be one of Enumeration
        (0, node_opcua_assert_1.assert)(this.isEnumeration(), "Variable is not an enumeration");
        const dataTypeNode = this.addressSpace.findDataType(this.dataType);
        return dataTypeNode._getEnumerationInfo();
    }
    asyncRefresh(...args) {
        if (this.$dataValue.statusCode.isGoodish()) {
            this.verifyVariantCompatibility(this.$dataValue.value);
        }
        const oldestDate = args[0];
        const callback = args[1];
        const lessOrEqual = (a, b) => {
            return (a.timestamp.getTime() < b.timestamp.getTime() ||
                (a.timestamp.getTime() === b.timestamp.getTime() && a.picoseconds <= b.picoseconds));
        };
        if (!this.refreshFunc) {
            // no refresh func
            const dataValue = this.readValue();
            dataValue.serverTimestamp = oldestDate.timestamp;
            dataValue.serverPicoseconds = oldestDate.picoseconds;
            const serverClock = (0, node_opcua_date_time_1.coerceClock)(dataValue.serverTimestamp, dataValue.serverPicoseconds);
            if (lessOrEqual(oldestDate, serverClock)) {
                return callback(null, dataValue);
            }
            else {
                // fake
                return callback(null, dataValue);
            }
        }
        const c1 = this.$dataValue.serverTimestamp
            ? (0, node_opcua_date_time_1.coerceClock)(this.$dataValue.serverTimestamp, this.$dataValue.serverPicoseconds)
            : null;
        if (this.$dataValue.serverTimestamp && lessOrEqual(oldestDate, c1)) {
            const dataValue = this.readValue().clone();
            dataValue.serverTimestamp = oldestDate.timestamp;
            dataValue.serverPicoseconds = oldestDate.picoseconds;
            return callback(null, dataValue);
        }
        try {
            this.refreshFunc.call(this, (err, dataValue) => {
                // istanbul ignore next
                if (err || !dataValue) {
                    errorLog("-------------- refresh call failed", this.browseName.toString(), this.nodeId.toString(), err?.message);
                    dataValue = { statusCode: node_opcua_status_code_1.StatusCodes.BadNoDataAvailable };
                }
                if (dataValue && dataValue !== this.$dataValue) {
                    this._internal_set_dataValue(coerceDataValue(dataValue), null);
                }
                callback(err, this.$dataValue);
            });
        }
        catch (err) {
            errorLog("-------------- refresh call failed 2", this.browseName.toString(), this.nodeId.toString());
            errorLog(err);
            const dataValue = new node_opcua_data_value_1.DataValue({ statusCode: node_opcua_status_code_1.StatusCodes.BadInternalError });
            this._internal_set_dataValue(dataValue, null);
            callback(err, this.$dataValue);
        }
    }
    readEnumValue() {
        const value = this.readValue().value.value;
        const enumInfo = this._getEnumerationInfo();
        const enumV = enumInfo.valueIndex[value];
        return { value, name: enumV ? enumV.name : "?????" };
    }
    writeEnumValue(value) {
        const enumInfo = this._getEnumerationInfo();
        if (typeof value === "string") {
            if (!Object.prototype.hasOwnProperty.call(enumInfo.nameIndex, value)) {
                const possibleValues = Object.keys(enumInfo.nameIndex).join(",");
                throw new Error("UAVariable#writeEnumValue: cannot find value " + value + " in [" + possibleValues + "]");
            }
            const valueIndex = enumInfo.nameIndex[value].value;
            value = valueIndex;
        }
        if (isFinite(value)) {
            const possibleValues = Object.keys(enumInfo.nameIndex).join(",");
            if (!enumInfo.valueIndex[value]) {
                throw new Error("UAVariable#writeEnumValue : value out of range " + value + " in [" + possibleValues + "]");
            }
            this.setValueFromSource({
                dataType: node_opcua_variant_1.DataType.Int32,
                value
            });
        }
        else {
            throw new Error("UAVariable#writeEnumValue:  value type mismatch");
        }
    }
    readAttribute(context, attributeId, indexRange, dataEncoding) {
        context = context || session_context_1.SessionContext.defaultContext;
        (0, node_opcua_assert_1.assert)(context instanceof session_context_1.SessionContext);
        const options = {};
        if (attributeId !== node_opcua_data_model_1.AttributeIds.Value) {
            if (indexRange && indexRange.isDefined()) {
                options.statusCode = node_opcua_status_code_1.StatusCodes.BadIndexRangeNoData;
                return new node_opcua_data_value_1.DataValue(options);
            }
            if ((0, node_opcua_data_model_1.isDataEncoding)(dataEncoding)) {
                options.statusCode = node_opcua_status_code_1.StatusCodes.BadDataEncodingInvalid;
                return new node_opcua_data_value_1.DataValue(options);
            }
        }
        switch (attributeId) {
            case node_opcua_data_model_1.AttributeIds.Value:
                return this.readValue(context, indexRange, dataEncoding);
            case node_opcua_data_model_1.AttributeIds.DataType:
                return this._readDataType();
            case node_opcua_data_model_1.AttributeIds.ValueRank:
                return this._readValueRank();
            case node_opcua_data_model_1.AttributeIds.ArrayDimensions:
                return this._readArrayDimensions();
            case node_opcua_data_model_1.AttributeIds.AccessLevel:
                return this._readAccessLevel(context);
            case node_opcua_data_model_1.AttributeIds.UserAccessLevel:
                return this._readUserAccessLevel(context);
            case node_opcua_data_model_1.AttributeIds.MinimumSamplingInterval:
                return this._readMinimumSamplingInterval();
            case node_opcua_data_model_1.AttributeIds.Historizing:
                return this._readHistorizing();
            case node_opcua_data_model_1.AttributeIds.AccessLevelEx:
                return this._readAccessLevelEx(context);
            default:
                return base_node_impl_1.BaseNodeImpl.prototype.readAttribute.call(this, context, attributeId);
        }
    }
    getBasicDataType() {
        return (0, get_basic_datatype_1._getBasicDataType)(this);
    }
    adjustVariant(variant) {
        return (0, node_opcua_variant_1.adjustVariant)(variant, this.valueRank, this.getBasicDataType());
    }
    verifyVariantCompatibility(variant) {
        try {
            // istanbul ignore next
            if (Object.prototype.hasOwnProperty.call(variant, "value")) {
                if (variant.dataType === null || variant.dataType === undefined) {
                    throw new Error("Variant must provide a valid dataType : variant = " +
                        variant.toString() +
                        " this.dataType= " +
                        this.dataType.toString());
                }
                if (variant.dataType === node_opcua_variant_1.DataType.Boolean &&
                    (this.dataType.namespace !== 0 || this.dataType.value !== node_opcua_variant_1.DataType.Boolean)) {
                    throw new Error("Variant must provide a valid Boolean : variant = " +
                        variant.toString() +
                        " this.dataType= " +
                        this.dataType.toString());
                }
                if (this.dataType.namespace === 0 &&
                    this.dataType.value === node_opcua_variant_1.DataType.LocalizedText &&
                    variant.dataType !== node_opcua_variant_1.DataType.LocalizedText &&
                    variant.dataType !== node_opcua_variant_1.DataType.Null) {
                    throw new Error("Variant must provide a valid LocalizedText : variant = " +
                        variant.toString() +
                        " this.dataType= " +
                        this.dataType.toString());
                }
            }
            const basicType = this.getBasicDataType();
            if (basicType === node_opcua_variant_1.DataType.String && variant.dataType === node_opcua_variant_1.DataType.ByteString) {
                return; // this is allowed
            }
            if (basicType === node_opcua_variant_1.DataType.ByteString && variant.dataType === node_opcua_variant_1.DataType.String) {
                return; // this is allowed
            }
            if (basicType !== node_opcua_variant_1.DataType.Null &&
                basicType !== node_opcua_variant_1.DataType.Variant &&
                variant.dataType !== node_opcua_variant_1.DataType.Null &&
                variant.dataType !== basicType) {
                const message = "UAVariable.setValueFromSource " +
                    this.browseName.toString() +
                    " nodeId:" +
                    this.nodeId.toString() +
                    " dataType:" +
                    this.dataType.toString() +
                    ":\n" +
                    "the provided variant must have the expected dataType!\n" +
                    "   - the expected dataType is " +
                    chalk_1.default.cyan(node_opcua_variant_1.DataType[basicType]) +
                    "\n" +
                    "   - the actual dataType   is " +
                    chalk_1.default.magenta(node_opcua_variant_1.DataType[variant.dataType]) +
                    "\n" +
                    "   - " +
                    variant.toString();
                throw new Error(message);
            }
        }
        catch (err) {
            errorLog("UAVariable  ", err?.message, this.browseName.toString(), " nodeId=", this.nodeId.toString());
            errorLog(err.message);
            errorLog(err.stack);
            throw err;
        }
    }
    /**
     * setValueFromSource is used to let the device sets the variable values
     * this method also records the current time as sourceTimestamp and serverTimestamp.
.     *
    * The method will raise an exception if the value is not compatible with the dataType and expected dimension
    *

    * @param variant  {Variant}
    * @param [statusCode  {StatusCode} = StatusCodes.Good]
    * @param [sourceTimestamp= Now]
    */
    setValueFromSource(variant, statusCode, sourceTimestamp) {
        try {
            statusCode = statusCode || node_opcua_status_code_1.StatusCodes.Good;
            const variant1 = node_opcua_variant_1.Variant.coerce(variant);
            this.verifyVariantCompatibility(variant1);
            const now = (0, node_opcua_date_time_1.coerceClock)(sourceTimestamp, 0);
            const dataValue = new node_opcua_data_value_1.DataValue(null);
            dataValue.serverPicoseconds = now.picoseconds;
            dataValue.serverTimestamp = now.timestamp;
            dataValue.sourcePicoseconds = now.picoseconds;
            dataValue.sourceTimestamp = now.timestamp;
            dataValue.statusCode = statusCode;
            dataValue.value = variant1;
            if (dataValue.value.dataType === node_opcua_variant_1.DataType.ExtensionObject) {
                const valueIsCorrect = this.checkExtensionObjectIsCorrect(dataValue.value.value);
                if (!valueIsCorrect) {
                    errorLog("setValueFromSource Invalid value !");
                    errorLog(this.toString());
                    errorLog(dataValue.toString());
                    this.checkExtensionObjectIsCorrect(dataValue.value.value);
                }
                // ----------------------------------
                if (this.$extensionObject || this.$$extensionObjectArray) {
                    // we have an extension object already bound to this node
                    // the client is asking us to replace the object entirely by a new one
                    // const ext = dataValue.value.value;
                    this._internal_set_dataValue(dataValue);
                    return;
                }
                else {
                    this.$dataValue = dataValue;
                }
            }
            else {
                this._internal_set_dataValue(dataValue);
            }
        }
        catch (err) {
            errorLog("UAVariable#setValueFromString Error : ", this.browseName.toString(), this.nodeId.toString());
            errorLog(err.message);
            errorLog(this.parent?.toString());
            throw err;
        }
    }
    adjustDataValueStatusCode(dataValue) {
        const statusCode = (0, adjust_datavalue_status_code_1.adjustDataValueStatusCode)(this, dataValue, this.acceptValueOutOfRange || false);
        return statusCode;
    }
    writeValue(context, dataValue, ...args) {
        context = context || session_context_1.SessionContext.defaultContext;
        (0, node_opcua_assert_1.assert)(context instanceof session_context_1.SessionContext);
        if (!dataValue.sourceTimestamp) {
            // source timestamp was not specified by the caller
            // we will set the timestamp ourself with the current clock
            if (context.currentTime) {
                dataValue.sourceTimestamp = context.currentTime.timestamp;
                dataValue.sourcePicoseconds = context.currentTime.picoseconds;
            }
            else {
                const { timestamp, picoseconds } = (0, node_opcua_date_time_1.getCurrentClock)();
                dataValue.sourceTimestamp = timestamp;
                dataValue.sourcePicoseconds = picoseconds;
            }
        }
        if (context.currentTime && !dataValue.serverTimestamp) {
            dataValue.serverTimestamp = context.currentTime.timestamp;
            dataValue.serverPicoseconds = context.currentTime.picoseconds;
        }
        // adjust arguments if optional indexRange Parameter is not given
        let indexRange = null;
        let callback;
        if (args.length === 1) {
            indexRange = new node_opcua_numeric_range_1.NumericRange();
            callback = args[0];
        }
        else if (args.length === 2) {
            indexRange = args[0];
            callback = args[1];
        }
        else {
            throw new Error("Invalid Number of args");
        }
        (0, node_opcua_assert_1.assert)(typeof callback === "function");
        (0, node_opcua_assert_1.assert)(dataValue instanceof node_opcua_data_value_1.DataValue);
        // index range could be string
        indexRange = node_opcua_numeric_range_1.NumericRange.coerce(indexRange);
        // test write permission
        if (!this.isWritable(context)) {
            return callback(null, node_opcua_status_code_1.StatusCodes.BadNotWritable);
        }
        if (!this.checkPermissionPrivate(context, node_opcua_types_1.PermissionType.Write)) {
            return new node_opcua_data_value_1.DataValue({ statusCode: node_opcua_status_code_1.StatusCodes.BadUserAccessDenied });
        }
        if (!this.isUserWritable(context)) {
            return callback(null, node_opcua_status_code_1.StatusCodes.BadWriteNotSupported);
        }
        // adjust special case
        const variant = adjustVariant2.call(this, dataValue.value);
        const statusCode = this.checkVariantCompatibility(variant);
        if (statusCode.isNot(node_opcua_status_code_1.StatusCodes.Good)) {
            return callback(null, statusCode);
        }
        // adjust dataValue.statusCode based on InstrumentRange and EngineeringUnits
        const statusCode2 = this.adjustDataValueStatusCode(dataValue);
        if (statusCode2.isNotGood()) {
            return callback(null, statusCode2);
        }
        const write_func = this._timestamped_set_func || default_func;
        if (!write_func) {
            warningLog(" warning " + this.nodeId.toString() + " " + this.browseName.toString() + " has no setter. \n");
            warningLog("Please make sure to bind the variable or to pass a valid value: new Variant({}) during construction time");
            return callback(null, node_opcua_status_code_1.StatusCodes.BadNotWritable);
        }
        (0, node_opcua_assert_1.assert)(write_func);
        write_func.call(this, dataValue, (err, statusCode1) => {
            if (!err) {
                dataValue && this.verifyVariantCompatibility(dataValue.value);
                if (indexRange && !indexRange.isEmpty()) {
                    if (!indexRange.isValid()) {
                        return callback(null, node_opcua_status_code_1.StatusCodes.BadIndexRangeInvalid);
                    }
                    const newArrayOrMatrix = dataValue.value.value;
                    if (dataValue.value.arrayType === node_opcua_variant_1.VariantArrayType.Array) {
                        if (this.$dataValue.value.arrayType !== node_opcua_variant_1.VariantArrayType.Array) {
                            return callback(null, node_opcua_status_code_1.StatusCodes.BadTypeMismatch);
                        }
                        // check that destination data is also an array
                        (0, node_opcua_assert_1.assert)(check_valid_array(this.$dataValue.value.dataType, this.$dataValue.value.value));
                        const destArr = this.$dataValue.value.value;
                        const result = indexRange.set_values(destArr, newArrayOrMatrix);
                        if (result.statusCode.isNot(node_opcua_status_code_1.StatusCodes.Good)) {
                            return callback(null, result.statusCode);
                        }
                        dataValue.value.value = result.array;
                        // scrap original array so we detect range
                        this.$dataValue.value.value = null;
                    }
                    else if (dataValue.value.arrayType === node_opcua_variant_1.VariantArrayType.Matrix) {
                        const dimensions = this.$dataValue.value.dimensions;
                        if (this.$dataValue.value.arrayType !== node_opcua_variant_1.VariantArrayType.Matrix || !dimensions) {
                            // not a matrix !
                            return callback(null, node_opcua_status_code_1.StatusCodes.BadTypeMismatch);
                        }
                        const matrix = this.$dataValue.value.value;
                        const result = indexRange.set_values_matrix({
                            matrix,
                            dimensions
                        }, newArrayOrMatrix);
                        if (result.statusCode.isNot(node_opcua_status_code_1.StatusCodes.Good)) {
                            return callback(null, result.statusCode);
                        }
                        dataValue.value.dimensions = this.$dataValue.value.dimensions;
                        dataValue.value.value = result.matrix;
                        // scrap original array so we detect range
                        this.$dataValue.value.value = null;
                    }
                    else {
                        return callback(null, node_opcua_status_code_1.StatusCodes.BadTypeMismatch);
                    }
                }
                try {
                    this._internal_set_dataValue(dataValue, indexRange);
                }
                catch (err) {
                    if (util_1.types.isNativeError(err)) {
                        warningLog(err.message);
                    }
                    return callback(null, node_opcua_status_code_1.StatusCodes.BadInternalError);
                }
            }
            callback(err || null, statusCode1);
        });
    }
    writeAttribute(context, writeValueOptions, callback) {
        // istanbul ignore next
        if (!callback) {
            throw new Error("Internal error");
        }
        if (!this.canUserWriteAttribute(context, writeValueOptions.attributeId)) {
            return callback(null, node_opcua_status_code_1.StatusCodes.BadUserAccessDenied);
        }
        const writeValue = writeValueOptions instanceof node_opcua_service_write_1.WriteValue ? writeValueOptions : new node_opcua_service_write_1.WriteValue(writeValueOptions);
        context = context || session_context_1.SessionContext.defaultContext;
        (0, node_opcua_assert_1.assert)(context instanceof session_context_1.SessionContext);
        (0, node_opcua_assert_1.assert)(writeValue instanceof node_opcua_service_write_1.WriteValue);
        (0, node_opcua_assert_1.assert)(writeValue.value instanceof node_opcua_data_value_1.DataValue);
        (0, node_opcua_assert_1.assert)(writeValue.value.value instanceof node_opcua_variant_1.Variant);
        (0, node_opcua_assert_1.assert)(typeof callback === "function");
        // Spec 1.0.2 Part 4 page 58
        // If the SourceTimestamp or the ServerTimestamp is specified, the Server shall
        // use these values.
        // xx _apply_default_timestamps(writeValue.value);
        switch (writeValue.attributeId) {
            case node_opcua_data_model_1.AttributeIds.Value:
                this.writeValue(context, writeValue.value, writeValue.indexRange, callback);
                break;
            case node_opcua_data_model_1.AttributeIds.Historizing:
                if (writeValue.value.value.dataType !== node_opcua_variant_1.DataType.Boolean) {
                    return callback(null, node_opcua_status_code_1.StatusCodes.BadTypeMismatch);
                }
                if (!this.checkPermissionPrivate(context, node_opcua_types_1.PermissionType.WriteHistorizing)) {
                    return callback(null, node_opcua_status_code_1.StatusCodes.BadUserAccessDenied);
                }
                if (!this.canUserWriteHistorizingAttribute(context)) {
                    return callback(null, node_opcua_status_code_1.StatusCodes.BadHistoryOperationUnsupported);
                }
                // if the variable has no historizing in place reject
                if (!this.getChildByName("HA Configuration")) {
                    return callback(null, node_opcua_status_code_1.StatusCodes.BadNotSupported);
                }
                // check if user is allowed to do that !
                // TODO
                this.historizing = !!writeValue.value.value.value; // yes ! indeed !
                return callback(null, node_opcua_status_code_1.StatusCodes.Good);
            default:
                super.writeAttribute(context, writeValue, callback);
                break;
        }
    }
    /**

     * note:
     *     this method is overridden in address-space-data-access
     * @return {StatusCode}
     */
    checkVariantCompatibility(value) {
        // test dataType
        if (!this._validate_DataType(value.dataType)) {
            return node_opcua_status_code_1.StatusCodes.BadTypeMismatch;
        }
        try {
            this.verifyVariantCompatibility(value);
        }
        catch (err) {
            return node_opcua_status_code_1.StatusCodes.BadTypeMismatch;
        }
        return node_opcua_status_code_1.StatusCodes.Good;
    }
    /**
     * touch the source timestamp of a Variable and cascade up the change
     * to the parent variable if any.
     */
    touchValue(optionalNow) {
        const now = optionalNow || (0, node_opcua_date_time_1.getCurrentClock)();
        (0, ua_variable_impl_ext_obj_1.propagateTouchValueUpward)(this, now);
    }
    /**
     * bind a variable with a get and set functions.
     *
     *  properties:
     *    - value: a Variant or a status code
     *    - sourceTimestamp
     *    - sourcePicoseconds
     * @param [options.timestamped_set]
     * @param [options.refreshFunc]      the variable asynchronous getter function.
     * @param [overwrite {Boolean} = false] set overwrite to true to overwrite existing binding
     * @return void
     *
     *
     * ### Providing read access to the underlying value
     *
     * #### Variation 1
     *
     * In this variation, the user provides a function that returns a Variant with the current value.
     *
     * The sourceTimestamp will be set automatically.
     *
     * The get function is called synchronously.
     *
     * @example
     *
     *
     * ```javascript
     *     ...
     *     var options =  {
     *       get : () => {
     *          return new Variant({...});
     *       },
     *       set : function(variant) {
     *          // store the variant somewhere
     *          return StatusCodes.Good;
     *       }
     *    };
     *    ...
     *    engine.bindVariable(nodeId,options):
     *    ...
     * ```
     *
     *
     * #### Variation 2:
     *
     * This variation can be used when the user wants to specify a specific '''sourceTimestamp''' associated
     * with the current value of the UAVariable.
     *
     * The provided ```timestamped_get``` function should return an object with three properties:
     * * value: containing the variant value or a error StatusCode,
     * * sourceTimestamp
     * * sourcePicoseconds
     *
     * ```javascript
     * ...
     * var myDataValue = new DataValue({
     *   value: {dataType: DataType.Double , value: 10.0},
     *   sourceTimestamp : new Date(),
     *   sourcePicoseconds: 0
     * });
     * ...
     * var options =  {
     *   timestamped_get : () => { return myDataValue;  }
     * };
     * ...
     * engine.bindVariable(nodeId,options):
     * ...
     * // record a new value
     * myDataValue.value.value = 5.0;
     * myDataValue.sourceTimestamp = new Date();
     * ...
     * ```
     *
     *
     * #### Variation 3:
     *
     * This variation can be used when the value associated with the variables requires a asynchronous function call to be
     * extracted. In this case, the user should provide an async method ```refreshFunc```.
     *
     *
     * The ```refreshFunc``` shall do whatever is necessary to fetch the most up to date version of the variable value, and
     * call the ```callback``` function when the data is ready.
     *
     *
     * The ```callback``` function follow the standard callback function signature:
     * * the first argument shall be **null** or **Error**, depending of the outcome of the fetch operation,
     * * the second argument shall be a DataValue with the new UAVariable Value,  a StatusCode, and time stamps.
     *
     *
     * Optionally, it is possible to pass a sourceTimestamp and a sourcePicoseconds value as a third and fourth arguments
     * of the callback. When sourceTimestamp and sourcePicoseconds are missing, the system will set their default value
     * to the current time..
     *
     *
     * ```javascript
     * ...
     * var options =  {
     *    refreshFunc : function(callback) {
     *      ... do_some_async_stuff_to_get_the_new_variable_value
     *      var dataValue = new DataValue({
     *          value: new Variant({...}),
     *          statusCode: StatusCodes.Good,
     *          sourceTimestamp: new Date()
     *      });
     *      callback(null,dataValue);
     *    }
     * };
     * ...
     * variable.bindVariable(nodeId,options):
     * ...
     * ```
     *
     * ### Providing write access to the underlying value
     *
     * #### Variation1 - provide a simple synchronous set function
     *
     *
     * #### Notes
     *   to do : explain return StatusCodes.GoodCompletesAsynchronously;
     *
     */
    bindVariable(options, overwrite) {
        if (overwrite) {
            this._timestamped_set_func = null;
            this._timestamped_get_func = null;
            this._get_func = null;
            this._set_func = null;
            this.refreshFunc = undefined;
            this._historyRead = UAVariableImpl.prototype._historyRead;
        }
        options = options || {};
        (0, node_opcua_assert_1.assert)(typeof this._timestamped_set_func !== "function", "UAVariable already bound");
        (0, node_opcua_assert_1.assert)(typeof this._timestamped_get_func !== "function", "UAVariable already bound");
        bind_getter.call(this, options);
        bind_setter.call(this, options);
        const _historyRead = options.historyRead;
        if (_historyRead) {
            (0, node_opcua_assert_1.assert)(typeof this._historyRead !== "function" || this._historyRead === UAVariableImpl.prototype._historyRead);
            (0, node_opcua_assert_1.assert)(typeof _historyRead === "function");
            this._historyRead = _historyRead;
            (0, node_opcua_assert_1.assert)(this._historyRead.length === 6);
        }
        // post conditions
        (0, node_opcua_assert_1.assert)(typeof this._timestamped_set_func === "function");
        (0, node_opcua_assert_1.assert)(this._timestamped_set_func.length === 2, "expecting 2 parameters");
    }
    readValueAsync(context, callback) {
        (0, node_opcua_assert_1.assert)(typeof callback === "function");
        context = context || session_context_1.SessionContext.defaultContext;
        this.__waiting_callbacks = this.__waiting_callbacks || [];
        this.__waiting_callbacks.push(callback);
        const _readValueAsync_in_progress = this.__waiting_callbacks.length >= 2;
        if (_readValueAsync_in_progress) {
            return;
        }
        if (this.isDisposed()) {
            return callback(null, new node_opcua_data_value_1.DataValue({ statusCode: node_opcua_status_code_1.StatusCodes.BadNodeIdUnknown }));
        }
        const readImmediate = (innerCallback) => {
            (0, node_opcua_assert_1.assert)(this.$dataValue instanceof node_opcua_data_value_1.DataValue);
            const dataValue = this.readValue(context);
            innerCallback(null, dataValue);
        };
        let func;
        if (!this.isReadable(context)) {
            func = (innerCallback) => {
                const dataValue = new node_opcua_data_value_1.DataValue({ statusCode: node_opcua_status_code_1.StatusCodes.BadNotReadable });
                innerCallback(null, dataValue);
            };
        }
        else if (!this.checkPermissionPrivate(context, node_opcua_types_1.PermissionType.Read)) {
            func = (innerCallback) => {
                const dataValue = new node_opcua_data_value_1.DataValue({ statusCode: node_opcua_status_code_1.StatusCodes.BadUserAccessDenied });
                innerCallback(null, dataValue);
            };
        }
        else if (!this.isUserReadable(context)) {
            func = (innerCallback) => {
                const dataValue = new node_opcua_data_value_1.DataValue({ statusCode: node_opcua_status_code_1.StatusCodes.BadNotReadable });
                innerCallback(null, dataValue);
            };
        }
        else {
            const clock = (0, node_opcua_date_time_1.getCurrentClock)();
            func = typeof this.refreshFunc === "function" ? this.asyncRefresh.bind(this, clock) : readImmediate;
        }
        const satisfy_callbacks = (err, dataValue) => {
            // now call all pending callbacks
            const callbacks = this.__waiting_callbacks || [];
            this.__waiting_callbacks = [];
            const n = callbacks.length;
            for (const callback1 of callbacks) {
                callback1.call(this, err, dataValue);
            }
        };
        try {
            func.call(this, satisfy_callbacks);
        }
        catch (err) {
            // istanbul ignore next
            if (doDebug) {
                debugLog(chalk_1.default.red("func readValueAsync has failed "));
                if (util_1.types.isNativeError(err)) {
                    debugLog(" stack", err.stack);
                }
            }
            satisfy_callbacks(err);
        }
    }
    getWriteMask() {
        return super.getWriteMask();
    }
    getUserWriteMask() {
        return super.getUserWriteMask();
    }
    clone(options, optionalFilter, extraInfo) {
        options = {
            ...options,
            // check this eventNotifier: this.eventNotifier,
            // check this symbolicName: this.symbolicName,
            accessLevel: this.accessLevel,
            arrayDimensions: this.arrayDimensions,
            dataType: this.dataType,
            historizing: this.historizing,
            minimumSamplingInterval: this.minimumSamplingInterval,
            userAccessLevel: this.userAccessLevel,
            valueRank: this.valueRank
        };
        const newVariable = (0, base_node_private_1._clone)(this, UAVariableImpl, options, optionalFilter || node_opcua_address_space_base_1.defaultCloneFilter, extraInfo || (0, node_opcua_address_space_base_1.makeDefaultCloneExtraInfo)(this));
        newVariable.bindVariable();
        (0, node_opcua_assert_1.assert)(typeof newVariable._timestamped_set_func === "function");
        (0, node_opcua_assert_1.assert)(newVariable.dataType === this.dataType);
        newVariable.$dataValue = this.$dataValue.clone();
        // also bind extension object
        const v = newVariable.$dataValue.value;
        if (v.dataType === node_opcua_variant_1.DataType.ExtensionObject && v.value && v.arrayType === node_opcua_variant_1.VariantArrayType.Scalar) {
            try {
                newVariable.bindExtensionObject(newVariable.$dataValue.value.value);
            }
            catch (err) {
                errorLog("Error binding extension objects");
                errorLog(err.message);
                errorLog(this.toString());
                errorLog("---------------------------------------");
                errorLog(this.$dataValue.toString());
                errorLog("---------------------------------------");
                errorLog(newVariable.$dataValue.toString());
                throw err;
            }
        }
        return newVariable;
    }
    getDataTypeNode() {
        const addressSpace = this.addressSpace;
        if (this.dataType.isEmpty()) {
            this.dataType = this.resolveNodeId(node_opcua_variant_1.DataType.Variant);
        }
        const dt = addressSpace.findNode(this.dataType);
        // istanbul ignore next
        if (!dt) {
            throw new Error("getDataTypeNode: cannot find dataType " + this.dataType.toString());
        }
        return dt;
    }
    get dataTypeObj() {
        return this.getDataTypeNode();
    }
    checkExtensionObjectIsCorrect(extObj) {
        if (!extObj) {
            return true;
        }
        const addressSpace = this.addressSpace;
        const dataType = addressSpace.findDataType(this.dataType);
        if (!dataType) {
            // may be we are in the process of loading a xml file and the corresponding dataType
            // has not yet been loaded !
            return true;
        }
        if (dataType.nodeId.namespace === 0 && dataType.nodeId.value === node_opcua_variant_1.DataType.ExtensionObject) {
            return true;
        }
        const Constructor = addressSpace.getExtensionObjectConstructor(this.dataType);
        if (this.valueRank === -1) {
            /** Scalar */
            if (extObj instanceof Array) {
                return false;
            }
            return checkExtensionObjectIsCorrectScalar.call(this, extObj);
        }
        else if (this.valueRank >= 1) {
            /** array */
            if (!(extObj instanceof Array)) {
                // let's coerce this scalar into an 1-element array if it is a valid extension object
                if (checkExtensionObjectIsCorrectScalar.call(this, extObj)) {
                    warningLog(`warning: checkExtensionObjectIsCorrect : expecting a array but got a scalar (value rank of '${this.browseName.toString()}' is 1)\nautomatic conversion from scalar to array with 1 element is taking place.`);
                    extObj = [extObj];
                }
                else {
                    return false;
                }
            }
            return checkExtensionObjectIsCorrectArray.call(this, extObj);
        }
        else if (this.valueRank === 0) {
            // Scalar or Array
            const isCorrectScalar = !Array.isArray(extObj) && checkExtensionObjectIsCorrectScalar.call(this, extObj);
            const isCorrectArray = Array.isArray(extObj) && checkExtensionObjectIsCorrectArray.call(this, extObj);
            return isCorrectArray || isCorrectScalar;
        }
        else {
            throw new Error(`checkExtensionObjectIsCorrect: Not Implemented case, please contact sterfive : this.valueRank =${this.valueRank}`);
        }
        function checkExtensionObjectIsCorrectScalar(extObj) {
            // istanbul ignore next
            if (!(extObj && extObj.constructor)) {
                errorLog(extObj);
                throw new Error("expecting an valid extension object");
            }
            return extObj.constructor.name === Constructor.name;
        }
        function checkExtensionObjectIsCorrectArray(extObjArray) {
            // istanbul ignore next
            for (const extObj of extObjArray) {
                if (!(extObj && extObj.constructor)) {
                    errorLog(extObj);
                    throw new Error("expecting an valid extension object");
                }
            }
            try {
                for (const e of extObjArray) {
                    if (!e) {
                        continue;
                    }
                    if (e.constructor.name !== Constructor.name) {
                        debugLog("extObj.constructor.name ", e.constructor.name, "expected", Constructor.name);
                        return false;
                    }
                }
                return true;
            }
            catch (err) {
                errorLog(err);
                return false;
            }
        }
    }
    changeDataType(newDataType, newValue) {
        changeUAVariableDataType(this, newDataType, newValue);
    }
    /**
     * @private
     * install UAVariable to exposed th
     *
     * precondition:
     */
    installExtensionObjectVariables() {
        (0, ua_variable_impl_ext_obj_1._installExtensionObjectBindingOnProperties)(this, { createMissingProp: true });
    }
    /**

     * @return {ExtensionObject}
     */
    bindExtensionObjectScalar(optionalExtensionObject, options) {
        (0, node_opcua_assert_1.assert)(this.valueRank === -1, "expecting an Scalar variable here");
        return (0, ua_variable_impl_ext_obj_1._bindExtensionObject)(this, optionalExtensionObject, options);
    }
    bindExtensionObjectArray(optionalExtensionObject, options) {
        (0, node_opcua_assert_1.assert)(this.valueRank >= 1, "expecting an Array or a Matrix variable here");
        return (0, ua_variable_impl_ext_obj_1._bindExtensionObjectArrayOrMatrix)(this, optionalExtensionObject, options);
    }
    bindExtensionObject(optionalExtensionObject, options) {
        // coerce to ExtensionObject[] when this.valueRank === 1
        if (optionalExtensionObject &&
            this.valueRank === 1 &&
            !Array.isArray(optionalExtensionObject) &&
            optionalExtensionObject instanceof node_opcua_extension_object_1.ExtensionObject) {
            warningLog("bindExtensionObject: coerce to ExtensionObject[] when valueRank === 1 and value is a scalar extension object");
            optionalExtensionObject = [optionalExtensionObject];
        }
        if (optionalExtensionObject) {
            if (optionalExtensionObject instanceof Array) {
                (0, node_opcua_assert_1.assert)(this.valueRank >= 1, "bindExtensionObject: expecting an Array of Matrix variable here");
                return (0, ua_variable_impl_ext_obj_1._bindExtensionObjectArrayOrMatrix)(this, optionalExtensionObject, options);
            }
            else {
                if (this.valueRank !== -1 && this.valueRank !== 0) {
                    throw new Error("bindExtensionObject: expecting an Scalar variable here but got value rank " + this.valueRank);
                }
                return (0, ua_variable_impl_ext_obj_1._bindExtensionObject)(this, optionalExtensionObject, options);
            }
        }
        (0, node_opcua_assert_1.assert)(optionalExtensionObject === undefined || optionalExtensionObject === null);
        if (this.valueRank === -1) {
            return (0, ua_variable_impl_ext_obj_1._bindExtensionObject)(this, undefined, options);
        }
        else if (this.valueRank === 1) {
            return (0, ua_variable_impl_ext_obj_1._bindExtensionObjectArrayOrMatrix)(this, undefined, options);
        }
        else if (this.valueRank > 1) {
            return (0, ua_variable_impl_ext_obj_1._bindExtensionObjectArrayOrMatrix)(this, undefined, options);
        }
        //  unsupported case ...
        return null;
    }
    updateExtensionObjectPartial(partialExtensionObject) {
        (0, ua_variable_impl_ext_obj_1.setExtensionObjectPartialValue)(this, partialExtensionObject);
        return this.$extensionObject;
    }
    incrementExtensionObjectPartial(path) {
        const extensionObject = this.readValue().value.value;
        const partialData = (0, ua_variable_impl_ext_obj_1.extractPartialData)(path, extensionObject);
        (0, ua_variable_impl_ext_obj_1.incrementElement)(path, partialData);
        (0, ua_variable_impl_ext_obj_1.setExtensionObjectPartialValue)(this, partialData);
    }
    toString() {
        const options = new base_node_private_1.ToStringBuilder();
        base_node_private_1.UAVariable_toString.call(this, options);
        return options.toString();
    }
    historyRead(context, historyReadDetails, indexRange, dataEncoding, continuationData, callback) {
        (0, node_opcua_assert_1.assert)(context instanceof session_context_1.SessionContext);
        (0, node_opcua_assert_1.assert)(typeof callback === "function");
        if (typeof this._historyRead !== "function") {
            return callback(null, new node_opcua_types_1.HistoryReadResult({ statusCode: node_opcua_status_code_1.StatusCodes.BadNotReadable }));
        }
        this._historyRead(context, historyReadDetails, indexRange, dataEncoding, continuationData, callback);
    }
    _historyReadRaw(context, historyReadRawModifiedDetails, indexRange, dataEncoding, continuationData, callback) {
        throw new Error("");
    }
    _historyReadRawModify(context, historyReadRawModifiedDetails, indexRange, dataEncoding, continuationData, callback) {
        throw new Error("");
    }
    _historyRead(context, historyReadDetails, indexRange, dataEncoding, continuationData, callback) {
        if (!this.checkPermissionPrivate(context, node_opcua_types_1.PermissionType.ReadHistory)) {
            const result = new node_opcua_types_1.HistoryReadResult({
                statusCode: node_opcua_status_code_1.StatusCodes.BadUserAccessDenied
            });
            callback(null, result);
        }
        if (!this.canUserReadHistory(context)) {
            const result = new node_opcua_types_1.HistoryReadResult({
                statusCode: node_opcua_status_code_1.StatusCodes.BadHistoryOperationUnsupported
            });
            callback(null, result);
        }
        const result = new node_opcua_types_1.HistoryReadResult({
            statusCode: node_opcua_status_code_1.StatusCodes.BadHistoryOperationUnsupported
        });
        callback(null, result);
    }
    _historyPush(newDataValue) {
        throw new Error("");
    }
    _historyReadRawAsync(historyReadRawModifiedDetails, maxNumberToExtract, isReversed, reverseDataValue, callback) {
        throw new Error("");
    }
    _historyReadModify(context, historyReadRawModifiedDetails, indexRange, dataEncoding, continuationData, callback) {
        throw new Error("");
    }
    _update_startOfOnlineArchive(newDate) {
        // please install
        throw new Error("");
    }
    _update_startOfArchive(newDate) {
        throw new Error("");
    }
    _validate_DataType(variantDataType) {
        return (0, validate_data_type_correctness_1.validateDataTypeCorrectness)(this.addressSpace, this.dataType, variantDataType, /* allow Nulls */ false, this.nodeId);
    }
    _internal_set_value(value) {
        if (value.dataType !== node_opcua_variant_1.DataType.Null) {
            this.verifyVariantCompatibility(value);
        }
        this.$dataValue.value = value;
    }
    _internal_set_dataValue(dataValue, indexRange) {
        (0, node_opcua_assert_1.assert)(dataValue, "expecting a dataValue");
        (0, node_opcua_assert_1.assert)(dataValue instanceof node_opcua_data_value_1.DataValue, "expecting dataValue to be a DataValue");
        (0, node_opcua_assert_1.assert)(dataValue !== this.$dataValue, "expecting dataValue to be different from previous DataValue instance");
        const addressSpace = this.addressSpace;
        // istanbul ignore next
        if (!addressSpace) {
            warningLog("UAVariable#_internal_set_dataValue : no addressSpace ! may be node has already been deleted ?");
            return;
        }
        // istanbul ignore next
        if (dataValue.value.arrayType === node_opcua_variant_1.VariantArrayType.Matrix) {
            if (!dataValue.value.dimensions) {
                throw new Error("missing dimensions: a Matrix Variant needs a dimension");
            }
            const nbElements = dataValue.value.dimensions.reduce((acc, x) => acc * x, 1);
            if (dataValue.value.value.length !== 0 && dataValue.value.value.length !== nbElements) {
                throw new Error(`Internal Error: matrix dimension doesn't match the number of element in the array : ${dataValue.toString()} "\n expecting ${nbElements} elements but got ${dataValue.value.value.length}`);
            }
        }
        // istanbul ignore next
        if (dataValue.value.dataType === node_opcua_variant_1.DataType.ExtensionObject) {
            // istanbul ignore next
            if (!this.checkExtensionObjectIsCorrect(dataValue.value.value)) {
                warningLog(dataValue.toString());
                throw new Error("Invalid Extension Object on nodeId =" + this.nodeId.toString());
            }
        }
        this.verifyVariantCompatibility(dataValue.value);
        this._inner_replace_dataValue(dataValue, indexRange);
    }
    /**
     * @private
     */
    _inner_replace_dataValue(dataValue, indexRange) {
        (0, node_opcua_assert_1.assert)(this.$dataValue.value instanceof node_opcua_variant_1.Variant);
        const old_dataValue = this.$dataValue.clone();
        if (this.$$extensionObjectArray && dataValue.value.arrayType !== node_opcua_variant_1.VariantArrayType.Scalar) {
            // we have a bounded array or matrix
            (0, node_opcua_assert_1.assert)(Array.isArray(dataValue.value.value));
            if (this.$$extensionObjectArray !== this.$dataValue.value.value) {
                throw new Error("internal error");
            }
            this.$$extensionObjectArray = dataValue.value.value;
            this.$dataValue.value.value = dataValue.value.value;
            this.$dataValue.statusCode = dataValue.statusCode || node_opcua_status_code_1.StatusCodes.Good;
            this.$dataValue.serverTimestamp = dataValue.serverTimestamp;
            this.$dataValue.serverPicoseconds = dataValue.serverPicoseconds;
            this.$dataValue.sourceTimestamp = dataValue.sourceTimestamp;
            this.$dataValue.sourcePicoseconds = dataValue.sourcePicoseconds;
        }
        else if (this._basicDataType === node_opcua_variant_1.DataType.ExtensionObject &&
            this.valueRank === -1 &&
            this.$set_ExtensionObject &&
            dataValue.value.arrayType === node_opcua_variant_1.VariantArrayType.Scalar) {
            // the entire extension object is changed.
            this.$dataValue.statusCode = this.$dataValue.statusCode || node_opcua_status_code_1.StatusCodes.Good;
            const preciseClock = (0, node_opcua_date_time_1.coerceClock)(this.$dataValue.sourceTimestamp, this.$dataValue.sourcePicoseconds);
            this.$set_ExtensionObject(dataValue.value.value, preciseClock, new Set());
        }
        else {
            this.$dataValue = dataValue;
            this.$dataValue.statusCode = this.$dataValue.statusCode || node_opcua_status_code_1.StatusCodes.Good;
        }
        // repair missing timestamps
        const now = new Date();
        if (!dataValue.serverTimestamp) {
            this.$dataValue.serverTimestamp = old_dataValue.serverTimestamp || now;
            this.$dataValue.serverPicoseconds = old_dataValue.serverPicoseconds || 0;
        }
        if (!dataValue.sourceTimestamp) {
            this.$dataValue.sourceTimestamp = old_dataValue.sourceTimestamp || now;
            this.$dataValue.sourcePicoseconds = old_dataValue.sourcePicoseconds || 0;
        }
        if (!(0, node_opcua_data_value_1.sameDataValue)(old_dataValue, dataValue)) {
            if (this.getBasicDataType() === node_opcua_variant_1.DataType.ExtensionObject) {
                const preciseClock = (0, node_opcua_date_time_1.coerceClock)(this.$dataValue.sourceTimestamp, this.$dataValue.sourcePicoseconds);
                const cache = new Set();
                if (this.$$extensionObjectArray) {
                    this.touchValue(preciseClock);
                    (0, ua_variable_impl_ext_obj_1.propagateTouchValueDownwardArray)(this, preciseClock, cache);
                }
                else {
                    this.touchValue(preciseClock);
                    (0, ua_variable_impl_ext_obj_1.propagateTouchValueDownward)(this, preciseClock, cache);
                }
            }
            else {
                this.emit("value_changed", this.$dataValue.clone(), indexRange);
            }
        }
    }
    _conditionRefresh(_cache) {
        apply_condition_refresh_1.apply_condition_refresh.call(this, _cache);
    }
    handle_semantic_changed() {
        this.semantic_version = this.semantic_version + 1;
        this.emit("semantic_changed");
    }
    _readDataType() {
        (0, node_opcua_assert_1.assert)(this.dataType instanceof node_opcua_nodeid_1.NodeId);
        const options = {
            statusCode: node_opcua_status_code_1.StatusCodes.Good,
            value: {
                dataType: node_opcua_variant_1.DataType.NodeId,
                value: this.dataType
            }
        };
        return new node_opcua_data_value_1.DataValue(options);
    }
    _readValueRank() {
        (0, node_opcua_assert_1.assert)(typeof this.valueRank === "number");
        const options = {
            statusCode: node_opcua_status_code_1.StatusCodes.Good,
            value: { dataType: node_opcua_variant_1.DataType.Int32, value: this.valueRank }
        };
        return new node_opcua_data_value_1.DataValue(options);
    }
    _readArrayDimensions() {
        (0, node_opcua_assert_1.assert)(Array.isArray(this.arrayDimensions) || this.arrayDimensions === null);
        (0, node_opcua_assert_1.assert)(!this.arrayDimensions || this.valueRank > 0, "arrayDimension must be null if valueRank <0");
        const options = {
            statusCode: node_opcua_status_code_1.StatusCodes.Good,
            value: { dataType: node_opcua_variant_1.DataType.UInt32, arrayType: node_opcua_variant_1.VariantArrayType.Array, value: this.arrayDimensions }
        };
        return new node_opcua_data_value_1.DataValue(options);
    }
    _readAccessLevel(context) {
        (0, node_opcua_assert_1.assert)(context instanceof session_context_1.SessionContext);
        const options = {
            statusCode: node_opcua_status_code_1.StatusCodes.Good,
            value: { dataType: node_opcua_variant_1.DataType.Byte, value: (0, node_opcua_data_model_1.convertAccessLevelFlagToByte)(this.accessLevel) }
        };
        return new node_opcua_data_value_1.DataValue(options);
    }
    _readAccessLevelEx(context) {
        (0, node_opcua_assert_1.assert)(context instanceof session_context_1.SessionContext);
        const options = {
            statusCode: node_opcua_status_code_1.StatusCodes.Good,
            // Extra flags are not supported yet. to do:
            value: { dataType: node_opcua_variant_1.DataType.UInt32, value: (0, node_opcua_data_model_1.convertAccessLevelFlagToByte)(this.accessLevel) }
        };
        return new node_opcua_data_value_1.DataValue(options);
    }
    _readUserAccessLevel(context) {
        (0, node_opcua_assert_1.assert)(context instanceof session_context_1.SessionContext);
        const effectiveUserAccessLevel = _calculateEffectiveUserAccessLevelFromPermission(this, context, this.userAccessLevel);
        const options = {
            value: {
                dataType: node_opcua_variant_1.DataType.Byte,
                statusCode: node_opcua_status_code_1.StatusCodes.Good,
                value: (0, node_opcua_data_model_1.convertAccessLevelFlagToByte)(effectiveUserAccessLevel)
            }
        };
        return new node_opcua_data_value_1.DataValue(options);
    }
    _readMinimumSamplingInterval() {
        // expect a Duration => Double
        const options = {};
        if (this.minimumSamplingInterval === undefined) {
            options.statusCode = node_opcua_status_code_1.StatusCodes.BadAttributeIdInvalid;
        }
        else {
            options.value = { dataType: node_opcua_variant_1.DataType.Double, value: this.minimumSamplingInterval };
            options.statusCode = node_opcua_status_code_1.StatusCodes.Good;
        }
        return new node_opcua_data_value_1.DataValue(options);
    }
    _readHistorizing() {
        (0, node_opcua_assert_1.assert)(typeof this.historizing === "boolean");
        const options = {
            statusCode: node_opcua_status_code_1.StatusCodes.Good,
            value: { dataType: node_opcua_variant_1.DataType.Boolean, value: !!this.historizing }
        };
        return new node_opcua_data_value_1.DataValue(options);
    }
}
exports.UAVariableImpl = UAVariableImpl;
// tslint:disable:no-var-requires
const thenify_ex_1 = require("thenify-ex");
UAVariableImpl.prototype.asyncRefresh = (0, thenify_ex_1.withCallback)(UAVariableImpl.prototype.asyncRefresh);
UAVariableImpl.prototype.writeValue = (0, thenify_ex_1.withCallback)(UAVariableImpl.prototype.writeValue);
UAVariableImpl.prototype.writeAttribute = (0, thenify_ex_1.withCallback)(UAVariableImpl.prototype.writeAttribute);
UAVariableImpl.prototype.historyRead = (0, thenify_ex_1.withCallback)(UAVariableImpl.prototype.historyRead);
UAVariableImpl.prototype.readValueAsync = (0, thenify_ex_1.withCallback)(UAVariableImpl.prototype.readValueAsync);
function check_valid_array(dataType, array) {
    if (Array.isArray(array)) {
        return true;
    }
    switch (dataType) {
        case node_opcua_variant_1.DataType.Double:
            return array instanceof Float64Array;
        case node_opcua_variant_1.DataType.Float:
            return array instanceof Float32Array;
        case node_opcua_variant_1.DataType.Int32:
            return array instanceof Int32Array;
        case node_opcua_variant_1.DataType.Int16:
            return array instanceof Int16Array;
        case node_opcua_variant_1.DataType.SByte:
            return array instanceof Int8Array;
        case node_opcua_variant_1.DataType.UInt32:
            return array instanceof Uint32Array;
        case node_opcua_variant_1.DataType.UInt16:
            return array instanceof Uint16Array;
        case node_opcua_variant_1.DataType.Byte:
            return array instanceof Uint8Array || array instanceof Buffer;
    }
    return false;
}
// function _apply_default_timestamps(dataValue: DataValue): void {
//     const now = getCurrentClock();
//     assert(dataValue instanceof DataValue);
//     if (!dataValue.sourceTimestamp) {
//         dataValue.sourceTimestamp = now.timestamp;
//         dataValue.sourcePicoseconds = now.picoseconds;
//     }
//     if (!dataValue.serverTimestamp) {
//         dataValue.serverTimestamp = now.timestamp;
//         dataValue.serverPicoseconds = now.picoseconds;
//     }
// }
function unsetFlag(flags, mask) {
    return flags & ~mask;
}
function setFlag(flags, mask) {
    return flags | mask;
}
function _calculateEffectiveUserAccessLevelFromPermission(node, context, userAccessLevel) {
    function __adjustFlag(permissionType, access, userAccessLevel1) {
        if ((node.accessLevel & access) === 0 || (userAccessLevel1 & access) === 0) {
            userAccessLevel1 = unsetFlag(userAccessLevel1, access);
        }
        else {
            if (!context.checkPermission(node, permissionType)) {
                userAccessLevel1 = unsetFlag(userAccessLevel1, access);
            }
        }
        return userAccessLevel1;
    }
    userAccessLevel = node.userAccessLevel === undefined ? node.accessLevel : node.userAccessLevel & node.accessLevel;
    if (context.checkPermission) {
        (0, node_opcua_assert_1.assert)(context.checkPermission instanceof Function);
        userAccessLevel = __adjustFlag(node_opcua_types_1.PermissionType.Read, node_opcua_data_model_1.AccessLevelFlag.CurrentRead, userAccessLevel);
        userAccessLevel = __adjustFlag(node_opcua_types_1.PermissionType.Write, node_opcua_data_model_1.AccessLevelFlag.CurrentWrite, userAccessLevel);
        userAccessLevel = __adjustFlag(node_opcua_types_1.PermissionType.Write, node_opcua_data_model_1.AccessLevelFlag.StatusWrite, userAccessLevel);
        userAccessLevel = __adjustFlag(node_opcua_types_1.PermissionType.Write, node_opcua_data_model_1.AccessLevelFlag.TimestampWrite, userAccessLevel);
        userAccessLevel = __adjustFlag(node_opcua_types_1.PermissionType.ReadHistory, node_opcua_data_model_1.AccessLevelFlag.HistoryRead, userAccessLevel);
        userAccessLevel = __adjustFlag(node_opcua_types_1.PermissionType.DeleteHistory, node_opcua_data_model_1.AccessLevelFlag.HistoryWrite, userAccessLevel);
        return userAccessLevel;
    }
    else {
        return userAccessLevel;
    }
}
function adjustVariant2(variant) {
    // convert Variant( Scalar|ByteString) =>  Variant(Array|ByteArray)
    const addressSpace = this.addressSpace;
    const basicType = this.getBasicDataType();
    variant = (0, node_opcua_variant_1.adjustVariant)(variant, this.valueRank, basicType);
    return variant;
}
function _not_writable_timestamped_set_func(dataValue, callback) {
    (0, node_opcua_assert_1.assert)(dataValue instanceof node_opcua_data_value_1.DataValue);
    callback(null, node_opcua_status_code_1.StatusCodes.BadNotWritable, null);
}
function _default_writable_timestamped_set_func(dataValue, callback) {
    (0, node_opcua_assert_1.assert)(dataValue instanceof node_opcua_data_value_1.DataValue);
    callback(null, node_opcua_status_code_1.StatusCodes.Good, dataValue);
}
function turn_sync_to_async(f, numberOfArgs) {
    if (f.length <= numberOfArgs) {
        return function (data, callback) {
            new Promise((resolve, reject) => {
                try {
                    // const ff1 = f as f1<T, D, R>;
                    const ff2 = f;
                    const r = ff2.call(this, data);
                    if (r instanceof Promise) {
                        r.then(resolve, reject);
                    }
                    else {
                        resolve(r);
                    }
                }
                catch (err) {
                    reject(err);
                }
            })
                .then((r) => {
                callback(null, r);
            })
                .catch((err) => {
                callback(err);
            });
        };
    }
    else {
        (0, node_opcua_assert_1.assert)(f.length === numberOfArgs + 1);
        return f;
    }
}
const _default_minimumSamplingInterval = 1000;
function coerceDataValue(dataValue) {
    if (dataValue instanceof node_opcua_data_value_1.DataValue) {
        return dataValue;
    }
    return new node_opcua_data_value_1.DataValue(dataValue);
}
// variation #3 :
function _Variable_bind_with_async_refresh(options) {
    (0, node_opcua_assert_1.assert)(this instanceof UAVariableImpl);
    (0, node_opcua_assert_1.assert)(typeof options.refreshFunc === "function");
    (0, node_opcua_assert_1.assert)(!options.get, "a getter shall not be specified when refreshFunc is set");
    (0, node_opcua_assert_1.assert)(!options.timestamped_get, "a getter shall not be specified when refreshFunc is set");
    (0, node_opcua_assert_1.assert)(!this.refreshFunc);
    this.refreshFunc = options.refreshFunc;
    // TO DO : REVISIT THIS ASSUMPTION
    if (false && this.minimumSamplingInterval === 0) {
        // when a getter /timestamped_getter or async_getter is provided
        // samplingInterval cannot be 0, as the item value must be scanned to be updated.
        this.minimumSamplingInterval = _default_minimumSamplingInterval; // MonitoredItem.minimumSamplingInterval;
        debugLog("adapting minimumSamplingInterval on " + this.browseName.toString() + " to " + this.minimumSamplingInterval);
    }
}
// variation 2
function _Variable_bind_with_timestamped_get(options) {
    /* jshint validthis: true */
    (0, node_opcua_assert_1.assert)(this instanceof UAVariableImpl);
    (0, node_opcua_assert_1.assert)(typeof options.timestamped_get === "function");
    (0, node_opcua_assert_1.assert)(!options.get, "should not specify 'get' when 'timestamped_get' exists ");
    (0, node_opcua_assert_1.assert)(!this._timestamped_get_func);
    const async_refresh_func = (callback) => {
        Promise.resolve(this._timestamped_get_func.call(this))
            .then((dataValue) => callback(null, dataValue))
            .catch((err) => {
            errorLog("asyncRefresh error: Variable is  ", this.nodeId.toString(), this.browseName.toString());
            callback(err);
        });
    };
    const pThis = this;
    if (options.timestamped_get.length === 0) {
        const timestamped_get = options.timestamped_get;
        // sync version | Promise version
        this._timestamped_get_func = timestamped_get;
        const dataValue_verify = timestamped_get.call(pThis);
        // dataValue_verify should be a DataValue or a Promise
        /* istanbul ignore next */
        if (!(dataValue_verify instanceof node_opcua_data_value_1.DataValue) && typeof dataValue_verify.then !== "function") {
            errorLog(chalk_1.default.red(" Bind variable error: "), " the timestamped_get function must return a DataValue or a Promise<DataValue>" +
                "\n value_check.constructor.name ", dataValue_verify ? dataValue_verify.constructor.name : "null");
            throw new Error(" Bind variable error: " + " the timestamped_get function must return a DataValue");
        }
        _Variable_bind_with_async_refresh.call(this, { refreshFunc: async_refresh_func });
    }
    else if (options.timestamped_get.length === 1) {
        _Variable_bind_with_async_refresh.call(this, { refreshFunc: options.timestamped_get });
    }
    else {
        errorLog("timestamped_get has a invalid number of argument , should be 0 or 1  ");
        throw new Error("timestamped_get has a invalid number of argument , should be 0 or 1  ");
    }
}
// variation 1
function _Variable_bind_with_simple_get(options) {
    (0, node_opcua_assert_1.assert)(this instanceof UAVariableImpl);
    (0, node_opcua_assert_1.assert)(typeof options.get === "function", "should specify get function");
    (0, node_opcua_assert_1.assert)(options.get.length === 0, "get function should not have arguments");
    (0, node_opcua_assert_1.assert)(!options.timestamped_get, "should not specify a timestamped_get function when get is specified");
    (0, node_opcua_assert_1.assert)(!this._timestamped_get_func);
    (0, node_opcua_assert_1.assert)(!this._get_func);
    this._get_func = options.get;
    const timestamped_get_func_from__Variable_bind_with_simple_get = () => {
        const value = this._get_func();
        /* istanbul ignore next */
        if (!is_Variant_or_StatusCode(value)) {
            errorLog(chalk_1.default.red(" Bind variable error: "), " : the getter must return a Variant or a StatusCode" + "\nvalue_check.constructor.name ", value ? value.constructor.name : "null");
            throw new Error(" bindVariable : the value getter function returns a invalid result ( expecting a Variant or a StatusCode !!!");
        }
        if (is_StatusCode(value)) {
            return new node_opcua_data_value_1.DataValue({ statusCode: value });
        }
        else {
            if (!this.$dataValue ||
                !this.$dataValue.statusCode.isGoodish() ||
                !(0, node_opcua_variant_1.sameVariant)(this.$dataValue.value, value)) {
                // rebuilding artificially timestamps with current clock as they are not provided
                // by the underlying getter function
                const { timestamp: sourceTimestamp, picoseconds: sourcePicoseconds } = (0, node_opcua_date_time_1.getCurrentClock)();
                const serverTimestamp = sourceTimestamp;
                const serverPicoseconds = sourcePicoseconds;
                this._inner_replace_dataValue(new node_opcua_data_value_1.DataValue({ value, sourceTimestamp, sourcePicoseconds, serverTimestamp, serverPicoseconds }));
            }
            return this.$dataValue;
        }
    };
    _Variable_bind_with_timestamped_get.call(this, {
        get: undefined,
        timestamped_get: timestamped_get_func_from__Variable_bind_with_simple_get
    });
}
function _Variable_bind_with_simple_set(options) {
    (0, node_opcua_assert_1.assert)(this instanceof UAVariableImpl);
    (0, node_opcua_assert_1.assert)(typeof options.set === "function", "should specify set function");
    (0, node_opcua_assert_1.assert)(!options.timestamped_set, "should not specify a timestamped_set function");
    (0, node_opcua_assert_1.assert)(!this._timestamped_set_func);
    (0, node_opcua_assert_1.assert)(!this._set_func);
    //
    this._set_func = turn_sync_to_async(options.set, 1);
    (0, node_opcua_assert_1.assert)(this._set_func.length === 2, " set function must have 2 arguments ( variant, callback)");
    this._timestamped_set_func = (timestamped_value, callback) => {
        (0, node_opcua_assert_1.assert)(timestamped_value instanceof node_opcua_data_value_1.DataValue);
        this._set_func(timestamped_value.value, (err, statusCode) => {
            // istanbul ignore next
            if (!err && !statusCode) {
                errorLog(chalk_1.default.red("UAVariable Binding Error _set_func must return a StatusCode, check the bindVariable parameters"));
                errorLog(chalk_1.default.yellow("StatusCode.Good is assumed"));
                return callback(err, node_opcua_status_code_1.StatusCodes.Good, timestamped_value);
            }
            if (statusCode && statusCode.isNotGood()) {
                // record the value but still record the statusCode !
                timestamped_value.statusCode = statusCode;
            }
            callback(err, statusCode, timestamped_value);
        });
    };
}
function _Variable_bind_with_timestamped_set(options) {
    (0, node_opcua_assert_1.assert)(options.timestamped_set.length === 2 || options.timestamped_set.length === 1, "timestamped_set must have 2 parameters  timestamped_set: function(dataValue,callback){} or one paramater  timestamped_set: function(dataValue): Promise<StatusCode>{}");
    (0, node_opcua_assert_1.assert)(!options.set, "should not specify set when timestamped_set_func exists ");
    this._timestamped_set_func = (0, multiform_func_1.convertToCallbackFunction1)(options.timestamped_set);
}
function bind_setter(options) {
    if (options.set && typeof options.set === "function") {
        // variation 1
        if (options.timestamped_set === undefined) {
            _Variable_bind_with_simple_set.call(this, options);
        }
        else {
            throw new Error("bind_setter : options should not specify both set and timestamped_set ");
        }
    }
    else if (typeof options.timestamped_set === "function") {
        // variation 2
        (0, node_opcua_assert_1.assert)(typeof options.timestamped_get === "function", "timestamped_set must be used with timestamped_get ");
        _Variable_bind_with_timestamped_set.call(this, {
            set: undefined,
            timestamped_set: options.timestamped_set
        });
    }
    else if (typeof options.timestamped_get === "function") {
        // timestamped_get is  specified but timestamped_set is not
        // => Value is read-only
        _Variable_bind_with_timestamped_set.call(this, {
            set: undefined,
            timestamped_set: _not_writable_timestamped_set_func
        });
    }
    else {
        _Variable_bind_with_timestamped_set.call(this, {
            set: undefined,
            timestamped_set: _default_writable_timestamped_set_func
        });
    }
}
function bind_getter(options) {
    if (typeof options.get === "function") {
        // variation 1
        _Variable_bind_with_simple_get.call(this, options);
    }
    else if (typeof options.timestamped_get === "function") {
        // variation 2
        _Variable_bind_with_timestamped_get.call(this, {
            get: undefined,
            timestamped_get: options.timestamped_get
        });
    }
    else if (typeof options.refreshFunc === "function") {
        // variation 3
        if (options.get !== undefined || options.timestamped_get !== undefined) {
            throw new Error("bind_getter : options should not specify both get and timestamped_get ");
        }
        _Variable_bind_with_async_refresh.call(this, { refreshFunc: options.refreshFunc });
    }
    else {
        (0, node_opcua_assert_1.assert)(!Object.prototype.hasOwnProperty.call(options, "set"), "getter is missing : a getter must be provided if a setter is provided");
        // xx bind_variant.call(this,options);
        if (options.dataType !== undefined) {
            // if (options.dataType !== DataType.ExtensionObject) {
            this.setValueFromSource(options);
            // }
        }
    }
}
class UAVariableImplT extends UAVariableImpl {
}
exports.UAVariableImplT = UAVariableImplT;
function changeUAVariableDataType(uaVariable, newDataType, valueLike) {
    let value = valueLike ? (valueLike instanceof node_opcua_variant_1.Variant ? valueLike : new node_opcua_variant_1.Variant(valueLike)) : null;
    const addressSpace = uaVariable.addressSpace;
    const newDataTypeNode = addressSpace.findNode(newDataType);
    // istanbul ignore next
    if (!newDataTypeNode || !(newDataTypeNode instanceof ua_data_type_impl_1.UADataTypeImpl)) {
        throw new Error("Cannot find newDataTypeNode " + newDataType.toString());
    }
    const newBaseDataType = newDataTypeNode.basicDataType;
    // istanbul ignore next
    if (newBaseDataType === node_opcua_variant_1.DataType.Null) {
        throw new Error("newDataTypeNode must be a DataType");
    }
    if (!value) {
        value = uaVariable.readValue().value;
        value.dataType = newBaseDataType;
    }
    if (newBaseDataType !== value.dataType) {
        throw new Error("newDataTypeNode must be of the same base dataType as the value");
    }
    // change uaVariable.dataType
    uaVariable.dataType = newDataTypeNode.nodeId;
    uaVariable._basicDataType = null;
    uaVariable.$dataValue.value.dataType = newDataType;
    uaVariable.$dataValue.value = value;
    // also change the value to ensure that we have a default value with the correct dataType
    uaVariable._internal_set_value(value);
}
//# sourceMappingURL=ua_variable_impl.js.map