"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.UAObjectTypeImpl = void 0;
/**
 * @module node-opcua-address-space
 */
const node_opcua_assert_1 = require("node-opcua-assert");
const node_opcua_data_model_1 = require("node-opcua-data-model");
const node_opcua_data_model_2 = require("node-opcua-data-model");
const node_opcua_data_value_1 = require("node-opcua-data-value");
const node_opcua_status_code_1 = require("node-opcua-status-code");
const node_opcua_utils_1 = require("node-opcua-utils");
const node_opcua_variant_1 = require("node-opcua-variant");
const session_context_1 = require("../source/session_context");
const base_node_impl_1 = require("./base_node_impl");
const base_node_private_1 = require("./base_node_private");
const tool_isSubtypeOf_1 = require("./tool_isSubtypeOf");
const ua_variable_type_impl_1 = require("./ua_variable_type_impl");
const _instantiate_helpers_1 = require("./_instantiate_helpers");
class UAObjectTypeImpl extends base_node_impl_1.BaseNodeImpl {
    nodeClass = node_opcua_data_model_1.NodeClass.ObjectType;
    isAbstract;
    /**
     * returns true if the object has some opcua methods
     */
    get hasMethods() {
        return this.getMethods().length > 0;
    }
    get subtypeOf() {
        return tool_isSubtypeOf_1.get_subtypeOf.call(this);
    }
    get subtypeOfObj() {
        return tool_isSubtypeOf_1.get_subtypeOfObj.call(this);
    }
    isSubtypeOf = (0, tool_isSubtypeOf_1.construct_isSubtypeOf)(UAObjectTypeImpl);
    /** @deprecated - use  isSubtypeOf instead */
    isSupertypeOf = (0, tool_isSubtypeOf_1.construct_isSubtypeOf)(UAObjectTypeImpl);
    constructor(options) {
        super(options);
        this.isAbstract = (0, node_opcua_utils_1.isNullOrUndefined)(options.isAbstract) ? false : options.isAbstract;
    }
    readAttribute(context, attributeId) {
        (0, node_opcua_assert_1.assert)(context instanceof session_context_1.SessionContext);
        const options = {};
        switch (attributeId) {
            case node_opcua_data_model_2.AttributeIds.IsAbstract:
                options.value = {
                    dataType: node_opcua_variant_1.DataType.Boolean,
                    value: !!this.isAbstract
                };
                options.statusCode = node_opcua_status_code_1.StatusCodes.Good;
                break;
            default:
                return base_node_impl_1.BaseNodeImpl.prototype.readAttribute.call(this, context, attributeId);
        }
        return new node_opcua_data_value_1.DataValue(options);
    }
    /**
     * instantiate an object of this UAObjectType
     * The instantiation takes care of object type inheritance when constructing inner properties and components.

     * @param options
     * @param options.browseName
     * @param [options.description]
     * @param [options.organizedBy] the parent Folder holding this object
     * @param [options.componentOf] the parent Object holding this object
     * @param [options.addInOf]
     * @param [options.notifierOf]
     * @param [options.eventSourceOf]
     * @param [options.optionals = [] name of the optional child to create
     * @param [options.modellingRule]
     * @param [options.encodingOf]
     *
     *
     * Note : HasComponent usage scope
     *
     *    Source          |     Destination
     * -------------------+---------------------------
     *  Object            | Object, Variable,Method
     *  ObjectType        |
     * -------------------+---------------------------
     *  DataVariable      | Variable
     *  DataVariableType  |
     */
    instantiate(options) {
        const addressSpace = this.addressSpace;
        (0, node_opcua_assert_1.assert)(!this.isAbstract, "cannot instantiate abstract UAObjectType");
        (0, node_opcua_assert_1.assert)(options, "missing option object");
        (0, node_opcua_assert_1.assert)(typeof options.browseName === "string" || (options.browseName !== null && typeof options.browseName === "object"), "expecting a browse name");
        (0, node_opcua_assert_1.assert)(!Object.prototype.hasOwnProperty.call(options, "propertyOf"), "an Object shall not be a PropertyOf an other object");
        (0, node_opcua_assert_1.assert)(!Object.prototype.hasOwnProperty.call(options, "optional"), "do you mean optionals ?");
        (0, ua_variable_type_impl_1.assertUnusedChildBrowseName)(addressSpace, options);
        const baseObjectType = addressSpace.findObjectType("BaseObjectType");
        (0, node_opcua_assert_1.assert)(baseObjectType, "BaseObjectType must be defined in the address space");
        const references = [];
        const copyAlsoModellingRules = (0, ua_variable_type_impl_1.topMostParentIsObjectTypeOrVariableType)(addressSpace, options);
        const opts = {
            browseName: options.browseName,
            componentOf: options.componentOf,
            displayName: options.displayName || "",
            description: options.description === undefined ? this.description?.clone() : options.description,
            encodingOf: options.encodingOf,
            eventSourceOf: options.eventSourceOf,
            notifierOf: options.notifierOf,
            organizedBy: options.organizedBy,
            addInOf: options.addInOf,
            references,
            typeDefinition: this.nodeId,
            nodeId: options.nodeId,
            eventNotifier: options.eventNotifier === undefined ? 0 : options.eventNotifier,
            modellingRule: options.modellingRule
        };
        const namespace = options.namespace || this.addressSpace.getOwnNamespace();
        const instance = namespace.addObject(opts);
        const copyAlsoAllOptionals = options.copyAlsoAllOptionals || false;
        (0, _instantiate_helpers_1.initialize_properties_and_components)(instance, baseObjectType, this, copyAlsoModellingRules, copyAlsoAllOptionals, options.optionals);
        (0, node_opcua_assert_1.assert)(instance.typeDefinition.toString() === this.nodeId.toString());
        instance.install_extra_properties();
        if (this._postInstantiateFunc) {
            this._postInstantiateFunc(instance, this, options);
        }
        return instance;
    }
    toString() {
        const options = new base_node_private_1.ToStringBuilder();
        base_node_private_1.UAObjectType_toString.call(this, options);
        return options.toString();
    }
}
exports.UAObjectTypeImpl = UAObjectTypeImpl;
//# sourceMappingURL=ua_object_type_impl.js.map