"use strict";
var __importDefault = (this && this.__importDefault) || function (mod) {
    return (mod && mod.__esModule) ? mod : { "default": mod };
};
Object.defineProperty(exports, "__esModule", { value: true });
exports.UAObjectImpl = void 0;
/**
 * @module node-opcua-address-space
 */
const chalk_1 = __importDefault(require("chalk"));
const node_opcua_assert_1 = require("node-opcua-assert");
const node_opcua_basic_types_1 = require("node-opcua-basic-types");
const node_opcua_data_model_1 = require("node-opcua-data-model");
const node_opcua_data_model_2 = require("node-opcua-data-model");
const node_opcua_data_value_1 = require("node-opcua-data-value");
const node_opcua_date_time_1 = require("node-opcua-date-time");
const node_opcua_nodeid_1 = require("node-opcua-nodeid");
const node_opcua_status_code_1 = require("node-opcua-status-code");
const node_opcua_variant_1 = require("node-opcua-variant");
const node_opcua_address_space_base_1 = require("node-opcua-address-space-base");
const node_opcua_debug_1 = require("node-opcua-debug");
const base_node_impl_1 = require("./base_node_impl");
const base_node_private_1 = require("./base_node_private");
const apply_condition_refresh_1 = require("./apply_condition_refresh");
const errorLog = (0, node_opcua_debug_1.make_errorLog)(__filename);
class UAObjectImpl extends base_node_impl_1.BaseNodeImpl {
    _eventNotifier;
    nodeClass = node_opcua_data_model_1.NodeClass.Object;
    get eventNotifier() {
        // ensure eventNotifier is set if the node has some event
        if (!this._eventNotifier) {
            // const s = this.getEventSources();
            const s = this.getEventSources();
            const n = this.getNotifiers();
            if (s.length > 0 || n.length > 0) {
                this._eventNotifier = this._eventNotifier | node_opcua_address_space_base_1.EventNotifierFlags.SubscribeToEvents;
            }
        }
        return this._eventNotifier;
    }
    symbolicName;
    get typeDefinitionObj() {
        // istanbul ignore next
        if (super.typeDefinitionObj.nodeClass !== node_opcua_data_model_1.NodeClass.ObjectType) {
            const msg = `Invalid type definition node class , expecting a ObjectType got ${node_opcua_data_model_1.NodeClass[super.typeDefinitionObj.nodeClass]}\n${this.browseName.toString()} ${this.nodeId.toString()}`;
            errorLog(msg);
            throw new Error(msg);
        }
        return super.typeDefinitionObj;
    }
    constructor(options) {
        super(options);
        this._eventNotifier = options.eventNotifier || node_opcua_address_space_base_1.EventNotifierFlags.None;
        (0, node_opcua_assert_1.assert)(typeof this.eventNotifier === "number" && (0, node_opcua_basic_types_1.isValidByte)(this.eventNotifier));
        this.symbolicName = options.symbolicName || null;
    }
    readAttribute(context, attributeId, indexRange, dataEncoding) {
        indexRange;
        dataEncoding;
        const now = (0, node_opcua_date_time_1.getCurrentClock)();
        const options = {};
        switch (attributeId) {
            case node_opcua_data_model_2.AttributeIds.EventNotifier:
                (0, node_opcua_assert_1.assert)((0, node_opcua_basic_types_1.isValidByte)(this.eventNotifier));
                options.value = { dataType: node_opcua_variant_1.DataType.Byte, value: this.eventNotifier };
                options.serverTimestamp = now.timestamp;
                options.serverPicoseconds = now.picoseconds;
                options.statusCode = node_opcua_status_code_1.StatusCodes.Good;
                break;
            default:
                return base_node_impl_1.BaseNodeImpl.prototype.readAttribute.call(this, context, attributeId, indexRange, dataEncoding);
        }
        return new node_opcua_data_value_1.DataValue(options);
    }
    clone(options, optionalFilter, extraInfo) {
        options = {
            eventNotifier: this.eventNotifier,
            symbolicName: this.symbolicName || undefined,
            ...options,
        };
        const cloneObject = (0, base_node_private_1._clone)(this, UAObjectImpl, options, optionalFilter || node_opcua_address_space_base_1.defaultCloneFilter, extraInfo || (0, node_opcua_address_space_base_1.makeDefaultCloneExtraInfo)(this));
        // xx  newObject.propagate_back_references();
        // xx newObject.install_extra_properties();
        return cloneObject;
    }
    /**
     * returns true if the object has some opcua methods
     */
    get hasMethods() {
        return this.getMethods().length > 0;
    }
    getMethodByName(methodName, namespaceIndex) {
        return super.getMethodByName(methodName, namespaceIndex);
    }
    getMethods() {
        return super.getMethods();
    }
    setEventNotifier(eventNotifierFlags) {
        this._eventNotifier = eventNotifierFlags;
    }
    /**
     * Raise a transient Event
     */
    raiseEvent(eventType, data) {
        const addressSpace = this.addressSpace;
        if (typeof eventType === "string") {
            const eventTypeFound = addressSpace.findEventType(eventType);
            if (!eventTypeFound) {
                throw new Error("raiseEvent: eventType cannot find event Type " + eventType.toString());
            }
            eventType = eventTypeFound;
            if (!eventType || eventType.nodeClass !== node_opcua_data_model_1.NodeClass.ObjectType) {
                throw new Error("eventType must exist and be an UAObjectType");
            }
        }
        else if (eventType instanceof node_opcua_nodeid_1.NodeId) {
            const eventTypeFound = addressSpace.findNode(eventType);
            if (!eventTypeFound) {
                throw new Error("raiseEvent: eventType cannot find event Type " + eventType.toString());
            }
            eventType = eventTypeFound;
            if (!eventType || eventType.nodeClass !== node_opcua_data_model_1.NodeClass.ObjectType) {
                throw new Error("eventType must exist and be an UAObjectType" + eventType.toString());
            }
        }
        eventType = eventType;
        let eventTypeNode = eventType;
        // istanbul ignore next
        if (!eventTypeNode) {
            throw new Error("UAObject#raiseEventType : Cannot find event type :" + eventType.toString());
        }
        // coerce EventType
        eventTypeNode = addressSpace.findEventType(eventType);
        const baseEventType = addressSpace.findEventType("BaseEventType");
        data.$eventDataSource = eventTypeNode;
        data.sourceNode = data.sourceNode || { dataType: node_opcua_variant_1.DataType.NodeId, value: this.nodeId };
        const eventData1 = addressSpace.constructEventData(eventTypeNode, data);
        this._bubble_up_event(eventData1);
    }
    _bubble_up_event(eventData) {
        const addressSpace = this.addressSpace;
        const queue = [];
        // walk up the hasNotify / hasEventSource chain
        const m = {};
        // all events are notified to the server object
        // emit on server object
        const server = addressSpace.findNode("Server");
        if (server) {
            (0, node_opcua_assert_1.assert)(server.eventNotifier > 0x00, "Server must be an event notifier");
            server.emit("event", eventData);
            m[server.nodeId.toString()] = server;
        }
        else {
            errorLog(chalk_1.default.yellow("Warning. ") +
                chalk_1.default.cyan("UAObject#raiseEvent") +
                chalk_1.default.red(" cannot find Server object on addressSpace"));
        }
        addInQueue(this);
        function addInQueue(obj) {
            const key = obj.nodeId.toString();
            if (!m[key]) {
                m[key] = obj;
                queue.push(obj);
            }
        }
        while (queue.length) {
            const obj = queue.pop();
            // emit on object
            obj.emit("event", eventData);
            const elements1 = obj.findReferencesAsObject("HasNotifier", false);
            elements1.forEach(addInQueue);
            const elements2 = obj.findReferencesAsObject("HasEventSource", false);
            elements2.forEach(addInQueue);
        }
    }
    _conditionRefresh(_cache) {
        apply_condition_refresh_1.apply_condition_refresh.call(this, _cache);
    }
    toString() {
        const options = new base_node_private_1.ToStringBuilder();
        base_node_private_1.UAObject_toString.call(this, options);
        return options.toString();
    }
}
exports.UAObjectImpl = UAObjectImpl;
//# sourceMappingURL=ua_object_impl.js.map