"use strict";
var __importDefault = (this && this.__importDefault) || function (mod) {
    return (mod && mod.__esModule) ? mod : { "default": mod };
};
Object.defineProperty(exports, "__esModule", { value: true });
exports.UADataTypeImpl = void 0;
exports.DataType_toString = DataType_toString;
/**
 * @module node-opcua-address-space
 */
const chalk_1 = __importDefault(require("chalk"));
const node_opcua_assert_1 = require("node-opcua-assert");
const node_opcua_data_model_1 = require("node-opcua-data-model");
const node_opcua_data_model_2 = require("node-opcua-data-model");
const node_opcua_data_value_1 = require("node-opcua-data-value");
const node_opcua_nodeid_1 = require("node-opcua-nodeid");
const node_opcua_numeric_range_1 = require("node-opcua-numeric-range");
const node_opcua_status_code_1 = require("node-opcua-status-code");
const node_opcua_types_1 = require("node-opcua-types");
const node_opcua_variant_1 = require("node-opcua-variant");
const node_opcua_constants_1 = require("node-opcua-constants");
const node_opcua_basic_types_1 = require("node-opcua-basic-types");
const session_context_1 = require("../source/session_context");
const base_node_impl_1 = require("./base_node_impl");
const base_node_private_1 = require("./base_node_private");
const tool_isSubtypeOf_1 = require("./tool_isSubtypeOf");
const tool_isSubtypeOf_2 = require("./tool_isSubtypeOf");
const tool_isSubtypeOf_3 = require("./tool_isSubtypeOf");
const base_node_private_2 = require("./base_node_private");
class UADataTypeImpl extends base_node_impl_1.BaseNodeImpl {
    nodeClass = node_opcua_data_model_1.NodeClass.DataType;
    definitionName = "";
    symbolicName;
    /**
     * returns true if this is a super type of baseType
     *
     * @example
     *
     *    var dataTypeDouble = addressSpace.findDataType("Double");
     *    var dataTypeNumber = addressSpace.findDataType("Number");
     *    assert(dataTypeDouble.isSubtypeOf(dataTypeNumber));
     *    assert(!dataTypeNumber.isSubtypeOf(dataTypeDouble));
     *
     */
    get subtypeOf() {
        return tool_isSubtypeOf_2.get_subtypeOf.call(this);
    }
    get subtypeOfObj() {
        return tool_isSubtypeOf_3.get_subtypeOfObj.call(this);
    }
    /** @deprecated */
    isSupertypeOf = (0, tool_isSubtypeOf_1.construct_isSubtypeOf)(UADataTypeImpl);
    isSubtypeOf = (0, tool_isSubtypeOf_1.construct_isSubtypeOf)(UADataTypeImpl);
    isAbstract;
    $isUnion;
    enumStrings;
    enumValues;
    $partialDefinition;
    $fullDefinition;
    constructor(options) {
        super(options);
        if (options.partialDefinition) {
            this.$partialDefinition = options.partialDefinition;
            this.$isUnion = options.isUnion;
        }
        this.isAbstract = options.isAbstract === undefined || options.isAbstract === null ? false : options.isAbstract;
        this.symbolicName = options.symbolicName || this.browseName.name;
    }
    get basicDataType() {
        return this.getBasicDataType();
    }
    getBasicDataType() {
        return this.addressSpace.findCorrespondingBasicDataType(this);
    }
    readAttribute(context, attributeId, indexRange, dataEncoding) {
        (0, node_opcua_assert_1.assert)(!context || context instanceof session_context_1.SessionContext);
        const options = {};
        switch (attributeId) {
            case node_opcua_data_model_2.AttributeIds.IsAbstract:
                options.statusCode = node_opcua_status_code_1.StatusCodes.Good;
                options.value = { dataType: node_opcua_variant_1.DataType.Boolean, value: !!this.isAbstract };
                break;
            case node_opcua_data_model_2.AttributeIds.DataTypeDefinition:
                {
                    const _definition = this._getDefinition()?.clone();
                    if (_definition) {
                        options.value = { dataType: node_opcua_variant_1.DataType.ExtensionObject, value: _definition };
                    }
                    else {
                        options.statusCode = node_opcua_status_code_1.StatusCodes.BadAttributeIdInvalid;
                    }
                }
                break;
            default:
                return super.readAttribute(context, attributeId, indexRange, dataEncoding);
        }
        return new node_opcua_data_value_1.DataValue(options);
    }
    getEncodingDefinition(encoding_name) {
        const encodingNode = this.getEncodingNode(encoding_name);
        if (!encodingNode) {
            throw new Error("Cannot find Encoding for " + encoding_name);
        }
        const indexRange = new node_opcua_numeric_range_1.NumericRange();
        const descriptionNodeRef = encodingNode.findReferences("HasDescription")[0];
        const descriptionNode = this.addressSpace.findNode(descriptionNodeRef.nodeId);
        if (!descriptionNode) {
            return null;
        }
        const dataValue = descriptionNode.readValue(session_context_1.SessionContext.defaultContext, indexRange);
        return dataValue.value.value.toString() || null;
    }
    getEncodingNode(encoding_name) {
        const _cache = (0, base_node_private_2.BaseNode_getCache)(this);
        _cache._encoding = _cache._encoding || new Map();
        const key = encoding_name + "Node";
        if (!_cache._encoding.has(key)) {
            (0, node_opcua_assert_1.assert)(encoding_name === "Default Binary" || encoding_name === "Default XML" || encoding_name === "Default JSON");
            // could be binary or xml
            const refs = this.findReferences("HasEncoding", true);
            const addressSpace = this.addressSpace;
            const encoding = refs
                .map((ref) => addressSpace.findNode(ref.nodeId))
                .filter((obj) => obj !== null)
                .filter((obj) => obj.browseName.toString() === encoding_name);
            const node = encoding.length === 0 ? null : encoding[0];
            _cache._encoding.set(key, node);
            return node;
        }
        return _cache._encoding.get(key) || null;
    }
    getEncodingNodeId(encoding_name) {
        const encoding = this.getEncodingNode(encoding_name);
        if (!encoding) {
            return null;
        }
        const namespaceUri = this.addressSpace.getNamespaceUri(encoding.nodeId.namespace);
        return node_opcua_nodeid_1.ExpandedNodeId.fromNodeId(encoding.nodeId, namespaceUri);
    }
    /**
     * returns the encoding of this node's
     * TODO objects have 2 encodings : XML and Binaries
     */
    get binaryEncoding() {
        return this.getEncodingNode("Default Binary");
    }
    get binaryEncodingDefinition() {
        return this.getEncodingDefinition("Default Binary");
    }
    get binaryEncodingNodeId() {
        return this.getEncodingNodeId("Default Binary");
    }
    get xmlEncoding() {
        return this.getEncodingNode("Default XML");
    }
    get xmlEncodingNodeId() {
        return this.getEncodingNodeId("Default XML");
    }
    get xmlEncodingDefinition() {
        return this.getEncodingDefinition("Default XML");
    }
    get jsonEncoding() {
        return this.getEncodingNode("Default JSON");
    }
    get jsonEncodingNodeId() {
        return this.getEncodingNodeId("Default JSON");
    }
    //  public get jsonEncodingDefinition(): string | null {
    //      return this.getEncodingDefinition("Default JSON");
    //  }
    _getEnumerationInfo() {
        let definition = [];
        if (this.enumStrings) {
            const enumStrings = this.enumStrings.readValue().value.value;
            (0, node_opcua_assert_1.assert)(Array.isArray(enumStrings));
            definition = enumStrings.map((e, index) => {
                return {
                    name: e.text,
                    value: index
                };
            });
        }
        else if (this.enumValues) {
            (0, node_opcua_assert_1.assert)(this.enumValues, "must have a enumValues property");
            const enumValues = this.enumValues.readValue().value.value;
            (0, node_opcua_assert_1.assert)(Array.isArray(enumValues));
            definition = enumValues.map((e) => {
                return {
                    name: e.displayName.text,
                    value: (0, node_opcua_basic_types_1.coerceInt64toInt32)(e.value)
                };
            });
        }
        // construct nameIndex and valueIndex
        const indexes = {
            nameIndex: {},
            valueIndex: {}
        };
        for (const e of definition) {
            indexes.nameIndex[e.name] = e;
            indexes.valueIndex[e.value] = e;
        }
        return indexes;
    }
    isStructure() {
        const definition = this._getDefinition();
        return !!definition && definition instanceof node_opcua_types_1.StructureDefinition;
    }
    getStructureDefinition() {
        const definition = this._getDefinition();
        (0, node_opcua_assert_1.assert)(definition instanceof node_opcua_types_1.StructureDefinition);
        return definition;
    }
    isEnumeration() {
        const definition = this._getDefinition();
        return !!definition && definition instanceof node_opcua_types_1.EnumDefinition;
    }
    getEnumDefinition() {
        const definition = this._getDefinition();
        (0, node_opcua_assert_1.assert)(definition instanceof node_opcua_types_1.EnumDefinition);
        return definition;
    }
    // eslint-disable-next-line complexity
    _getDefinition() {
        if (this.$fullDefinition !== undefined) {
            return this.$fullDefinition;
        }
        const addressSpace = this.addressSpace;
        const enumeration = addressSpace.findDataType("Enumeration");
        const structure = addressSpace.findDataType("Structure");
        const union = addressSpace.findDataType("Union");
        // we have a data type from a companion specification
        // let's see if this data type need to be registered
        const isEnumeration = enumeration && this.isSubtypeOf(enumeration);
        const isStructure = structure && this.isSubtypeOf(structure);
        const isUnion = !!(structure && union && this.isSubtypeOf(union));
        const isRootDataType = (n) => n.nodeId.namespace === 0 && n.nodeId.value === node_opcua_constants_1.DataTypeIds.BaseDataType;
        // https://reference.opcfoundation.org/v104/Core/docs/Part3/8.49/#Table34
        if (isStructure) {
            // eslint-disable-next-line @typescript-eslint/no-this-alias
            let dataTypeNode = this;
            const allPartialDefinitions = [];
            while (dataTypeNode && !isRootDataType(dataTypeNode)) {
                if (dataTypeNode.$partialDefinition) {
                    allPartialDefinitions.push(dataTypeNode.$partialDefinition);
                }
                dataTypeNode = dataTypeNode.subtypeOfObj;
            }
            // merge them:
            const definitionFields = [];
            for (const dd of allPartialDefinitions.reverse()) {
                definitionFields.push(...dd);
            }
            const basicDataType = this.subtypeOfObj?.nodeId || new node_opcua_nodeid_1.NodeId();
            const defaultEncodingId = this.binaryEncodingNodeId || this.xmlEncodingNodeId || new node_opcua_nodeid_1.NodeId();
            const definitionName = this.browseName.name;
            this.$fullDefinition = makeStructureDefinition(definitionName, basicDataType, defaultEncodingId, definitionFields, isUnion);
        }
        else if (isEnumeration) {
            const allPartialDefinitions = [];
            // eslint-disable-next-line @typescript-eslint/no-this-alias
            let dataTypeNode = this;
            while (dataTypeNode && !isRootDataType(dataTypeNode)) {
                if (dataTypeNode.$partialDefinition) {
                    allPartialDefinitions.push(dataTypeNode.$partialDefinition);
                }
                dataTypeNode = dataTypeNode.subtypeOfObj;
            }
            // merge them:
            const definitionFields = [];
            for (const dd of allPartialDefinitions.reverse()) {
                definitionFields.push(...dd);
            }
            this.$fullDefinition = makeEnumDefinition(definitionFields);
        }
        return this.$fullDefinition;
    }
    getDefinition() {
        const d = this._getDefinition();
        if (!d) {
            throw new Error("DataType has no definition property");
        }
        return d;
    }
    install_extra_properties() {
        //
    }
    toString() {
        const options = new base_node_private_1.ToStringBuilder();
        DataType_toString.call(this, options);
        return options.toString();
    }
}
exports.UADataTypeImpl = UADataTypeImpl;
function dataTypeDefinition_toString(options) {
    const definition = this._getDefinition();
    if (!definition) {
        return;
    }
    const output = definition.toString();
    options.add(options.padding + chalk_1.default.yellow(" Definition                   :             "));
    for (const str of output.split("\n")) {
        options.add(options.padding + chalk_1.default.yellow("                              :   " + str));
    }
}
function DataType_toString(options) {
    base_node_private_1.BaseNode_toString.call(this, options);
    options.add(options.padding + chalk_1.default.yellow("          isAbstract          : " + this.isAbstract));
    options.add(options.padding + chalk_1.default.yellow("          definitionName      : " + this.definitionName));
    options.add(options.padding +
        chalk_1.default.yellow("          binaryEncodingNodeId: ") +
        (this.binaryEncodingNodeId ? this.binaryEncodingNodeId.toString() : "<none>"));
    options.add(options.padding +
        chalk_1.default.yellow("          xmlEncodingNodeId   : ") +
        (this.xmlEncodingNodeId ? this.xmlEncodingNodeId.toString() : "<none>"));
    options.add(options.padding +
        chalk_1.default.yellow("          jsonEncodingNodeId  : ") +
        (this.jsonEncodingNodeId ? this.jsonEncodingNodeId.toString() : "<none>"));
    if (this.subtypeOfObj) {
        options.add(options.padding +
            chalk_1.default.yellow("          subtypeOfObj        : ") +
            (this.subtypeOfObj ? this.subtypeOfObj.browseName.toString() : ""));
    }
    // references
    base_node_private_1.BaseNode_References_toString.call(this, options);
    dataTypeDefinition_toString.call(this, options);
}
const defaultEnumValue = (0, node_opcua_basic_types_1.coerceInt64)(-1);
function makeEnumDefinition(definitionFields) {
    return new node_opcua_types_1.EnumDefinition({
        fields: definitionFields.map((x) => ({
            description: x.description,
            name: x.name,
            value: x.value === undefined ? defaultEnumValue : (0, node_opcua_basic_types_1.coerceInt64)(x.value)
        }))
    });
}
function makeStructureDefinition(name, baseDataType, defaultEncodingId, fields, isUnion) {
    // Structure = 0,
    // StructureWithOptionalFields = 1,
    // Union = 2,
    const hasOptionalFields = fields.filter((field) => field.isOptional).length > 0;
    const structureType = isUnion
        ? node_opcua_types_1.StructureType.Union
        : hasOptionalFields
            ? node_opcua_types_1.StructureType.StructureWithOptionalFields
            : node_opcua_types_1.StructureType.Structure;
    const sd = new node_opcua_types_1.StructureDefinition({
        baseDataType,
        defaultEncodingId,
        fields,
        structureType
    });
    return sd;
}
//# sourceMappingURL=ua_data_type_impl.js.map