"use strict";
var __importDefault = (this && this.__importDefault) || function (mod) {
    return (mod && mod.__esModule) ? mod : { "default": mod };
};
Object.defineProperty(exports, "__esModule", { value: true });
exports.ReferenceImpl = void 0;
exports.isNodeIdString = isNodeIdString;
exports.resolveReferenceNode = resolveReferenceNode;
exports.resolveReferenceType = resolveReferenceType;
/**
 * @module node-opcua-address-space
 */
// tslint:disable:no-console
const chalk_1 = __importDefault(require("chalk"));
const node_opcua_assert_1 = require("node-opcua-assert");
const node_opcua_nodeid_1 = require("node-opcua-nodeid");
const node_opcua_utils_1 = require("node-opcua-utils");
function isNodeIdString(str) {
    (0, node_opcua_assert_1.assert)(typeof str === "string");
    return str.substring(0, 2) === "i=" || str.substring(0, 3) === "ns=";
}
function is_valid_reference(ref) {
    const hasRequestedProperties = Object.prototype.hasOwnProperty.call(ref, "referenceType") &&
        Object.prototype.hasOwnProperty.call(ref, "nodeId") &&
        !(0, node_opcua_utils_1.isNullOrUndefined)(ref.isForward);
    if (!hasRequestedProperties) {
        return false;
    }
    (0, node_opcua_assert_1.assert)(ref.referenceType instanceof node_opcua_nodeid_1.NodeId);
    (0, node_opcua_assert_1.assert)(!ref.node || (0, node_opcua_nodeid_1.sameNodeId)(ref.node.nodeId, ref.nodeId));
    // xx assert(!ref.referenceTypeName || typeof ref.referenceTypeName === "string");
    // xx // referenceType shall no be a nodeId string (this could happen by mistake)
    // xx assert(!isNodeIdString(ref.referenceType));
    return true;
}
/**
 * @private
 *
 * @example
 *       ---- some text ----->
 */
function _arrow(text, length, isForward) {
    length = Math.max(length, text.length + 8);
    const nb = Math.floor((length - text.length - 2) / 2);
    const h = Array(nb).join("-");
    const extra = text.length % 2 === 1 ? "-" : "";
    if (isForward) {
        return extra + h + " " + text + " " + h + "> ";
    }
    return "<" + h + " " + text + " " + h + extra + " ";
}
function _w(str, width) {
    return str.padEnd(width).substring(0, width);
}
function _localCoerceToNodeID(nodeIdLike) {
    if (Object.prototype.hasOwnProperty.call(nodeIdLike, "nodeId")) {
        return nodeIdLike.nodeId;
    }
    return (0, node_opcua_nodeid_1.coerceNodeId)(nodeIdLike);
}
function resolveReferenceNode(addressSpace, reference) {
    const _reference = reference;
    if (!_reference.node) {
        _reference.node = addressSpace.findNode(reference.nodeId);
    }
    return _reference.node;
}
function resolveReferenceType(addressSpace, reference) {
    const _reference = reference;
    if (!_reference._referenceType) {
        if (!_reference.referenceType) {
            errorLog(chalk_1.default.red("ERROR MISSING reference"), reference);
        }
        _reference._referenceType = addressSpace.findReferenceType(reference.referenceType);
    }
    return _reference._referenceType;
}
/**
 * @class Reference
 * @param options.referenceType {NodeId}
 * @param options.nodeId        {NodeId}
 * @param options.isForward     {Boolean}
 * @constructor
 */
class ReferenceImpl {
    static resolveReferenceNode(addressSpace, reference) {
        return resolveReferenceNode(addressSpace, reference);
    }
    static resolveReferenceType(addressSpace, reference) {
        return resolveReferenceType(addressSpace, reference);
    }
    nodeId;
    referenceType;
    _referenceType;
    isForward;
    node;
    // cache
    __hash;
    constructor(options) {
        (0, node_opcua_assert_1.assert)(options.referenceType instanceof node_opcua_nodeid_1.NodeId);
        (0, node_opcua_assert_1.assert)(options.nodeId instanceof node_opcua_nodeid_1.NodeId);
        this.referenceType = (0, node_opcua_nodeid_1.coerceNodeId)(options.referenceType);
        this.isForward = options.isForward === undefined ? true : !!options.isForward;
        this.nodeId = _localCoerceToNodeID(options.nodeId);
        // optional to speed up when AddReferenceOpts is in fact a Reference !
        this._referenceType = options._referenceType;
        this.node = options.node;
        (0, node_opcua_assert_1.assert)(is_valid_reference(this));
    }
    /**
     * turn reference into a arrow :   ---- ReferenceType --> [NodeId]

     * @return {String}
     */
    toString(options) {
        let infoNode = _w(this.nodeId.toString(), 24);
        let refType = this.referenceType.toString();
        if (options && options.addressSpace) {
            const node = options.addressSpace.findNode(this.nodeId);
            infoNode = "[" + infoNode + "]" + _w(node?.browseName.toString(), 40);
            const ref = options.addressSpace.findReferenceType(this.referenceType);
            const refNode = options.addressSpace.findNode(ref.nodeId);
            refType = refNode.browseName.toString() + " (" + ref.nodeId.toString() + ")";
        }
        return _arrow(refType, 40, this.isForward) + infoNode;
    }
    /**
     * @internal
     */
    get hash() {
        if (!this.__hash) {
            this.__hash = (this.isForward ? "" : "!") + this.referenceType.toString() + "-" + this.nodeId.toString();
        }
        return this.__hash;
    }
    /**
     * @internal
     */
    dispose() {
        this.__hash = undefined;
        this.node = undefined;
        /*
        this._referenceType = null;
        this.nodeId = null as NodeId;
        this.referenceType = null as NodeId;
        */
    }
}
exports.ReferenceImpl = ReferenceImpl;
function errorLog(arg0, reference) {
    throw new Error("Function not implemented.");
}
//# sourceMappingURL=reference_impl.js.map