"use strict";
var __importDefault = (this && this.__importDefault) || function (mod) {
    return (mod && mod.__esModule) ? mod : { "default": mod };
};
Object.defineProperty(exports, "__esModule", { value: true });
exports.NamespaceImpl = void 0;
exports._handle_hierarchy_parent = _handle_hierarchy_parent;
exports.isNonEmptyQualifiedName = isNonEmptyQualifiedName;
/**
 * @module node-opcua-address-space
 */
// tslint:disable:no-console
const chalk_1 = __importDefault(require("chalk"));
const node_opcua_assert_1 = require("node-opcua-assert");
const node_opcua_basic_types_1 = require("node-opcua-basic-types");
const node_opcua_data_access_1 = require("node-opcua-data-access");
const node_opcua_data_model_1 = require("node-opcua-data-model");
const node_opcua_data_model_2 = require("node-opcua-data-model");
const node_opcua_data_model_3 = require("node-opcua-data-model");
const node_opcua_data_model_4 = require("node-opcua-data-model");
const node_opcua_debug_1 = require("node-opcua-debug");
const node_opcua_nodeid_1 = require("node-opcua-nodeid");
const node_opcua_nodeid_2 = require("node-opcua-nodeid");
const node_opcua_status_code_1 = require("node-opcua-status-code");
const node_opcua_types_1 = require("node-opcua-types");
const node_opcua_utils_1 = require("node-opcua-utils");
const node_opcua_variant_1 = require("node-opcua-variant");
const address_space_change_event_tools_1 = require("./address_space_change_event_tools");
const ua_condition_impl_1 = require("./alarms_and_conditions/ua_condition_impl");
const ua_discrete_alarm_impl_1 = require("./alarms_and_conditions/ua_discrete_alarm_impl");
const ua_exclusive_deviation_alarm_impl_1 = require("./alarms_and_conditions/ua_exclusive_deviation_alarm_impl");
const ua_exclusive_limit_alarm_impl_1 = require("./alarms_and_conditions/ua_exclusive_limit_alarm_impl");
const ua_limit_alarm_impl_1 = require("./alarms_and_conditions/ua_limit_alarm_impl");
const ua_non_exclusive_deviation_alarm_impl_1 = require("./alarms_and_conditions/ua_non_exclusive_deviation_alarm_impl");
const ua_non_exclusive_limit_alarm_impl_1 = require("./alarms_and_conditions/ua_non_exclusive_limit_alarm_impl");
const alarms_and_conditions_1 = require("./alarms_and_conditions");
const ua_off_normal_alarm_impl_1 = require("./alarms_and_conditions/ua_off_normal_alarm_impl");
const add_dataItem_stuff_1 = require("./data_access/add_dataItem_stuff");
const ua_multistate_discrete_impl_1 = require("./data_access/ua_multistate_discrete_impl");
// state machine
const ua_two_state_variable_1 = require("./state_machine/ua_two_state_variable");
//
const namespace_private_1 = require("./namespace_private");
const base_node_impl_1 = require("./base_node_impl");
const ua_variable_impl_1 = require("./ua_variable_impl");
const nodeid_manager_1 = require("./nodeid_manager");
const ua_two_state_discrete_impl_1 = require("./data_access/ua_two_state_discrete_impl");
const role_permissions_1 = require("./role_permissions");
const ua_object_impl_1 = require("./ua_object_impl");
const ua_data_type_impl_1 = require("./ua_data_type_impl");
const ua_object_type_impl_1 = require("./ua_object_type_impl");
const ua_method_impl_1 = require("./ua_method_impl");
const ua_variable_type_impl_1 = require("./ua_variable_type_impl");
const ua_reference_type_impl_1 = require("./ua_reference_type_impl");
const ua_view_impl_1 = require("./ua_view_impl");
const ua_multistate_value_discrete_impl_1 = require("./data_access/ua_multistate_value_discrete_impl");
function _makeHashKey(nodeId) {
    switch (nodeId.identifierType) {
        case node_opcua_nodeid_1.NodeIdType.STRING:
        case node_opcua_nodeid_1.NodeIdType.GUID:
            return nodeId.value;
        case node_opcua_nodeid_1.NodeIdType.NUMERIC:
            return nodeId.value;
        default:
            // istanbul ignore next
            if (nodeId.identifierType !== node_opcua_nodeid_1.NodeIdType.BYTESTRING) {
                throw new Error("invalid nodeIdType");
            }
            return nodeId.value ? nodeId.value.toString() : "OPAQUE:0";
    }
}
const doDebug = false;
const errorLog = (0, node_opcua_debug_1.make_errorLog)("AddressSpace");
const warningLog = (0, node_opcua_debug_1.make_warningLog)("AddressSpace");
const regExp1 = /^(s|i|b|g)=/;
const regExpNamespaceDotBrowseName = /^[0-9]+:(.*)/;
function detachNode(node) {
    const addressSpace = node.addressSpace;
    const nonHierarchicalReferences = node.findReferencesEx("NonHierarchicalReferences", node_opcua_data_model_3.BrowseDirection.Inverse);
    for (const ref of nonHierarchicalReferences) {
        (0, node_opcua_assert_1.assert)(!ref.isForward);
        ref.node.removeReference({
            isForward: !ref.isForward,
            nodeId: node.nodeId,
            referenceType: ref.referenceType
        });
    }
    const nonHierarchicalReferencesF = node.findReferencesEx("NonHierarchicalReferences", node_opcua_data_model_3.BrowseDirection.Forward);
    for (const ref of nonHierarchicalReferencesF) {
        if (!ref.node) {
            // could be a special case of a frequently use target node such as ModellingRule_Mandatory that do not back trace
            // their reference
            continue;
        }
        (0, node_opcua_assert_1.assert)(ref.isForward);
        ref.node.removeReference({
            isForward: !ref.isForward,
            nodeId: node.nodeId,
            referenceType: ref.referenceType
        });
    }
    // remove reversed Hierarchical references
    const hierarchicalReferences = node.findReferencesEx("HierarchicalReferences", node_opcua_data_model_3.BrowseDirection.Inverse);
    for (const ref of hierarchicalReferences) {
        (0, node_opcua_assert_1.assert)(!ref.isForward);
        const parent = addressSpace.findNode(ref.nodeId);
        parent.removeReference({
            isForward: !ref.isForward,
            nodeId: node.nodeId,
            referenceType: ref.referenceType
        });
    }
    node.unpropagate_back_references();
}
/**
 *
 * @constructor
 * @params options {Object}
 * @params options.namespaceUri {string}
 * @params options.addressSpace {IAddressSpace}
 * @params options.index {number}
 * @params options.version="" {string}
 * @params options.publicationDate="" {Date}
 *
 */
class NamespaceImpl {
    static _handle_hierarchy_parent = _handle_hierarchy_parent;
    static isNonEmptyQualifiedName = isNonEmptyQualifiedName;
    namespaceUri;
    addressSpace;
    index;
    emulateVersion103 = false;
    version = "0.0.0";
    publicationDate = new Date(Date.UTC(1900, 0, 1));
    registerSymbolicNames = false;
    _requiredModels;
    _objectTypeMap;
    _variableTypeMap;
    _referenceTypeMap;
    _dataTypeMap;
    _referenceTypeMapInv;
    _nodeIdManager;
    _nodeid_index;
    _aliases;
    defaultAccessRestrictions;
    defaultRolePermissions;
    constructor(options) {
        // istanbul ignore next
        if (!(typeof options.namespaceUri === "string")) {
            throw new Error("NamespaceImpl constructor: namespaceUri must exists and be a string : got " + options.namespaceUri);
        }
        // istanbul ignore next
        if (typeof options.index !== "number") {
            throw new Error("NamespaceImpl constructor: index must be a number");
        }
        // istanbul ignore next
        if (!options.addressSpace) {
            throw new Error("NamespaceImpl constructor: Must specify a valid address space");
        }
        this.namespaceUri = options.namespaceUri;
        this.addressSpace = options.addressSpace;
        this.index = options.index;
        this._nodeid_index = new Map();
        this._aliases = new Map();
        this._objectTypeMap = new Map();
        this._variableTypeMap = new Map();
        this._referenceTypeMap = new Map();
        this._referenceTypeMapInv = new Map();
        this._dataTypeMap = new Map();
        this._nodeIdManager = new nodeid_manager_1.NodeIdManager(this.index, this.addressSpace);
    }
    getDefaultNamespace() {
        return this.index === 0 ? this : this.addressSpace.getDefaultNamespace();
    }
    dispose() {
        for (const node of this.nodeIterator()) {
            node.dispose();
        }
        this._nodeid_index = new Map();
        this._aliases = new Map();
        this.addressSpace = {};
        this._objectTypeMap = new Map();
        this._variableTypeMap = new Map();
        this._referenceTypeMap = new Map();
        this._referenceTypeMapInv = new Map();
        this._dataTypeMap = new Map();
    }
    nodeIterator() {
        return this._nodeid_index.values();
    }
    _objectTypeIterator() {
        return this._objectTypeMap.values();
    }
    _objectTypeCount() {
        return this._objectTypeMap.size;
    }
    _variableTypeIterator() {
        return this._variableTypeMap.values();
    }
    _variableTypeCount() {
        return this._variableTypeMap.size;
    }
    _dataTypeIterator() {
        return this._dataTypeMap.values();
    }
    _dataTypeCount() {
        return this._dataTypeMap.size;
    }
    _referenceTypeIterator() {
        return this._referenceTypeMap.values();
    }
    _referenceTypeCount() {
        return this._referenceTypeMap.size;
    }
    _aliasCount() {
        return this._aliases.size;
    }
    findNode2(nodeId) {
        // this one is faster assuming you have a nodeId
        (0, node_opcua_assert_1.assert)(nodeId.namespace === this.index);
        return this._nodeid_index.get(_makeHashKey(nodeId)) || null;
    }
    findNode(nodeId) {
        if (typeof nodeId === "string") {
            if (nodeId.match(regExp1)) {
                nodeId = "ns=" + this.index + ";" + nodeId;
            }
        }
        nodeId = (0, node_opcua_nodeid_1.resolveNodeId)(nodeId);
        return this.findNode2(nodeId);
    }
    /**
     *
     * @param objectTypeName {String}
     * @return {UAObjectType|null}
     */
    findObjectType(objectTypeName) {
        return this._objectTypeMap.get(objectTypeName) || null;
    }
    /**
     *
     * @param variableTypeName {String}
     * @returns {UAVariableType|null}
     */
    findVariableType(variableTypeName) {
        return this._variableTypeMap.get(variableTypeName) || null;
    }
    /**
     *
     * @param dataTypeName {String}
     * @returns {UADataType|null}
     */
    findDataType(dataTypeName) {
        return this._dataTypeMap.get(dataTypeName) || null;
    }
    /**
     *
     * @param referenceTypeName {String}
     * @returns  {ReferenceType|null}
     */
    findReferenceType(referenceTypeName) {
        return this._referenceTypeMap.get(referenceTypeName) || null;
    }
    /**
     * find a ReferenceType by its inverse name.

     * @param inverseName {String} the inverse name of the ReferenceType to find
     * @return {ReferenceType}
     */
    findReferenceTypeFromInverseName(inverseName) {
        (0, node_opcua_assert_1.assert)(typeof inverseName === "string");
        const node = this._referenceTypeMapInv.get(inverseName);
        (0, node_opcua_assert_1.assert)(!node || (node.nodeClass === node_opcua_data_model_4.NodeClass.ReferenceType && node.inverseName.text === inverseName));
        return node ? node : null;
    }
    /**
     *

     * @param alias_name {String} the alias name
     * @param nodeId {NodeId} NodeId must belong to this namespace
     */
    addAlias(alias_name, nodeId) {
        (0, node_opcua_assert_1.assert)(typeof alias_name === "string");
        (0, node_opcua_assert_1.assert)(nodeId instanceof node_opcua_nodeid_2.NodeId);
        (0, node_opcua_assert_1.assert)(nodeId.namespace === this.index);
        this._aliases.set(alias_name, nodeId);
    }
    resolveAlias(name) {
        return this._aliases.get(name) || null;
    }
    /**
     * add a new Object type to the address space

     * @param options
     * @param options.browseName {String} the object type name
     * @param [options.displayName] {String|LocalizedText} the display name
     * @param [options.subtypeOf="BaseObjectType"] {String|NodeId|BaseNode} the base class
     * @param [options.nodeId] {String|NodeId} an optional nodeId for this objectType,
     *                                         if not provided a new nodeId will be created
     * @param [options.isAbstract = false] {Boolean}
     * @param [options.eventNotifier = 0] {Integer}
     * @param [options.postInstantiateFunc = null] {Function}
     *
     */
    addObjectType(options) {
        (0, node_opcua_assert_1.assert)(!Object.prototype.hasOwnProperty.call(options, "dataType"), "an objectType should not have a dataType");
        (0, node_opcua_assert_1.assert)(!Object.prototype.hasOwnProperty.call(options, "valueRank"), "an objectType should not have a valueRank");
        (0, node_opcua_assert_1.assert)(!Object.prototype.hasOwnProperty.call(options, "arrayDimensions"), "an objectType should not have a arrayDimensions");
        return this._addObjectOrVariableType(options, "BaseObjectType", node_opcua_data_model_4.NodeClass.ObjectType);
    }
    /**
     * add a new Variable type to the address space

     * @param options
     * @param options.browseName {String} the object type name
     * @param [options.displayName] {String|LocalizedText} the display name
     * @param [options.subtypeOf="BaseVariableType"] {String|NodeId|BaseNode} the base class
     * @param [options.nodeId] {String|NodeId} an optional nodeId for this objectType,
     *                                             if not provided a new nodeId will be created
     * @param [options.isAbstract = false] {Boolean}
     * @param options.dataType {String|NodeId} the variable DataType
     * @param [options.valueRank = -1]
     * @param [options.arrayDimensions = null] { Array<Int>>
     *
     */
    addVariableType(options) {
        (0, node_opcua_assert_1.assert)(!Object.prototype.hasOwnProperty.call(options, "arrayDimension"), "Do you mean ArrayDimensions ?");
        // dataType
        options.dataType = options.dataType || "Int32";
        options.dataType = this.addressSpace._coerce_DataType(options.dataType);
        // valueRank/ arrayDimensions
        (0, node_opcua_variant_1.verifyRankAndDimensions)(options);
        // arrayDimensions
        const variableType = this._addObjectOrVariableType(options, "BaseVariableType", node_opcua_data_model_4.NodeClass.VariableType);
        variableType.dataType = options.dataType;
        variableType.valueRank = options.valueRank || 0;
        variableType.arrayDimensions = options.arrayDimensions;
        return variableType;
    }
    /**
     * add a variable as a component of the parent node
     */
    addVariable(options) {
        (0, node_opcua_assert_1.assert)(arguments.length === 1, "Invalid arguments IAddressSpace#addVariable now takes only one argument.");
        if (Object.prototype.hasOwnProperty.call(options, "propertyOf") && options.propertyOf) {
            (0, node_opcua_assert_1.assert)(!options.typeDefinition || options.typeDefinition === "PropertyType");
            options.typeDefinition = options.typeDefinition || "PropertyType";
        }
        else {
            (0, node_opcua_assert_1.assert)(!options.typeDefinition || options.typeDefinition !== "PropertyType");
        }
        return this._addVariable(options);
    }
    addView(options) {
        (0, node_opcua_assert_1.assert)(arguments.length === 1, "Namespace#addView expecting a single argument");
        (0, node_opcua_assert_1.assert)(options);
        (0, node_opcua_assert_1.assert)(Object.prototype.hasOwnProperty.call(options, "browseName"));
        (0, node_opcua_assert_1.assert)(Object.prototype.hasOwnProperty.call(options, "organizedBy"));
        const browseName = options.browseName;
        (0, node_opcua_assert_1.assert)(typeof browseName === "string");
        const addressSpace = this.addressSpace;
        const baseDataVariableTypeId = addressSpace.findVariableType("BaseDataVariableType").nodeId;
        // ------------------------------------------ TypeDefinition
        const typeDefinition = options.typeDefinition || baseDataVariableTypeId;
        options.references = options.references || [];
        options.references.push({
            isForward: true,
            nodeId: typeDefinition,
            referenceType: "HasTypeDefinition"
        });
        const createOptions = options;
        (0, node_opcua_assert_1.assert)(!createOptions.nodeClass);
        createOptions.nodeClass = node_opcua_data_model_4.NodeClass.View;
        const view = this.createNode(createOptions);
        (0, node_opcua_assert_1.assert)(view.nodeId instanceof node_opcua_nodeid_2.NodeId);
        (0, node_opcua_assert_1.assert)(view.nodeClass === node_opcua_data_model_4.NodeClass.View);
        return view;
    }
    addObject(options1) {
        const options = options1;
        (0, node_opcua_assert_1.assert)(!options.nodeClass || options.nodeClass === node_opcua_data_model_4.NodeClass.Object);
        options.nodeClass = node_opcua_data_model_4.NodeClass.Object;
        const typeDefinition = options.typeDefinition || "BaseObjectType";
        options.references = options.references || [];
        options.references.push({ referenceType: "HasTypeDefinition", isForward: true, nodeId: typeDefinition });
        options.eventNotifier = +(options.eventNotifier || 0);
        const obj = this.createNode(options);
        (0, node_opcua_assert_1.assert)(obj instanceof ua_object_impl_1.UAObjectImpl);
        (0, node_opcua_assert_1.assert)(obj.nodeClass === node_opcua_data_model_4.NodeClass.Object);
        return obj;
    }
    /**
     *

     * @param parentFolder
     * @param options {String|Object}
     * @param options.browseName {String} the name of the folder
     * @param [options.nodeId] {NodeId}. An optional nodeId for this object
     *
     * @return {BaseNode}
     */
    addFolder(parentFolder, options) {
        if (typeof options === "string") {
            options = { browseName: options };
        }
        const addressSpace = this.addressSpace;
        (0, node_opcua_assert_1.assert)(!options.typeDefinition, "addFolder does not expect typeDefinition to be defined ");
        const typeDefinition = addressSpace._coerceTypeDefinition("FolderType");
        parentFolder = addressSpace._coerceFolder(parentFolder);
        options.nodeClass = node_opcua_data_model_4.NodeClass.Object;
        options.references = [
            { referenceType: "HasTypeDefinition", isForward: true, nodeId: typeDefinition },
            { referenceType: "Organizes", isForward: false, nodeId: parentFolder.nodeId }
        ];
        const node = this.createNode(options);
        return node;
    }
    /**

     * @param options
     * @param options.isAbstract
     * @param options.browseName
     * @param options.inverseName
     */
    addReferenceType(options) {
        const addressSpace = this.addressSpace;
        const options1 = options;
        options1.nodeClass = node_opcua_data_model_4.NodeClass.ReferenceType;
        options1.references = options.references || [];
        options1.nodeId = options.nodeId;
        if (options.subtypeOf) {
            const subtypeOfNodeId = addressSpace._coerceType(options.subtypeOf, "References", node_opcua_data_model_4.NodeClass.ReferenceType);
            (0, node_opcua_assert_1.assert)(subtypeOfNodeId);
            options1.references.push({
                isForward: false,
                nodeId: subtypeOfNodeId,
                referenceType: "HasSubtype"
            });
        }
        const node = this.internalCreateNode(options1);
        node.propagate_back_references();
        return node;
    }
    /**
     */
    addMultiStateDiscrete(options) {
        return (0, ua_multistate_discrete_impl_1._addMultiStateDiscrete)(this, options);
    }
    /**

     */
    createDataType(options) {
        (0, node_opcua_assert_1.assert)(Object.prototype.hasOwnProperty.call(options, "isAbstract"), "must provide isAbstract");
        (0, node_opcua_assert_1.assert)(!Object.prototype.hasOwnProperty.call(options, "nodeClass"));
        (0, node_opcua_assert_1.assert)(Object.prototype.hasOwnProperty.call(options, "browseName"), "must provide a browseName");
        const options1 = options;
        options1.nodeClass = node_opcua_data_model_4.NodeClass.DataType;
        options1.references = options.references || [];
        if (options1.references.length === 0) {
            if (!options1.subtypeOf) {
                throw new Error("must provide a subtypeOf");
            }
        }
        if (options1.subtypeOf) {
            if (!(options1.subtypeOf instanceof ua_data_type_impl_1.UADataTypeImpl)) {
                options1.subtypeOf = this.addressSpace.findDataType(options1.subtypeOf);
            }
            if (!options1.subtypeOf) {
                throw new Error("cannot find subtypeOf ");
            }
            options1.references.push({
                isForward: false,
                nodeId: options1.subtypeOf.nodeId,
                referenceType: "HasSubtype"
            });
        }
        const node = this.internalCreateNode(options);
        node.propagate_back_references();
        return node;
    }
    /**

     * @param options
     * @param options.nodeClass
     * @param [options.nodeVersion {String} = "0" ] install nodeVersion
     * @param [options.modellingRule {String} = null]
     * @internal
     */
    createNode(options) {
        let node = null;
        const addressSpace = this.addressSpace;
        addressSpace.modelChangeTransaction(() => {
            (0, node_opcua_assert_1.assert)(isNonEmptyQualifiedName(options.browseName));
            // xx assert(Object.prototype.hasOwnProperty.call(options,"browseName") && options.browseName.length > 0);
            (0, node_opcua_assert_1.assert)(Object.prototype.hasOwnProperty.call(options, "nodeClass"));
            options.references = addressSpace.normalizeReferenceTypes(options.references);
            const references = _copy_references(options.references);
            _handle_hierarchy_parent(addressSpace, references, options);
            _handle_event_hierarchy_parent(addressSpace, references, options);
            (0, namespace_private_1.UANamespace_process_modelling_rule)(references, options.modellingRule);
            options.references = references;
            node = this.internalCreateNode(options);
            (0, node_opcua_assert_1.assert)(node.nodeId instanceof node_opcua_nodeid_2.NodeId);
            node.propagate_back_references();
            node.install_extra_properties();
            _create_node_version_if_needed(node, options);
            (0, address_space_change_event_tools_1._handle_model_change_event)(node);
        });
        return node;
    }
    /**
     * remove the specified Node from the address space
     *

     * @param  nodeOrNodeId
     *
     *
     */
    deleteNode(nodeOrNodeId) {
        let node = null;
        let nodeId = new node_opcua_nodeid_2.NodeId();
        if (nodeOrNodeId instanceof node_opcua_nodeid_2.NodeId) {
            nodeId = nodeOrNodeId;
            node = this.findNode(nodeId);
            // istanbul ignore next
            if (!node) {
                throw new Error(" deleteNode : cannot find node with nodeId" + nodeId.toString());
            }
        }
        else if (nodeOrNodeId instanceof base_node_impl_1.BaseNodeImpl) {
            node = nodeOrNodeId;
            nodeId = node.nodeId;
        }
        // istanbul ignore next
        if (nodeId.namespace !== this.index) {
            throw new Error("this node doesn't belong to this namespace");
        }
        const addressSpace = this.addressSpace;
        addressSpace.modelChangeTransaction(() => {
            /* istanbul ignore next */
            if (!node) {
                throw new Error("this node doesn't belong to this namespace");
            }
            // notify parent that node is being removed
            const hierarchicalReferences = node.findReferencesEx("HierarchicalReferences", node_opcua_data_model_3.BrowseDirection.Inverse);
            for (const ref of hierarchicalReferences) {
                (0, node_opcua_assert_1.assert)(!ref.isForward);
                const parent = addressSpace.findNode(ref.nodeId);
                (0, node_opcua_assert_1.assert)(parent);
                parent._on_child_removed(node);
            }
            function deleteNodePointedByReference(ref) {
                const o = addressSpace.findNode(ref.nodeId);
                addressSpace.deleteNode(o.nodeId);
            }
            // recursively delete all nodes below in the hierarchy of nodes
            // TODO : a better idea would be to extract any references of type "HasChild"
            const components = node.findReferencesEx("HasComponent", node_opcua_data_model_3.BrowseDirection.Forward);
            const properties = node.findReferencesEx("HasProperty", node_opcua_data_model_3.BrowseDirection.Forward);
            // TODO: shall we delete nodes pointed by "Organizes" links here ?
            const subFolders = node.findReferencesEx("Organizes", node_opcua_data_model_3.BrowseDirection.Forward);
            for (const r of components) {
                deleteNodePointedByReference(r);
            }
            for (const r of properties) {
                deleteNodePointedByReference(r);
            }
            for (const r of subFolders) {
                deleteNodePointedByReference(r);
            }
            (0, address_space_change_event_tools_1._handle_delete_node_model_change_event)(node);
            detachNode(node);
            // delete nodes from global index
            const namespace = addressSpace.getNamespace(node.nodeId.namespace);
            namespace._deleteNode(node);
        });
    }
    /**
     * @internal
     */
    getStandardsNodeIds() {
        const standardNodeIds = {
            objectTypeIds: {},
            referenceTypeIds: {}
        };
        for (const referenceType of this._referenceTypeMap.values()) {
            standardNodeIds.referenceTypeIds[referenceType.browseName.name] = referenceType.nodeId.toString();
        }
        for (const objectType of this._objectTypeMap.values()) {
            standardNodeIds.objectTypeIds[objectType.browseName.name] = objectType.nodeId.toString();
        }
        return standardNodeIds;
    }
    // - Events --------------------------------------------------------------------------------------
    /**
     * add a new event type to the address space

     * @param options
     * @param options.browseName {String} the eventType name
     * @param [options.subtypeOf ="BaseEventType"]
     * @param [options.isAbstract = true]
     * @return {UAObjectType} : the object type
     *
     * @example
     *
     *      var evtType = namespace.addEventType({
     *          browseName: "MyAuditEventType",
     *          subtypeOf:  "AuditEventType"
     *      });
     *      var myConditionType = namespace.addEventType({
     *          browseName: "MyConditionType",
     *          subtypeOf:  "ConditionType",
     *          isAbstract: false
     *      });
     *
     */
    addEventType(options) {
        options.subtypeOf = options.subtypeOf || "BaseEventType";
        // are eventType always abstract ?? No => Condition can be instantiated!
        // but, by default is abstract is true
        options.isAbstract = Object.prototype.hasOwnProperty.call(options, "isAbstract") ? !!options.isAbstract : true;
        return this.addObjectType(options);
    }
    // ---------------------------------------------------------------------------------------------------
    /**
     *
     */
    addDataItem(options) {
        const addressSpace = this.addressSpace;
        const dataType = options.dataType || "Number";
        const dataItemType = addressSpace.findVariableType("DataItemType");
        if (!dataItemType) {
            throw new Error("Cannot find DataItemType");
        }
        const variable = this.addVariable({
            ...options,
            dataType,
            typeDefinition: dataItemType.nodeId
        });
        (0, add_dataItem_stuff_1.add_dataItem_stuff)(variable, options);
        variable.install_extra_properties();
        return variable;
    }
    /**
     *

     *
     * AnalogDataItem DataItems that represent continuously-variable physical quantities ( e.g., length, temperature),
     * in contrast to the digital representation of data in discrete  items
     * NOTE Typical examples are the values provided by temperature sensors or pressure sensors. OPC UA defines a
     * specific UAVariableType to identify an AnalogItem. Properties describe the possible ranges of  AnalogItems.
     *
     *
     * @example:
     *
     *
     *   namespace.add_analog_dataItem({
     *      componentOf: parentObject,
     *      browseName: "TemperatureSensor",
     *
     *      definition: "(tempA -25) + tempB",
     *      valuePrecision: 0.5,
     *      //-
     *      instrumentRange: { low: 100 , high: 200}, // optional
     *      engineeringUnitsRange: { low: 100 , high: 200}, // mandatory
     *      engineeringUnits: standardUnits.degree_celsius,, // optional
     *
     *      // access level
     *      accessLevel: 1
     *      minimumSamplingInterval: 10,
     *
     *   });
     *
  
     * @return {UAVariable}
     */
    addAnalogDataItem(options) {
        const addressSpace = this.addressSpace;
        (0, node_opcua_assert_1.assert)(Object.prototype.hasOwnProperty.call(options, "engineeringUnitsRange"), "expecting engineeringUnitsRange");
        const dataType = options.dataType || "Number";
        const analogItemType = addressSpace.findVariableType("AnalogItemType");
        if (!analogItemType) {
            throw new Error("expecting AnalogItemType to be defined , check nodeset xml file");
        }
        const clone_options = { ...options, dataType, typeDefinition: analogItemType.nodeId };
        const variable = this.addVariable(clone_options);
        (0, add_dataItem_stuff_1.add_dataItem_stuff)(variable, options);
        // mandatory (EURange in the specs)
        // OPC Unified Architecture, Part 8  6  Release 1.02
        // EURange  defines the value range likely to be obtained in normal operation. It is intended for such
        // use as automatically scaling a bar graph display
        // Sensor or instrument failure or deactivation can result in a return ed item value which is actually
        // outside  of  this range. Client software must be prepared to deal with this   possibility. Similarly a client
        // may attempt to write a value that is outside  of  this range back to the server. The exact behavior
        // (accept, reject, clamp, etc.) in this case is server - dependent. However ,   in general servers  shall  be
        // prepared to handle this.
        //     Example:    EURange ::= {-200.0,1400.0}
        const euRange = this.addVariable({
            browseName: { name: "EURange", namespaceIndex: 0 },
            dataType: "Range",
            minimumSamplingInterval: 0,
            modellingRule: options.modellingRule,
            propertyOf: variable,
            typeDefinition: "PropertyType",
            value: new node_opcua_variant_1.Variant({
                dataType: node_opcua_variant_1.DataType.ExtensionObject,
                value: new node_opcua_types_1.Range(options.engineeringUnitsRange)
            })
        });
        (0, node_opcua_assert_1.assert)(euRange.readValue().value.value instanceof node_opcua_types_1.Range);
        const handler = variable.handle_semantic_changed.bind(variable);
        euRange.on("value_changed", handler);
        if (Object.prototype.hasOwnProperty.call(options, "instrumentRange")) {
            const instrumentRange = this.addVariable({
                accessLevel: "CurrentRead | CurrentWrite",
                browseName: { name: "InstrumentRange", namespaceIndex: 0 },
                dataType: "Range",
                minimumSamplingInterval: 0,
                modellingRule: options.modellingRule ? "Mandatory" : undefined,
                propertyOf: variable,
                typeDefinition: "PropertyType",
                value: new node_opcua_variant_1.Variant({
                    dataType: node_opcua_variant_1.DataType.ExtensionObject,
                    value: new node_opcua_types_1.Range(options.instrumentRange)
                })
            });
            instrumentRange.on("value_changed", handler);
        }
        variable.acceptValueOutOfRange = options.acceptValueOutOfRange;
        if (Object.prototype.hasOwnProperty.call(options, "engineeringUnits")) {
            const engineeringUnits = new node_opcua_types_1.EUInformation(options.engineeringUnits);
            (0, node_opcua_assert_1.assert)(engineeringUnits instanceof node_opcua_types_1.EUInformation, "expecting engineering units");
            // EngineeringUnits  specifies the units for the   DataItem‟s value (e.g., degree, hertz, seconds).   The
            // EUInformation   type is specified in   5.6.3.
            const eu = this.addVariable({
                accessLevel: "CurrentRead",
                browseName: { name: "EngineeringUnits", namespaceIndex: 0 },
                dataType: "EUInformation",
                minimumSamplingInterval: 0,
                modellingRule: options.modellingRule ? "Mandatory" : undefined,
                propertyOf: variable,
                typeDefinition: "PropertyType",
                value: new node_opcua_variant_1.Variant({
                    dataType: node_opcua_variant_1.DataType.ExtensionObject,
                    value: engineeringUnits
                })
            });
            eu.on("value_changed", handler);
        }
        variable.install_extra_properties();
        return variable;
    }
    /**
     *

     * @param options {Object}
     * @param options.browseName {String}
     * @param [options.nodeId  {NodeId}]
     * @param [options.value {UInt32} = 0 }
     * @param options.enumValues { EnumValueType[]| {Key,Value} }
     * @return {Object|UAVariable}
     *
     * @example
     *
     *
     *      namespace.addMultiStateValueDiscrete({
     *          componentOf:parentObj,
     *          browseName: "myVar",
     *          enumValues: {
     *              "Red":    0xFF0000,
     *              "Green":  0x00FF00,
     *              "Blue":   0x0000FF
     *          }
     *      });
     *      addMultiStateValueDiscrete(parentObj,{
     *          browseName: "myVar",
     *          enumValues: [
     *              {
     *                 value: 0xFF0000,
     *                 displayName: "Red",
     *                 description: " The color Red"
     *              },
     *              {
     *                 value: 0x00FF000,
     *                 displayName: "Green",
     *                 description: " The color Green"
     *              },
     *              {
     *                 value: 0x0000FF,
     *                 displayName: "Blue",
     *                 description: " The color Blue"
     *              }
     *
     *          ]
     *      });
     */
    addMultiStateValueDiscrete(options) {
        return (0, ua_multistate_value_discrete_impl_1._addMultiStateValueDiscrete)(this, options);
    }
    // -
    /**
     *

     * @param options
     * @param options.componentOf {NodeId}
     * @param options.browseName {String}
     * @param options.title {String}
     * @param [options.instrumentRange]
     * @param [options.instrumentRange.low] {Double}
     * @param [options.instrumentRange.high] {Double}
     * @param options.engineeringUnitsRange.low {Double}
     * @param options.engineeringUnitsRange.high {Double}
     * @param options.engineeringUnits {String}
     * @param [options.nodeId = {NodeId}]
     * @param options.accessLevel
     * @param options.userAccessLevel
     * @param options.title {String}
     * @param options.axisScaleType {AxisScaleEnumeration}
     *
     * @param options.xAxisDefinition {AxisInformation}
     * @param options.xAxisDefinition.engineeringUnits  EURange
     * @param options.xAxisDefinition.range
     * @param options.xAxisDefinition.range.low
     * @param options.xAxisDefinition.range.high
     * @param options.xAxisDefinition.title  {LocalizedText}
     * @param options.xAxisDefinition.axisScaleType {AxisScaleEnumeration}
     * @param options.xAxisDefinition.axisSteps = <null>  {Array<Double>}
     * @param options.value
     */
    addYArrayItem(options) {
        (0, node_opcua_assert_1.assert)(Object.prototype.hasOwnProperty.call(options, "engineeringUnitsRange"), "expecting engineeringUnitsRange");
        (0, node_opcua_assert_1.assert)(Object.prototype.hasOwnProperty.call(options, "axisScaleType"), "expecting axisScaleType");
        (0, node_opcua_assert_1.assert)(options.xAxisDefinition !== null && typeof options.xAxisDefinition === "object", "expecting a xAxisDefinition");
        const addressSpace = this.addressSpace;
        const YArrayItemType = addressSpace.findVariableType("YArrayItemType");
        if (!YArrayItemType) {
            throw new Error("expecting YArrayItemType to be defined , check nodeset xml file");
        }
        function toNodeId(options) {
            if (!options) {
                return (0, node_opcua_nodeid_1.resolveNodeId)(node_opcua_variant_1.DataType.Float);
            }
            if (Object.prototype.hasOwnProperty.call(options, "nodeId") || options instanceof base_node_impl_1.BaseNodeImpl) {
                return options.nodeId;
            }
            return (0, node_opcua_nodeid_1.resolveNodeId)(options);
        }
        const dataType = toNodeId(options.dataType);
        const optionals = [];
        if (Object.prototype.hasOwnProperty.call(options, "instrumentRange")) {
            optionals.push("InstrumentRange");
        }
        const variable = YArrayItemType.instantiate({
            browseName: options.browseName,
            componentOf: options.componentOf,
            dataType,
            optionals
        });
        function coerceAxisScale(value) {
            const ret = node_opcua_data_access_1.AxisScaleEnumeration[value];
            (0, node_opcua_assert_1.assert)(!(0, node_opcua_utils_1.isNullOrUndefined)(ret));
            return ret;
        }
        variable.setValueFromSource(options.value, node_opcua_status_code_1.StatusCodes.Good);
        variable.euRange.setValueFromSource(new node_opcua_variant_1.Variant({
            dataType: node_opcua_variant_1.DataType.ExtensionObject,
            value: new node_opcua_types_1.Range(options.engineeringUnitsRange)
        }));
        if (Object.prototype.hasOwnProperty.call(options, "instrumentRange") && variable.instrumentRange) {
            variable.instrumentRange.setValueFromSource(new node_opcua_variant_1.Variant({
                dataType: node_opcua_variant_1.DataType.ExtensionObject,
                value: new node_opcua_types_1.Range(options.instrumentRange)
            }));
        }
        variable.title.setValueFromSource(new node_opcua_variant_1.Variant({
            dataType: node_opcua_variant_1.DataType.LocalizedText,
            value: (0, node_opcua_data_model_1.coerceLocalizedText)(options.title || "")
        }));
        // Linear/Log/Ln
        variable.axisScaleType.setValueFromSource(new node_opcua_variant_1.Variant({
            dataType: node_opcua_variant_1.DataType.Int32,
            value: coerceAxisScale(options.axisScaleType)
        }));
        variable.xAxisDefinition.setValueFromSource(new node_opcua_variant_1.Variant({
            dataType: node_opcua_variant_1.DataType.ExtensionObject,
            value: new node_opcua_types_1.AxisInformation(options.xAxisDefinition)
        }));
        return variable;
    }
    // - Methods ----------------------------------------------------------------------------------------------------
    /**

     * @param parentObject {Object}
     * @param options {Object}
     * @param [options.nodeId=null] {NodeId} the object nodeid.
     * @param [options.browseName=""] {String} the object browse name.
     * @param [options.description=""] {String} the object description.
     * @param options.inputArguments  {Array<Argument>}
     * @param options.outputArguments {Array<Argument>}
     * @return {Object}
     */
    addMethod(parentObject, options) {
        const addressSpace = this.addressSpace;
        (0, node_opcua_assert_1.assert)(parentObject !== null && typeof parentObject === "object" && parentObject instanceof base_node_impl_1.BaseNodeImpl, "expecting a valid parent object");
        (0, node_opcua_assert_1.assert)(Object.prototype.hasOwnProperty.call(options, "browseName"));
        options.componentOf = parentObject;
        const method = this._addMethod(options);
        const propertyTypeId = addressSpace._coerce_VariableTypeIds("PropertyType");
        const nodeId_ArgumentDataType = "Argument"; // makeNodeId(DataTypeIds.Argument);
        if (options.inputArguments) {
            const _inputArgs = new node_opcua_variant_1.Variant({
                arrayType: node_opcua_variant_1.VariantArrayType.Array,
                dataType: node_opcua_variant_1.DataType.ExtensionObject,
                value: options.inputArguments.map((opt) => new node_opcua_types_1.Argument(opt))
            });
            const inputArguments = this.addVariable({
                accessLevel: "CurrentRead",
                arrayDimensions: [_inputArgs.value.length],
                browseName: { name: "InputArguments", namespaceIndex: 0 },
                dataType: nodeId_ArgumentDataType,
                description: "the definition of the input argument of method " +
                    parentObject.browseName.toString() +
                    "." +
                    method.browseName.toString(),
                minimumSamplingInterval: -1,
                modellingRule: "Mandatory",
                propertyOf: method,
                typeDefinition: "PropertyType",
                value: _inputArgs,
                valueRank: 1
            });
            inputArguments.setValueFromSource(_inputArgs);
            (0, node_opcua_assert_1.assert)(inputArguments.typeDefinition.toString() === propertyTypeId.toString());
            (0, node_opcua_assert_1.assert)(Array.isArray(inputArguments.arrayDimensions));
        }
        if (options.outputArguments) {
            const _outputArgs = new node_opcua_variant_1.Variant({
                arrayType: node_opcua_variant_1.VariantArrayType.Array,
                dataType: node_opcua_variant_1.DataType.ExtensionObject,
                value: options.outputArguments.map((opts) => new node_opcua_types_1.Argument(opts))
            });
            const outputArguments = this.addVariable({
                accessLevel: "CurrentRead",
                arrayDimensions: [_outputArgs.value.length],
                browseName: { name: "OutputArguments", namespaceIndex: 0 },
                dataType: nodeId_ArgumentDataType,
                description: "the definition of the output arguments of method " +
                    parentObject.browseName.toString() +
                    "." +
                    method.browseName.toString(),
                minimumSamplingInterval: -1,
                modellingRule: "Mandatory",
                propertyOf: method,
                typeDefinition: "PropertyType",
                value: _outputArgs,
                valueRank: 1
            });
            outputArguments.setValueFromSource(_outputArgs);
            (0, node_opcua_assert_1.assert)(outputArguments.typeDefinition.toString() === propertyTypeId.toString());
            (0, node_opcua_assert_1.assert)(Array.isArray(outputArguments.arrayDimensions));
        }
        // verifying post-conditions
        parentObject.install_extra_properties();
        return method;
    }
    // - Enumeration ------------------------------------------------------------------------------------------------
    /**
     *

     * @param options
     * @param options.browseName  {String}
     * @param options.enumeration {Array}
     * @param options.enumeration[].displayName {String|LocalizedText}
     * @param options.enumeration[].value       {Number}
     * @param options.enumeration[].description {String|LocalizedText|null}
     */
    addEnumerationType(options) {
        // Release 1.03 OPC Unified Architecture, Part 3 - page 34
        // Enumeration DataTypes are DataTypes that represent discrete sets of named values.
        // Enumerations are always encoded as Int32 on the wire as defined in Part 6. Enumeration
        // DataTypes inherit directly or indirectly from the DataType Enumeration defined in 8.14.
        // Enumerations have no encodings exposed in the IAddressSpace. To expose the human readable
        // representation of an enumerated value the DataType Node may have the EnumString
        // Property that contains an array of LocalizedText. The Integer representation of the enumeration
        // value points to a position within that array. EnumValues Property can be used instead of the
        // EnumStrings to support integer representation of enumerations that are not zero-based or have
        // gaps. It contains an array of a Structured DataType containing the integer representation as
        // well as the human-readable representation. An example of an enumeration DataType containing
        // a sparse list of Integers is NodeClass which is defined in 8.30.
        // OPC Unified Architecture, Part 3  Release 1.03 page 35
        // Table 11 – DataType NodeClass
        // EnumStrings O LocalizedText[] The EnumStrings Property only applies for Enumeration DataTypes.
        //                               It shall not be applied for other DataTypes. If the EnumValues
        //                               Property is provided, the EnumStrings Property shall not be provided.
        //                               Each entry of the array of LocalizedText in this Property represents
        //                               the human-readable representation of an enumerated value. The
        //                               Integer representation of the enumeration value points to a position
        //                               of the array.
        // EnumValues O EnumValueType[]  The EnumValues Property only applies for Enumeration DataTypes.
        //                               It shall not be applied for other DataTypes. If the EnumStrings
        //                               Property is provided, the EnumValues Property shall not be provided.
        //                               Using the EnumValues Property it is possible to represent.
        //                               Enumerations with integers that are not zero-based or have gaps
        //                               (e.g. 1, 2, 4, 8, 16).
        //                               Each entry of the array of EnumValueType in this Property
        //                               represents one enumeration value with its integer notation, human readable
        //                                representation and help information.
        // The Property EnumStrings contains human-readable representations of enumeration values and is
        // only applied to Enumeration DataTypes. Instead of the EnumStrings Property an Enumeration
        // DataType can also use the EnumValues Property to represent Enumerations with integer values that are not
        // zero-based or containing gaps. There are no additional Properties defined for DataTypes in this standard.
        // Additional parts of this series of standards may define additional Properties for DataTypes.
        // 8.40 EnumValueType
        // This Structured DataType is used to represent a human-readable representation of an
        // Enumeration. Its elements are described inTable 27. When this type is used in an array representing
        // human-readable representations of an enumeration, each Value shall be unique in that array.
        // Table 27 – EnumValueType Definition
        // Name               Type            Description
        // EnumValueType structure
        // Value              Int64           The Integer representation of an Enumeration.
        // DisplayName        LocalizedText   A human-readable representation of the Value of the Enumeration.
        // Description        LocalizedText   A localized description of the enumeration value. This field can
        //                                    contain an empty string if no description is available.
        // Note that the EnumValueType has been defined with a Int64 Value to meet a variety of usages.
        // When it is used to define the string representation of an Enumeration DataType, the value range
        // is limited to Int32, because the Enumeration DataType is a subtype of Int32. Part 8 specifies
        // other usages where the actual value might be between 8 and 64 Bit.
        (0, node_opcua_assert_1.assert)(typeof options.browseName === "string");
        (0, node_opcua_assert_1.assert)(Array.isArray(options.enumeration));
        const addressSpace = this.addressSpace;
        let definition;
        const enumerationType = addressSpace.findDataType("Enumeration");
        (0, node_opcua_assert_1.assert)(enumerationType.nodeId instanceof node_opcua_nodeid_2.NodeId);
        (0, node_opcua_assert_1.assert)(enumerationType instanceof ua_data_type_impl_1.UADataTypeImpl);
        const references = [{ referenceType: "HasSubtype", isForward: false, nodeId: enumerationType.nodeId }];
        const opts = {
            browseName: options.browseName,
            definition,
            description: (0, node_opcua_data_model_1.coerceLocalizedText)(options.description) || null,
            displayName: options.displayName || null,
            isAbstract: false,
            nodeClass: node_opcua_data_model_4.NodeClass.DataType,
            references
        };
        const enumType = this.internalCreateNode(opts); //  as UAEnumeration;
        enumType.propagate_back_references();
        if (typeof options.enumeration[0] === "string") {
            const enumeration = options.enumeration;
            // enumeration is a array of string
            definition = enumeration.map((str, index) => (0, node_opcua_data_model_1.coerceLocalizedText)(str));
            const value = new node_opcua_variant_1.Variant({
                arrayType: node_opcua_variant_1.VariantArrayType.Array,
                dataType: node_opcua_variant_1.DataType.LocalizedText,
                value: definition
            });
            const enumStrings = this.addVariable({
                browseName: { name: "EnumStrings", namespaceIndex: 0 },
                dataType: "LocalizedText",
                description: "",
                modellingRule: "Mandatory",
                propertyOf: enumType,
                value,
                valueRank: 1
            });
            (0, node_opcua_assert_1.assert)(enumStrings.browseName.toString() === "EnumStrings");
            // set $definition
            // EnumDefinition
            //   This Structured DataType is used to provide the metadata for a custom Enumeration or
            //   OptionSet DataType. It is derived from the DataType DataTypeDefinition.
            // Enum Field:
            //   This Structured DataType is used to provide the metadata for a field of a custom Enumeration
            //   or OptionSet DataType. It is derived from the DataType EnumValueType. If used for an
            //   OptionSet, the corresponding Value in the base type contains the number of the bit associated
            //   with the field. The EnumField is formally defined in Table 37.
            enumType.$fullDefinition = new node_opcua_types_1.EnumDefinition({
                fields: enumeration.map((x, index) => new node_opcua_types_1.EnumField({
                    name: x,
                    description: (0, node_opcua_data_model_1.coerceLocalizedText)(x),
                    value: (0, node_opcua_basic_types_1.coerceInt64)(index)
                }))
            });
        }
        else {
            const enumeration = options.enumeration;
            // construct the definition object
            definition = enumeration.map((enumItem) => {
                return new node_opcua_types_1.EnumValueType({
                    description: (0, node_opcua_data_model_1.coerceLocalizedText)(enumItem.description),
                    displayName: (0, node_opcua_data_model_1.coerceLocalizedText)(enumItem.displayName),
                    value: (0, node_opcua_basic_types_1.coerceInt64)(enumItem.value)
                });
            });
            const value = new node_opcua_variant_1.Variant({
                arrayType: node_opcua_variant_1.VariantArrayType.Array,
                dataType: node_opcua_variant_1.DataType.ExtensionObject,
                value: definition
            });
            const enumValues = this.addVariable({
                browseName: { name: "EnumValues", namespaceIndex: 0 },
                dataType: "EnumValueType",
                description: undefined,
                modellingRule: "Mandatory",
                propertyOf: enumType,
                value,
                valueRank: 1
            });
            (0, node_opcua_assert_1.assert)(enumValues.browseName.toString() === "EnumValues");
            enumType.$fullDefinition = new node_opcua_types_1.EnumDefinition({
                fields: enumeration.map((x, index) => new node_opcua_types_1.EnumField({
                    name: x.displayName.toString(),
                    description: x.description || "",
                    value: (0, node_opcua_basic_types_1.coerceInt64)(x.value)
                }))
            });
        }
        // now create the string value property
        // <UAVariable NodeId="i=7612" BrowseName="EnumStrings"
        //               ParentNodeId="i=852" DataType="LocalizedText" ValueRank="1">
        // <DisplayName>EnumStrings</DisplayName>
        // <References>
        //   <Reference ReferenceType="HasTypeDefinition">i=68</Reference>
        //   <Reference ReferenceType="HasModellingRule">i=78</Reference>
        //    <Reference ReferenceType="HasProperty" IsForward="false">i=852</Reference>
        // </References>
        // <Value>
        //    <ListOfLocalizedText xmlns="http://opcfoundation.org/UA/2008/02/Types.xsd">
        //      <LocalizedText><Locale></Locale><Text>Running</Text></LocalizedText>
        //      <LocalizedText><Locale></Locale><Text>Failed</Text>
        //    </ListOfLocalizedText>
        // </Value>
        // </UAVariable>
        return enumType;
    }
    // -------------------------------------------------------------------------
    // State and Transition
    // -------------------------------------------------------------------------
    toNodeset2XML() {
        return "<toNodeset2XML>has not be installed</toNodeset2XML>!";
    }
    setRequiredModels(requiredModels) {
        this._requiredModels = requiredModels;
    }
    getRequiredModels() {
        return this._requiredModels;
    }
    // -------------------------------------------------------------------------
    // State and Transition
    // -------------------------------------------------------------------------
    /**
    */
    addState(component, stateName, stateNumber, isInitialState) {
        const addressSpace = this.addressSpace;
        isInitialState = !!isInitialState;
        const _component = component;
        (0, node_opcua_assert_1.assert)(_component.nodeClass === node_opcua_data_model_4.NodeClass.Object || _component.nodeClass === node_opcua_data_model_4.NodeClass.ObjectType);
        const initialStateType = addressSpace.findObjectType("InitialStateType");
        const stateType = addressSpace.findObjectType("StateType");
        let state;
        if (isInitialState) {
            state = initialStateType.instantiate({
                browseName: stateName,
                componentOf: _component
            });
        }
        else {
            state = stateType.instantiate({
                browseName: stateName,
                componentOf: _component
            });
        }
        // ensure state number is unique
        state.stateNumber.setValueFromSource({
            dataType: node_opcua_variant_1.DataType.UInt32,
            value: stateNumber
        });
        return state;
    }
    /**
     */
    addTransition(component, fromState, toState, transitionNumber, browseName) {
        const addressSpace = this.addressSpace;
        const _component = component;
        (0, node_opcua_assert_1.assert)(_component.nodeClass === node_opcua_data_model_4.NodeClass.Object || _component.nodeClass === node_opcua_data_model_4.NodeClass.ObjectType);
        (0, node_opcua_assert_1.assert)(typeof fromState === "string");
        (0, node_opcua_assert_1.assert)(typeof toState === "string");
        (0, node_opcua_assert_1.assert)(isFinite(transitionNumber));
        const fromStateNode = _component.getComponentByName(fromState);
        // istanbul ignore next
        if (!fromStateNode) {
            throw new Error("Cannot find state with name " + fromState);
        }
        (0, node_opcua_assert_1.assert)(fromStateNode.browseName.name.toString() === fromState);
        const toStateNode = _component.getComponentByName(toState);
        // istanbul ignore next
        if (!toStateNode) {
            throw new Error("Cannot find state with name " + toState);
        }
        (0, node_opcua_assert_1.assert)(toStateNode.browseName.name.toString() === toState);
        const transitionType = addressSpace.findObjectType("TransitionType");
        if (!transitionType) {
            throw new Error("Cannot find TransitionType");
        }
        browseName = browseName || fromState + "To" + toState; //  "Transition";
        const transition = transitionType.instantiate({
            browseName,
            componentOf: _component
        });
        transition.addReference({
            isForward: true,
            nodeId: toStateNode.nodeId,
            referenceType: "ToState"
        });
        transition.addReference({
            isForward: true,
            nodeId: fromStateNode.nodeId,
            referenceType: "FromState"
        });
        transition.transitionNumber.setValueFromSource({
            dataType: node_opcua_variant_1.DataType.UInt32,
            value: transitionNumber
        });
        return transition;
    }
    /**

     *
     * @return {UATwoStateVariable}
     */
    addTwoStateVariable(options) {
        return (0, ua_two_state_variable_1._addTwoStateVariable)(this, options);
    }
    /**

     *
     * Add a TwoStateDiscrete Variable
     * @return {UATwoStateDiscrete}
     */
    addTwoStateDiscrete(options) {
        return (0, ua_two_state_discrete_impl_1._addTwoStateDiscrete)(this, options);
    }
    // --- Alarms & Conditions -------------------------------------------------
    instantiateCondition(conditionTypeId, options, data) {
        return ua_condition_impl_1.UAConditionImpl.instantiate(this, conditionTypeId, options, data);
    }
    instantiateAcknowledgeableCondition(conditionTypeId, options, data) {
        return alarms_and_conditions_1.UAAcknowledgeableConditionImpl.instantiate(this, conditionTypeId, options, data);
    }
    instantiateAlarmCondition(alarmConditionTypeId, options, data) {
        return alarms_and_conditions_1.UAAlarmConditionImpl.instantiate(this, alarmConditionTypeId, options, data);
    }
    instantiateLimitAlarm(limitAlarmTypeId, options, data) {
        return ua_limit_alarm_impl_1.UALimitAlarmImpl.instantiate(this, limitAlarmTypeId, options, data);
    }
    instantiateExclusiveLimitAlarm(exclusiveLimitAlarmTypeId, options, data) {
        return ua_exclusive_limit_alarm_impl_1.UAExclusiveLimitAlarmImpl.instantiate(this, exclusiveLimitAlarmTypeId, options, data);
    }
    instantiateExclusiveDeviationAlarm(options, data) {
        return ua_exclusive_deviation_alarm_impl_1.UAExclusiveDeviationAlarmImpl.instantiate(this, "ExclusiveDeviationAlarmType", options, data);
    }
    instantiateNonExclusiveLimitAlarm(nonExclusiveLimitAlarmTypeId, options, data) {
        return ua_non_exclusive_limit_alarm_impl_1.UANonExclusiveLimitAlarmImpl.instantiate(this, nonExclusiveLimitAlarmTypeId, options, data);
    }
    instantiateNonExclusiveDeviationAlarm(options, data) {
        return ua_non_exclusive_deviation_alarm_impl_1.UANonExclusiveDeviationAlarmImpl.instantiate(this, "NonExclusiveDeviationAlarmType", options, data);
    }
    instantiateDiscreteAlarm(discreteAlarmType, options, data) {
        return ua_discrete_alarm_impl_1.UADiscreteAlarmImpl.instantiate(this, discreteAlarmType, options, data);
    }
    instantiateOffNormalAlarm(options, data) {
        return ua_off_normal_alarm_impl_1.UAOffNormalAlarmImpl.instantiate(this, "OffNormalAlarmType", options, data);
    }
    // default roles and permissions
    setDefaultRolePermissions(rolePermissions) {
        this.defaultRolePermissions = rolePermissions ? (0, role_permissions_1.coerceRolePermissions)(rolePermissions) : undefined;
    }
    getDefaultRolePermissions() {
        return this.defaultRolePermissions || null;
    }
    setDefaultAccessRestrictions(accessRestrictions) {
        this.defaultAccessRestrictions = accessRestrictions;
    }
    getDefaultAccessRestrictions() {
        return this.defaultAccessRestrictions || node_opcua_data_model_1.AccessRestrictionsFlag.None;
    }
    // --- internal stuff
    constructNodeId(options) {
        return this._nodeIdManager.constructNodeId({
            registerSymbolicNames: this.registerSymbolicNames,
            ...options
        });
    }
    _register(node) {
        (0, node_opcua_assert_1.assert)(node instanceof base_node_impl_1.BaseNodeImpl, "Expecting a instance of BaseNode in _register");
        (0, node_opcua_assert_1.assert)(node.nodeId instanceof node_opcua_nodeid_2.NodeId, "Expecting a NodeId");
        // istanbul ignore next
        if (node.nodeId.namespace !== this.index) {
            throw new Error("node must belong to this namespace : " +
                node.nodeId.toString() +
                "  " +
                " node.browseName = " +
                node.browseName.toString() +
                " this.index = " +
                this.index);
        }
        (0, node_opcua_assert_1.assert)(node.nodeId.namespace === this.index, "node must belongs to this namespace");
        (0, node_opcua_assert_1.assert)(Object.prototype.hasOwnProperty.call(node, "browseName"), "Node must have a browseName");
        const hashKey = _makeHashKey(node.nodeId);
        // istanbul ignore next
        if (this._nodeid_index.has(hashKey)) {
            const existingNode = this.findNode(node.nodeId);
            throw new Error("node " +
                node.browseName.toString() +
                " nodeId = " +
                node.nodeId.displayText() +
                " already registered " +
                node.nodeId.toString() +
                "\n" +
                " in namespace " +
                this.namespaceUri +
                " index = " +
                this.index +
                "\n" +
                "existing node = " +
                existingNode.toString() +
                "this parent : " +
                node.parentNodeId?.toString());
        }
        this._nodeid_index.set(hashKey, node);
        switch (node.nodeClass) {
            case node_opcua_data_model_4.NodeClass.ObjectType:
                this._registerObjectType(node);
                break;
            case node_opcua_data_model_4.NodeClass.VariableType:
                this._registerVariableType(node);
                break;
            case node_opcua_data_model_4.NodeClass.ReferenceType:
                this._registerReferenceType(node);
                break;
            case node_opcua_data_model_4.NodeClass.DataType:
                this._registerDataType(node);
                break;
            case node_opcua_data_model_4.NodeClass.Object:
            case node_opcua_data_model_4.NodeClass.Variable:
            case node_opcua_data_model_4.NodeClass.Method:
            case node_opcua_data_model_4.NodeClass.View:
                break;
            default:
                // tslint:disable-next-line:no-console
                errorLog("Invalid class Name", node.nodeClass);
                throw new Error("Invalid class name specified");
        }
    }
    /**

     * @internal
     */
    internalCreateNode(options) {
        (0, node_opcua_assert_1.assert)(options.nodeClass !== undefined, " options.nodeClass must be specified");
        (0, node_opcua_assert_1.assert)(options.browseName, "options.browseName must be specified");
        // xx assert(options.browseName instanceof QualifiedName
        // ? (options.browseName.namespaceIndex === this.index): true,
        // "Expecting browseName to have the same namespaceIndex as the namespace");
        options.description = (0, node_opcua_data_model_1.coerceLocalizedText)(options.description);
        // browseName adjustment
        if (typeof options.browseName === "string") {
            const match = options.browseName.match(regExpNamespaceDotBrowseName);
            if (match) {
                const correctedName = match[1];
                // the application is using an old scheme
                warningLog(chalk_1.default.green("Warning : since node-opcua 0.4.2 " + "namespace index should not be prepended to the browse name anymore"));
                warningLog("   ", options.browseName, " will be replaced with ", correctedName);
                warningLog(" Please update your code");
                const indexVerify = parseInt(match[0], 10);
                if (indexVerify !== this.index) {
                    errorLog(chalk_1.default.red.bold("Error: namespace index used at the front of the browseName " +
                        indexVerify +
                        " do not match the index of the current namespace (" +
                        this.index +
                        ")"));
                    errorLog(" Please fix your code so that the created node is inserted in the correct namespace," +
                        " please refer to the NodeOPCUA documentation");
                }
            }
            options.browseName = new node_opcua_data_model_2.QualifiedName({ name: options.browseName, namespaceIndex: this.index });
        }
        else if (!(options.browseName instanceof node_opcua_data_model_2.QualifiedName)) {
            options.browseName = new node_opcua_data_model_2.QualifiedName(options.browseName);
        }
        (0, node_opcua_assert_1.assert)(options.browseName instanceof node_opcua_data_model_2.QualifiedName, "Expecting options.browseName to be instanceof  QualifiedName ");
        // ------------- set display name
        if (!options.displayName) {
            (0, node_opcua_assert_1.assert)(typeof options.browseName.name === "string");
            options.displayName = (0, node_opcua_data_model_1.coerceLocalizedText)(options.browseName.name);
        }
        if (!options.nodeClass || options.nodeClass == undefined) {
            throw new Error("nodeclass must be specified");
        }
        // --- nodeId adjustment
        options.nodeId = this.constructNodeId(options);
        (0, node_opcua_debug_1.dumpIf)(!options.nodeId, options); // missing node Id
        (0, node_opcua_assert_1.assert)(options.nodeId instanceof node_opcua_nodeid_2.NodeId);
        // assert(options.browseName.namespaceIndex === this.index,"Expecting browseName to have
        // the same namespaceIndex as the namespace");
        const Constructor = _constructors_map[node_opcua_data_model_4.NodeClass[options.nodeClass]];
        if (!Constructor) {
            throw new Error(" missing constructor for NodeClass " + node_opcua_data_model_4.NodeClass[options.nodeClass]);
        }
        options.addressSpace = this.addressSpace;
        const node = new Constructor(options);
        this._register(node);
        // object shall now be registered
        // istanbul ignore next
        if (doDebug) {
            (0, node_opcua_assert_1.assert)(this.findNode(node.nodeId) !== null && typeof this.findNode(node.nodeId) === "object");
        }
        return node;
    }
    _deleteNode(node) {
        (0, node_opcua_assert_1.assert)(node instanceof base_node_impl_1.BaseNodeImpl);
        const hashKey = _makeHashKey(node.nodeId);
        // istanbul ignore next
        if (!this._nodeid_index.has(hashKey)) {
            throw new Error("deleteNode : nodeId " + node.nodeId.displayText() + " is not registered " + node.nodeId.toString());
        }
        switch (node.nodeClass) {
            case node_opcua_data_model_4.NodeClass.ObjectType:
                this._unregisterObjectType(node);
                break;
            case node_opcua_data_model_4.NodeClass.VariableType:
                this._unregisterVariableType(node);
                break;
            case node_opcua_data_model_4.NodeClass.Object:
            case node_opcua_data_model_4.NodeClass.Variable:
            case node_opcua_data_model_4.NodeClass.Method:
            case node_opcua_data_model_4.NodeClass.View:
                break;
            default:
                // tslint:disable:no-console
                warningLog("Invalid class Name", node.nodeClass);
                throw new Error("Invalid class name specified");
        }
        const deleted = this._nodeid_index.delete(hashKey);
        (0, node_opcua_assert_1.assert)(deleted);
        node.dispose();
    }
    // --- Private stuff
    _addObjectOrVariableType(options1, topMostBaseType, nodeClass) {
        const addressSpace = this.addressSpace;
        (0, node_opcua_assert_1.assert)(typeof topMostBaseType === "string");
        (0, node_opcua_assert_1.assert)(nodeClass === node_opcua_data_model_4.NodeClass.ObjectType || nodeClass === node_opcua_data_model_4.NodeClass.VariableType);
        const options = options1;
        (0, node_opcua_assert_1.assert)(!options.nodeClass);
        (0, node_opcua_assert_1.assert)(options.browseName);
        (0, node_opcua_assert_1.assert)(typeof options.browseName === "string");
        if (Object.prototype.hasOwnProperty.call(options, "references")) {
            throw new Error("options.references should not be provided, use options.subtypeOf instead");
        }
        const references = [];
        function process_subtypeOf_options(options2, references1) {
            // check common misspelling mistake
            (0, node_opcua_assert_1.assert)(!options2.subTypeOf, "misspell error : it should be 'subtypeOf' instead");
            if (Object.prototype.hasOwnProperty.call(options2, "hasTypeDefinition")) {
                throw new Error("hasTypeDefinition option is invalid. Do you mean typeDefinition instead ?");
            }
            (0, node_opcua_assert_1.assert)(!options2.typeDefinition, " do you mean subtypeOf ?");
            const subtypeOfNodeId = addressSpace._coerceType(options2.subtypeOf, topMostBaseType, nodeClass);
            (0, node_opcua_assert_1.assert)(subtypeOfNodeId);
            references1.push({
                isForward: false,
                nodeId: subtypeOfNodeId,
                referenceType: "HasSubtype"
            });
        }
        process_subtypeOf_options.call(this, options, references);
        const objectType = this.internalCreateNode({
            browseName: options.browseName,
            displayName: options.displayName,
            description: options.description,
            eventNotifier: +(options.eventNotifier || 0),
            isAbstract: !!options.isAbstract,
            nodeClass,
            nodeId: options.nodeId,
            references
        });
        objectType.propagate_back_references();
        objectType.install_extra_properties();
        if (options.postInstantiateFunc) {
            objectType.installPostInstallFunc(options.postInstantiateFunc);
        }
        return objectType;
    }
    // private  _adjust_options(options: any) {
    //     const ns = this.addressSpace.getNamespaceIndex(this.namespaceUri);
    //     if (!options.nodeId) {
    //         const id = this._getNextAvailableId();
    //         options.nodeId = new NodeId(NodeId.NodeIdType.NUMERIC, id, ns);
    //     }
    //     options.nodeId = NodeId.coerce(options.nodeId);
    //     if (typeof options.browseName === "string") {
    //         options.browseName = new QualifiedName({
    //             name: options.browseName,
    //             namespaceIndex: ns
    //         });
    //     }
    //     return options;
    // }
    _registerObjectType(node) {
        (0, node_opcua_assert_1.assert)(this.index === node.nodeId.namespace);
        const key = node.browseName.name;
        if (this._objectTypeMap.has(key)) {
            throw new Error(" UAObjectType already declared " + node.browseName.toString() + "  " + node.nodeId.toString());
        }
        this._objectTypeMap.set(key, node);
    }
    _registerVariableType(node) {
        (0, node_opcua_assert_1.assert)(this.index === node.nodeId.namespace);
        const key = node.browseName.name;
        (0, node_opcua_assert_1.assert)(!this._variableTypeMap.has(key), " UAVariableType already declared");
        this._variableTypeMap.set(key, node);
    }
    _registerReferenceType(node) {
        (0, node_opcua_assert_1.assert)(this.index === node.nodeId.namespace);
        (0, node_opcua_assert_1.assert)(node.browseName instanceof node_opcua_data_model_2.QualifiedName);
        const key = node.browseName.name;
        this._referenceTypeMap.set(key, node);
        this._referenceTypeMapInv.set(node.inverseName.text, node);
    }
    _registerDataType(node) {
        (0, node_opcua_assert_1.assert)(this.index === node.nodeId.namespace);
        const key = node.browseName.name;
        (0, node_opcua_assert_1.assert)(node.browseName instanceof node_opcua_data_model_2.QualifiedName);
        (0, node_opcua_assert_1.assert)(!this._dataTypeMap.has(key), " DataType already declared");
        this._dataTypeMap.set(key, node);
    }
    _unregisterObjectType(node) {
        const key = node.browseName.name;
        this._objectTypeMap.delete(key);
    }
    _unregisterVariableType(node) {
        const key = node.browseName.name;
        this._variableTypeMap.delete(key);
    }
    /**
     * @private
     */
    _addVariable(options) {
        const addressSpace = this.addressSpace;
        const baseDataVariableType = addressSpace.findVariableType("BaseDataVariableType");
        if (!baseDataVariableType) {
            throw new Error("cannot find BaseDataVariableType");
        }
        const baseDataVariableTypeId = baseDataVariableType.nodeId;
        (0, node_opcua_assert_1.assert)(Object.prototype.hasOwnProperty.call(options, "browseName"), "options.browseName must be provided");
        (0, node_opcua_assert_1.assert)(Object.prototype.hasOwnProperty.call(options, "dataType"), "options.dataType must be provided");
        options.historizing = !!options.historizing;
        // istanbul ignore next
        if (Object.prototype.hasOwnProperty.call(options, "hasTypeDefinition")) {
            throw new Error("hasTypeDefinition option is invalid. Do you mean typeDefinition instead ?");
        }
        // ------------------------------------------ TypeDefinition
        let typeDefinition = options.typeDefinition || baseDataVariableTypeId;
        if (typeDefinition instanceof base_node_impl_1.BaseNodeImpl) {
            // istanbul ignore next
            if (typeDefinition.nodeClass !== node_opcua_data_model_4.NodeClass.VariableType) {
                const message = `invalid typeDefinition expecting a VariableType got ${node_opcua_data_model_4.NodeClass[typeDefinition.nodeClass]}`;
                errorLog(message);
                throw new Error(message);
            }
        }
        typeDefinition = addressSpace._coerce_VariableTypeIds(typeDefinition);
        (0, node_opcua_assert_1.assert)(typeDefinition instanceof node_opcua_nodeid_2.NodeId);
        // ------------------------------------------ DataType
        options.dataType = addressSpace._coerce_DataType(options.dataType);
        options.valueRank = (0, node_opcua_utils_1.isNullOrUndefined)(options.valueRank)
            ? options.arrayDimensions
                ? options.arrayDimensions.length
                : -1
            : options.valueRank;
        (0, node_opcua_assert_1.assert)(typeof options.valueRank === "number" && isFinite(options.valueRank));
        options.arrayDimensions = options.arrayDimensions || null;
        (0, node_opcua_assert_1.assert)(options.arrayDimensions === null || Array.isArray(options.arrayDimensions));
        // -----------------------------------------------------
        const hasGetter = (options) => {
            return typeof options.value?.get === "function" || typeof options.value?.timestamped_get === "function";
        };
        // istanbul ignore next
        if (options.minimumSamplingInterval === undefined && hasGetter(options)) {
            // a getter has been specified and no options.minimumSamplingInterval has been specified
            warningLog("[NODE-OPCUA-W30", "namespace#addVariable a getter has been specified and minimumSamplingInterval is missing.\nMinimumSamplingInterval has been adjusted to 1000 ms\nvariable = " +
                options?.browseName?.toString());
            options.minimumSamplingInterval = 1000;
        }
        options.minimumSamplingInterval = options.minimumSamplingInterval !== undefined ? +options.minimumSamplingInterval : 0;
        // istanbul ignore next
        if (options.minimumSamplingInterval === 0 && hasGetter(options)) {
            warningLog("[NODE-OPCUA-W31", "namespace#addVariable a getter has been specified and minimumSamplingInterval is 0.\nThis may conduct to an unpredictable behavior.\nPlease specify a non zero minimum sampling interval");
        }
        let references = options.references || [];
        references = [].concat(references, [
            {
                isForward: true,
                nodeId: typeDefinition,
                referenceType: "HasTypeDefinition"
            }
        ]);
        (0, node_opcua_assert_1.assert)(!options.nodeClass || options.nodeClass === node_opcua_data_model_4.NodeClass.Variable);
        options.nodeClass = node_opcua_data_model_4.NodeClass.Variable;
        options.references = references;
        const variable = this.createNode(options);
        return variable;
    }
    /**
     * @private
     */
    _addMethod(options) {
        const addressSpace = this.addressSpace;
        (0, node_opcua_assert_1.assert)(isNonEmptyQualifiedName(options.browseName));
        const references = [];
        (0, node_opcua_assert_1.assert)(isNonEmptyQualifiedName(options.browseName));
        _handle_hierarchy_parent(addressSpace, references, options);
        (0, namespace_private_1.UANamespace_process_modelling_rule)(references, options.modellingRule);
        const method = this.internalCreateNode({
            browseName: options.browseName,
            description: options.description || "",
            displayName: options.displayName,
            eventNotifier: +options.eventNotifier,
            isAbstract: false,
            nodeClass: node_opcua_data_model_4.NodeClass.Method,
            nodeId: options.nodeId,
            references,
            rolePermissions: options.rolePermissions
        });
        (0, node_opcua_assert_1.assert)(method.nodeId !== null);
        method.propagate_back_references();
        (0, node_opcua_assert_1.assert)(!method.typeDefinition);
        return method;
    }
}
exports.NamespaceImpl = NamespaceImpl;
const _constructors_map = {
    DataType: ua_data_type_impl_1.UADataTypeImpl,
    Method: ua_method_impl_1.UAMethodImpl,
    Object: ua_object_impl_1.UAObjectImpl,
    ObjectType: ua_object_type_impl_1.UAObjectTypeImpl,
    ReferenceType: ua_reference_type_impl_1.UAReferenceTypeImpl,
    Variable: ua_variable_impl_1.UAVariableImpl,
    VariableType: ua_variable_type_impl_1.UAVariableTypeImpl,
    View: ua_view_impl_1.UAViewImpl
};
/**

 * convert a 'string' , NodeId or Object into a valid and existing object
 * @param addressSpace  {IAddressSpace}
 * @param value
 * @param coerceFunc
 * @private
 */
function _coerce_parent(addressSpace, value, coerceFunc) {
    (0, node_opcua_assert_1.assert)(typeof coerceFunc === "function");
    if (value) {
        if (typeof value === "string") {
            value = coerceFunc.call(addressSpace, value);
        }
        if (value instanceof node_opcua_nodeid_2.NodeId) {
            value = addressSpace.findNode(value);
        }
    }
    (0, node_opcua_assert_1.assert)(!value || value instanceof base_node_impl_1.BaseNodeImpl);
    return value;
}
function _handle_event_hierarchy_parent(addressSpace, references, options) {
    options.eventSourceOf = _coerce_parent(addressSpace, options.eventSourceOf, addressSpace._coerceNode);
    options.notifierOf = _coerce_parent(addressSpace, options.notifierOf, addressSpace._coerceNode);
    if (options.eventSourceOf) {
        (0, node_opcua_assert_1.assert)(!options.notifierOf, "notifierOf shall not be provided with eventSourceOf ");
        references.push({
            isForward: false,
            nodeId: options.eventSourceOf.nodeId,
            referenceType: "HasEventSource"
        });
        options.eventNotifier = options.eventNotifier || 1;
    }
    else if (options.notifierOf) {
        (0, node_opcua_assert_1.assert)(!options.eventSourceOf, "eventSourceOf shall not be provided with notifierOf ");
        references.push({
            isForward: false,
            nodeId: options.notifierOf.nodeId,
            referenceType: "HasNotifier"
        });
    }
}
function _handle_hierarchy_parent(addressSpace, references, options) {
    options.addInOf = _coerce_parent(addressSpace, options.addInOf, addressSpace._coerceNode);
    options.componentOf = _coerce_parent(addressSpace, options.componentOf, addressSpace._coerceNode);
    options.propertyOf = _coerce_parent(addressSpace, options.propertyOf, addressSpace._coerceNode);
    options.organizedBy = _coerce_parent(addressSpace, options.organizedBy, addressSpace._coerceFolder);
    options.encodingOf = _coerce_parent(addressSpace, options.encodingOf, addressSpace._coerceNode);
    if (options.addInOf) {
        (0, node_opcua_assert_1.assert)(!options.componentOf);
        (0, node_opcua_assert_1.assert)(!options.propertyOf);
        (0, node_opcua_assert_1.assert)(!options.organizedBy);
        (0, node_opcua_assert_1.assert)(options.addInOf.nodeClass === node_opcua_data_model_4.NodeClass.Object || options.addInOf.nodeClass === node_opcua_data_model_4.NodeClass.ObjectType, "addInOf must be of nodeClass Object or ObjectType");
        references.push({
            isForward: false,
            nodeId: options.addInOf.nodeId,
            referenceType: "HasAddIn"
        });
    }
    if (options.componentOf) {
        (0, node_opcua_assert_1.assert)(!options.addInOf);
        (0, node_opcua_assert_1.assert)(!options.propertyOf);
        (0, node_opcua_assert_1.assert)(!options.organizedBy);
        (0, node_opcua_assert_1.assert)(addressSpace.rootFolder.objects, "addressSpace must have a rootFolder.objects folder");
        (0, node_opcua_assert_1.assert)(options.componentOf.nodeId !== addressSpace.rootFolder.objects.nodeId, "Only Organizes References are used to relate Objects to the 'Objects' standard Object.");
        references.push({
            isForward: false,
            nodeId: options.componentOf.nodeId,
            referenceType: "HasComponent"
        });
    }
    if (options.propertyOf) {
        (0, node_opcua_assert_1.assert)(!options.addInOf);
        (0, node_opcua_assert_1.assert)(!options.componentOf);
        (0, node_opcua_assert_1.assert)(!options.organizedBy);
        (0, node_opcua_assert_1.assert)(options.propertyOf.nodeId !== addressSpace.rootFolder.objects.nodeId, "Only Organizes References are used to relate Objects to the 'Objects' standard Object.");
        references.push({
            isForward: false,
            nodeId: options.propertyOf.nodeId,
            referenceType: "HasProperty"
        });
    }
    if (options.organizedBy) {
        (0, node_opcua_assert_1.assert)(!options.addInOf);
        (0, node_opcua_assert_1.assert)(!options.propertyOf);
        (0, node_opcua_assert_1.assert)(!options.componentOf);
        references.push({
            isForward: false,
            nodeId: options.organizedBy.nodeId,
            referenceType: "Organizes"
        });
    }
    if (options.encodingOf) {
        // parent must be a DataType
        (0, node_opcua_assert_1.assert)(options.encodingOf.nodeClass === node_opcua_data_model_4.NodeClass.DataType, "encodingOf must be toward a DataType");
        references.push({
            isForward: false,
            nodeId: options.encodingOf.nodeId,
            referenceType: "HasEncoding"
        });
    }
}
function _copy_reference(reference) {
    (0, node_opcua_assert_1.assert)(Object.prototype.hasOwnProperty.call(reference, "referenceType"));
    (0, node_opcua_assert_1.assert)(Object.prototype.hasOwnProperty.call(reference, "isForward"));
    (0, node_opcua_assert_1.assert)(Object.prototype.hasOwnProperty.call(reference, "nodeId"));
    (0, node_opcua_assert_1.assert)(reference.nodeId instanceof node_opcua_nodeid_2.NodeId);
    return {
        isForward: reference.isForward,
        nodeId: reference.nodeId,
        referenceType: reference.referenceType
    };
}
function _copy_references(references) {
    references = references || [];
    return references.map(_copy_reference);
}
function isNonEmptyQualifiedName(browseName) {
    if (!browseName) {
        return false;
    }
    if (typeof browseName === "string") {
        return browseName.length >= 0;
    }
    if (!(browseName instanceof node_opcua_data_model_2.QualifiedName)) {
        browseName = new node_opcua_data_model_2.QualifiedName(browseName);
    }
    (0, node_opcua_assert_1.assert)(browseName instanceof node_opcua_data_model_2.QualifiedName);
    return browseName.name.length > 0;
}
function _create_node_version_if_needed(node, options) {
    (0, node_opcua_assert_1.assert)(options);
    if (typeof options.nodeVersion == "string") {
        (0, node_opcua_assert_1.assert)(node.nodeClass === node_opcua_data_model_4.NodeClass.Variable || node.nodeClass === node_opcua_data_model_4.NodeClass.Object);
        // istanbul ignore next
        if (node.getNodeVersion()) {
            return; // already exists
        }
        const namespace = node.namespace;
        const nodeVersion = namespace.addVariable({
            browseName: (0, node_opcua_data_model_1.coerceQualifiedName)({ name: "NodeVersion", namespaceIndex: 0 }),
            dataType: node_opcua_variant_1.DataType.String,
            propertyOf: node
        });
        const initialValue = typeof options.nodeVersion === "string" ? options.nodeVersion : "0";
        nodeVersion.setValueFromSource({ dataType: "String", value: initialValue });
    }
}
//# sourceMappingURL=namespace_impl.js.map