"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.UAMultiStateDiscreteImpl = void 0;
exports.promoteToMultiStateDiscrete = promoteToMultiStateDiscrete;
exports._addMultiStateDiscrete = _addMultiStateDiscrete;
/**
 * @module node-opcua-address-space.DataAccess
 */
const node_opcua_assert_1 = require("node-opcua-assert");
const node_opcua_constants_1 = require("node-opcua-constants");
const node_opcua_data_model_1 = require("node-opcua-data-model");
const node_opcua_status_code_1 = require("node-opcua-status-code");
const node_opcua_variant_1 = require("node-opcua-variant");
const register_node_promoter_1 = require("../../source/loader/register_node_promoter");
const ua_variable_impl_1 = require("../ua_variable_impl");
const add_dataItem_stuff_1 = require("./add_dataItem_stuff");
/**
 * @class UAMultiStateDiscrete
 */
class UAMultiStateDiscreteImpl extends ua_variable_impl_1.UAVariableImpl {
    getValue() {
        return this.readValue().value.value;
    }
    getValueAsString() {
        const index = this.getValue();
        const arr = this.enumStrings.readValue().value.value;
        (0, node_opcua_assert_1.assert)(Array.isArray(arr));
        return arr[index].text ? arr[index].text.toString() : "????";
    }
    getIndex(value) {
        const arr = this.enumStrings.readValue().value.value;
        (0, node_opcua_assert_1.assert)(Array.isArray(arr));
        const index = arr.findIndex((a) => a.text === value);
        return index;
    }
    setValue(value, options) {
        if (typeof value === "string") {
            const index = this.getIndex(value);
            if (index < 0) {
                throw new Error("UAMultiStateDiscrete#setValue invalid multi state value provided : " + value);
            }
            return this.setValue(index, options);
        }
        const arrayEnumStrings = this.enumStrings.readValue().value.value;
        if (value >= arrayEnumStrings.length) {
            throw new Error("UAMultiStateDiscrete#setValue BadOutOfRange " + value);
        }
        (0, node_opcua_assert_1.assert)(isFinite(value));
        return this.setValueFromSource(new node_opcua_variant_1.Variant({ dataType: node_opcua_variant_1.DataType.UInt32, value }));
    }
    checkVariantCompatibility(value) {
        if (!this._validate_DataType(value.dataType)) {
            return node_opcua_status_code_1.StatusCodes.BadTypeMismatch;
        }
        if (this.enumStrings) {
            const arrayEnumStrings = this.enumStrings.readValue().value.value;
            // MultiStateDiscreteType
            (0, node_opcua_assert_1.assert)(value.dataType === node_opcua_variant_1.DataType.UInt32);
            if (value.value >= arrayEnumStrings.length) {
                return node_opcua_status_code_1.StatusCodes.BadOutOfRange;
            }
        }
        return node_opcua_status_code_1.StatusCodes.Good;
    }
    _post_initialize() {
        /* empty */
    }
    clone(options1, optionalFilter, extraInfo) {
        const variable1 = ua_variable_impl_1.UAVariableImpl.prototype.clone.call(this, options1, optionalFilter, extraInfo);
        return promoteToMultiStateDiscrete(variable1);
    }
}
exports.UAMultiStateDiscreteImpl = UAMultiStateDiscreteImpl;
function promoteToMultiStateDiscrete(node) {
    if (node instanceof UAMultiStateDiscreteImpl) {
        return node; // already promoted
    }
    Object.setPrototypeOf(node, UAMultiStateDiscreteImpl.prototype);
    (0, node_opcua_assert_1.assert)(node instanceof UAMultiStateDiscreteImpl, "should now  be a State Machine");
    const _node = node;
    _node._post_initialize();
    (0, node_opcua_assert_1.assert)(_node.enumStrings.browseName.toString() === "EnumStrings");
    const handler = _node.handle_semantic_changed.bind(_node);
    _node.enumStrings.on("value_changed", handler);
    _node.install_extra_properties();
    return node;
}
(0, register_node_promoter_1.registerNodePromoter)(node_opcua_constants_1.VariableTypeIds.MultiStateDiscreteType, promoteToMultiStateDiscrete);
function _addMultiStateDiscrete(namespace, options) {
    const addressSpace = namespace.addressSpace;
    (0, node_opcua_assert_1.assert)(Object.prototype.hasOwnProperty.call(options, "enumStrings"));
    (0, node_opcua_assert_1.assert)(!Object.prototype.hasOwnProperty.call(options, "ValuePrecision"));
    const multiStateDiscreteType = addressSpace.findVariableType("MultiStateDiscreteType");
    if (!multiStateDiscreteType) {
        throw new Error("Cannot find MultiStateDiscreteType");
    }
    // todo : if options.typeDefinition is specified, check that type is SubTypeOf MultiStateDiscreteType
    options.value = options.value === undefined ? 0 : options.value;
    let value;
    if (typeof options.value === "number") {
        value = new node_opcua_variant_1.Variant({
            dataType: node_opcua_variant_1.DataType.UInt32,
            value: options.value
        });
    }
    else {
        value = options.value;
    }
    const variable = namespace.addVariable({
        ...options,
        dataType: "Number",
        typeDefinition: multiStateDiscreteType.nodeId,
        value,
        valueRank: -2
    });
    (0, add_dataItem_stuff_1.add_dataItem_stuff)(variable, options);
    const enumStrings = options.enumStrings.map((value) => {
        return (0, node_opcua_data_model_1.coerceLocalizedText)(value);
    });
    const enumStringsNode = namespace.addVariable({
        accessLevel: "CurrentRead", // | CurrentWrite",
        browseName: { name: "EnumStrings", namespaceIndex: 0 },
        dataType: "LocalizedText",
        minimumSamplingInterval: 0,
        modellingRule: options.modellingRule ? "Mandatory" : undefined,
        propertyOf: variable,
        typeDefinition: "PropertyType",
        userAccessLevel: "CurrentRead", // CurrentWrite",
        value: new node_opcua_variant_1.Variant({
            arrayType: node_opcua_variant_1.VariantArrayType.Array,
            dataType: node_opcua_variant_1.DataType.LocalizedText,
            value: enumStrings
        })
    });
    return promoteToMultiStateDiscrete(variable);
}
//# sourceMappingURL=ua_multistate_discrete_impl.js.map