"use strict";
/**
 * @module node-opcua-address-space
 */
Object.defineProperty(exports, "__esModule", { value: true });
exports.SessionContext = exports.WellKnownRolesNodeId = exports.WellKnownRoles = exports.PermissionType = exports.RolePermissionType = void 0;
exports.makeRoles = makeRoles;
const node_opcua_assert_1 = require("node-opcua-assert");
const web_1 = require("node-opcua-crypto/web");
const node_opcua_data_model_1 = require("node-opcua-data-model");
const node_opcua_nodeid_1 = require("node-opcua-nodeid");
const node_opcua_types_1 = require("node-opcua-types");
var node_opcua_types_2 = require("node-opcua-types");
Object.defineProperty(exports, "RolePermissionType", { enumerable: true, get: function () { return node_opcua_types_2.RolePermissionType; } });
Object.defineProperty(exports, "PermissionType", { enumerable: true, get: function () { return node_opcua_types_2.PermissionType; } });
function getUserName(userIdentityToken) {
    if (userIdentityToken instanceof node_opcua_types_1.AnonymousIdentityToken) {
        return "anonymous";
    }
    if (userIdentityToken instanceof node_opcua_types_1.X509IdentityToken) {
        const certInfo = (0, web_1.exploreCertificate)(userIdentityToken.certificateData);
        const userName = certInfo.tbsCertificate.subject.commonName || "";
        if (typeof userName !== "string") {
            throw new Error("Invalid username");
        }
        return userName;
    }
    if (userIdentityToken instanceof node_opcua_types_1.UserNameIdentityToken) {
        if (userIdentityToken.policyId === "anonymous") {
            return "anonymous";
        }
        (0, node_opcua_assert_1.assert)(Object.prototype.hasOwnProperty.call(userIdentityToken, "userName"));
        return userIdentityToken.userName;
    }
    throw new Error("Invalid user identity token");
}
/**
 *
 */
var WellKnownRoles;
(function (WellKnownRoles) {
    WellKnownRoles[WellKnownRoles["Anonymous"] = 15644] = "Anonymous";
    WellKnownRoles[WellKnownRoles["AuthenticatedUser"] = 15656] = "AuthenticatedUser";
    WellKnownRoles[WellKnownRoles["ConfigureAdmin"] = 15716] = "ConfigureAdmin";
    WellKnownRoles[WellKnownRoles["Engineer"] = 16036] = "Engineer";
    WellKnownRoles[WellKnownRoles["Observer"] = 15668] = "Observer";
    WellKnownRoles[WellKnownRoles["Operator"] = 15680] = "Operator";
    WellKnownRoles[WellKnownRoles["SecurityAdmin"] = 15704] = "SecurityAdmin";
    WellKnownRoles[WellKnownRoles["Supervisor"] = 15692] = "Supervisor";
})(WellKnownRoles || (exports.WellKnownRoles = WellKnownRoles = {}));
var WellKnownRolesNodeId;
(function (WellKnownRolesNodeId) {
    WellKnownRolesNodeId[WellKnownRolesNodeId["Anonymous"] = 15644] = "Anonymous";
    WellKnownRolesNodeId[WellKnownRolesNodeId["AuthenticatedUser"] = 15656] = "AuthenticatedUser";
    WellKnownRolesNodeId[WellKnownRolesNodeId["ConfigureAdmin"] = 15716] = "ConfigureAdmin";
    WellKnownRolesNodeId[WellKnownRolesNodeId["Engineer"] = 16036] = "Engineer";
    WellKnownRolesNodeId[WellKnownRolesNodeId["Observer"] = 15668] = "Observer";
    WellKnownRolesNodeId[WellKnownRolesNodeId["Operator"] = 15680] = "Operator";
    WellKnownRolesNodeId[WellKnownRolesNodeId["SecurityAdmin"] = 15704] = "SecurityAdmin";
    WellKnownRolesNodeId[WellKnownRolesNodeId["Supervisor"] = 15692] = "Supervisor";
})(WellKnownRolesNodeId || (exports.WellKnownRolesNodeId = WellKnownRolesNodeId = {}));
function getPermissionForRole(rolePermissions, role) {
    if (rolePermissions === null) {
        return node_opcua_data_model_1.allPermissions;
    }
    const a = rolePermissions.find((r) => {
        return (0, node_opcua_nodeid_1.sameNodeId)((0, node_opcua_nodeid_1.resolveNodeId)(r.roleId), role);
    });
    return a !== undefined ? a.permissions | node_opcua_data_model_1.PermissionFlag.None : node_opcua_data_model_1.PermissionFlag.None;
}
function isDefaultContext(context) {
    return context === SessionContext.defaultContext;
}
function getAccessRestrictionsOnNamespace(namespace, context) {
    // ignore permission when default context is provided (to avoid recursion)
    if (isDefaultContext(context)) {
        return node_opcua_data_model_1.AccessRestrictionsFlag.None;
    }
    const namespaces = namespace.addressSpace.rootFolder?.objects?.server?.namespaces;
    if (!namespaces) {
        return node_opcua_data_model_1.AccessRestrictionsFlag.None;
    }
    const namespaceObject = namespaces.getChildByName(namespace.namespaceUri);
    if (!namespaceObject) {
        return node_opcua_data_model_1.AccessRestrictionsFlag.None;
    }
    const defaultAccessRestriction = namespaceObject.getChildByName("defaultAccessRestriction");
    if (defaultAccessRestriction) {
        const dataValue = defaultAccessRestriction.readAttribute(null, node_opcua_data_model_1.AttributeIds.Value);
        if (dataValue && dataValue.statusCode.isGood()) {
            return dataValue.value.value;
        }
    }
    return node_opcua_data_model_1.AccessRestrictionsFlag.None;
}
function getDefaultUserRolePermissionsOnNamespace(namespace, context) {
    // ignore permission when default context is provided
    if (isDefaultContext(context)) {
        return null;
    }
    const namespaces = namespace.addressSpace.rootFolder?.objects?.server?.namespaces;
    if (!namespaces) {
        return null;
    }
    const uaNamespaceObject = namespaces.getChildByName(namespace.namespaceUri);
    if (!uaNamespaceObject) {
        return null;
    }
    const defaultUserRolePermissions = uaNamespaceObject.getChildByName("DefaultUserRolePermissions");
    if (defaultUserRolePermissions) {
        const dataValue = defaultUserRolePermissions.readValue();
        if (dataValue && dataValue.statusCode.isGood() && dataValue.value.value && dataValue.value.value.length > 0) {
            return dataValue.value.value;
        }
    }
    const defaultRolePermissions = uaNamespaceObject.getChildByName("DefaultRolePermissions");
    if (defaultRolePermissions) {
        const dataValue = defaultRolePermissions.readValue();
        if (dataValue && dataValue.statusCode.isGood()) {
            return dataValue.value.value;
        }
    }
    return null;
}
function makeRoles(roleIds) {
    if (typeof roleIds === "number") {
        roleIds = [roleIds];
    }
    if (typeof roleIds === "string") {
        roleIds = roleIds.split(";").map((r) => (0, node_opcua_nodeid_1.resolveNodeId)("WellKnownRole_" + r));
    }
    return roleIds.map((r) => (0, node_opcua_nodeid_1.resolveNodeId)(r));
}
class SessionContext {
    static defaultContext = new SessionContext({});
    object;
    currentTime;
    continuationPoints = [];
    userIdentity;
    session;
    server;
    constructor(options) {
        options = options || {};
        this.session = options.session;
        this.object = options.object;
        this.server = options.server;
        this.currentTime = undefined;
    }
    /**
     * getCurrentUserRoles
     *
     * guest   => anonymous user (unauthenticated)
     * default => default authenticated user
     *
     */
    getCurrentUserRoles() {
        if (!this.session) {
            return []; // default context => no Session
        }
        (0, node_opcua_assert_1.assert)(this.session != null, "expecting a session");
        const userIdentityToken = this.session.userIdentityToken;
        if (!userIdentityToken) {
            return [];
        }
        const anonymous = makeRoles([WellKnownRoles.Anonymous]);
        const username = getUserName(userIdentityToken);
        if (username === "anonymous") {
            return anonymous;
        }
        if (!this.server || !this.server.userManager) {
            return anonymous;
        }
        (0, node_opcua_assert_1.assert)(this.server != null, "expecting a server");
        if (typeof this.server.userManager.getUserRoles !== "function") {
            return anonymous;
        }
        const rolesNodeId = this.server.userManager.getUserRoles(username);
        if (rolesNodeId.findIndex((r) => r.namespace === 0 && r.value === WellKnownRoles.AuthenticatedUser) < 0) {
            rolesNodeId.push((0, node_opcua_nodeid_1.resolveNodeId)(WellKnownRoles.AuthenticatedUser));
        }
        return rolesNodeId;
    }
    getApplicableRolePermissions(node) {
        if (!node.rolePermissions) {
            const namespace = node.namespace;
            const defaultUserRolePermissions = getDefaultUserRolePermissionsOnNamespace(namespace, this);
            return defaultUserRolePermissions;
        }
        return node.rolePermissions;
    }
    getPermissions(node) {
        const applicableRolePermissions = this.getApplicableRolePermissions(node);
        const roles = this.getCurrentUserRoles();
        if (roles.length === 0) {
            return node_opcua_data_model_1.allPermissions;
        }
        let orFlags = 0;
        for (const role of roles) {
            orFlags = orFlags | getPermissionForRole(applicableRolePermissions, role);
        }
        return orFlags;
    }
    getAccessRestrictions(node) {
        if (node.accessRestrictions === undefined) {
            const namespace = node.namespace;
            const accessRestrictions = getAccessRestrictionsOnNamespace(namespace, this);
            return accessRestrictions;
        }
        return node.accessRestrictions;
    }
    /**
     *
     * @param node
     * @returns true if the browse is denied (access is restricted)
     */
    isBrowseAccessRestricted(node) {
        if (this.checkPermission(node, node_opcua_types_1.PermissionType.Browse)) {
            return false; // can browse
        }
        return true; // browse restriction
    }
    /**
     *
     * @param node
     * @returns true  if the context is access restricted
     */
    isAccessRestricted(node) {
        const accessRestrictions = this.getAccessRestrictions(node);
        if (accessRestrictions === node_opcua_data_model_1.AccessRestrictionsFlag.None) {
            return false;
        }
        if (accessRestrictions & node_opcua_data_model_1.AccessRestrictionsFlag.SessionRequired) {
            if (!this.session) {
                return true;
            }
        }
        if (!this.session) {
            return false;
        }
        const securityMode = this.session?.channel?.securityMode;
        if (accessRestrictions & node_opcua_data_model_1.AccessRestrictionsFlag.SigningRequired) {
            if (securityMode !== node_opcua_types_1.MessageSecurityMode.Sign && securityMode !== node_opcua_types_1.MessageSecurityMode.SignAndEncrypt) {
                return true;
            }
        }
        if (accessRestrictions & node_opcua_data_model_1.AccessRestrictionsFlag.EncryptionRequired) {
            if (securityMode !== node_opcua_types_1.MessageSecurityMode.SignAndEncrypt) {
                return true;
            }
        }
        return false;
    }
    /**
     */
    checkPermission(node, requestedPermission) {
        const permissions = this.getPermissions(node);
        return (permissions & requestedPermission) === requestedPermission;
    }
    currentUserHasRole(role) {
        const currentUserRole = this.getCurrentUserRoles();
        const n = (0, node_opcua_nodeid_1.resolveNodeId)(role);
        return currentUserRole.findIndex((r) => (0, node_opcua_nodeid_1.sameNodeId)(r, n)) >= 0;
    }
}
exports.SessionContext = SessionContext;
//# sourceMappingURL=session_context.js.map