/**
 * @module node-opcua-address-space
 */
import { AccessRestrictionsFlag, PermissionFlag } from "node-opcua-data-model";
import { PreciseClock } from "node-opcua-date-time";
import { NodeId, NodeIdLike } from "node-opcua-nodeid";
import { PermissionType, RolePermissionType } from "node-opcua-types";
import { ISessionContext, UAObject, UAObjectType, BaseNode, ISessionBase } from "node-opcua-address-space-base";
export { RolePermissionType, RolePermissionTypeOptions, PermissionType } from "node-opcua-types";
/**
 *
 */
export declare enum WellKnownRoles {
    Anonymous = 15644,
    AuthenticatedUser = 15656,
    ConfigureAdmin = 15716,
    Engineer = 16036,
    Observer = 15668,
    Operator = 15680,
    SecurityAdmin = 15704,
    Supervisor = 15692
}
export declare enum WellKnownRolesNodeId {
    Anonymous = 15644,
    AuthenticatedUser = 15656,
    ConfigureAdmin = 15716,
    Engineer = 16036,
    Observer = 15668,
    Operator = 15680,
    SecurityAdmin = 15704,
    Supervisor = 15692
}
/**
 * OPC Unified Architecture, Part 3 13 Release 1.04
 * 4.8.2 Well Known Roles
 * All Servers should support the well-known Roles which are defined in Table 2. The NodeIds
 * for the well-known Roles are defined in Part 6.
 * Table 2 – Well-Known Roles
 * BrowseName           Suggested Permissions
 *
 * Anonymous            The Role has very limited access for use when a Session has anonymous credentials.
 * AuthenticatedUser    The Role has limited access for use when a Session has valid non-anonymous credentials
 *                      but has not been explicitly granted access to a Role.
 * Observer             The Role is allowed to browse, read live data, read historical data/events or subscribe to data/events.
 * Operator             The Role is allowed to browse, read live data, read historical data/events or subscribe to data/events.
 *                      In addition, the Session is allowed to write some live data and call some Methods.
 * Engineer             The Role is allowed to browse, read/write configuration data, read historical data/events,
 *                      call Methods or subscribe to data/events.
 * Supervisor           The Role is allowed to browse, read live data, read historical data/events, call Methods or
 *                      subscribe to data/events.
 * ConfigureAdmin       The Role is allowed to change the non-security related config
 * SecurityAdmin	    The Role is allowed to change security related settings.
 */
export type WellKnownRolesSemiColumnSeparated = string;
export interface IUserManager {
    /**
     * retrieve the roles of the given user
     *  @returns semicolon separated list of roles
     */
    getUserRoles?: (user: string) => NodeId[];
}
export interface IServerBase {
    userManager?: IUserManager;
}
export interface SessionContextOptions {
    session?: ISessionBase;
    object?: UAObject | UAObjectType;
    server?: IServerBase;
}
export declare function makeRoles(roleIds: NodeIdLike[] | string | WellKnownRoles): NodeId[];
export declare class SessionContext implements ISessionContext {
    static defaultContext: SessionContext;
    object: any;
    currentTime?: PreciseClock;
    continuationPoints: Buffer[];
    userIdentity?: string;
    readonly session?: ISessionBase;
    readonly server?: IServerBase;
    constructor(options?: SessionContextOptions);
    /**
     * getCurrentUserRoles
     *
     * guest   => anonymous user (unauthenticated)
     * default => default authenticated user
     *
     */
    getCurrentUserRoles(): NodeId[];
    getApplicableRolePermissions(node: BaseNode): RolePermissionType[] | null;
    getPermissions(node: BaseNode): PermissionFlag;
    getAccessRestrictions(node: BaseNode): AccessRestrictionsFlag;
    /**
     *
     * @param node
     * @returns true if the browse is denied (access is restricted)
     */
    isBrowseAccessRestricted(node: BaseNode): boolean;
    /**
     *
     * @param node
     * @returns true  if the context is access restricted
     */
    isAccessRestricted(node: BaseNode): boolean;
    /**
     */
    checkPermission(node: BaseNode, requestedPermission: PermissionType): boolean;
    currentUserHasRole(role: NodeIdLike): boolean;
}
