"use strict";
var __importDefault = (this && this.__importDefault) || function (mod) {
    return (mod && mod.__esModule) ? mod : { "default": mod };
};
Object.defineProperty(exports, "__esModule", { value: true });
exports.makeExtensionObjectInnerParser = makeExtensionObjectInnerParser;
exports.makeExtensionObjectParser = makeExtensionObjectParser;
exports.createXMLExtensionObjectDecodingTask = createXMLExtensionObjectDecodingTask;
const assert_1 = __importDefault(require("assert"));
const node_opcua_extension_object_1 = require("node-opcua-extension-object");
const node_opcua_nodeid_1 = require("node-opcua-nodeid");
const node_opcua_xml2json_1 = require("node-opcua-xml2json");
const decode_xml_extension_object_1 = require("../decode_xml_extension_object");
const node_opcua_data_model_1 = require("node-opcua-data-model");
const node_opcua_types_1 = require("node-opcua-types");
const localized_text_parser_1 = require("./localized_text_parser");
const node_opcua_basic_types_1 = require("node-opcua-basic-types");
const node_opcua_debug_1 = require("node-opcua-debug");
const node_opcua_data_access_1 = require("node-opcua-data-access");
const debugLog = (0, node_opcua_debug_1.make_debugLog)("ExtensionObjectParser");
const errorLog = (0, node_opcua_debug_1.make_debugLog)("ExtensionObjectParser");
const doDebug = false;
const makeArgumentParser = (_translateNodeId) => ({
    Argument: {
        init() {
            this.argument = new node_opcua_types_1.Argument({});
        },
        parser: {
            Name: {
                finish() {
                    this.parent.argument.name = this.text.trim();
                }
            },
            DataType: {
                parser: {
                    Identifier: {
                        finish() {
                            this.parent.parent.argument.dataType = _translateNodeId((0, node_opcua_nodeid_1.resolveNodeId)(this.text.trim()).toString());
                        }
                    }
                }
            },
            ValueRank: {
                finish() {
                    this.parent.argument.valueRank = parseInt(this.text.trim(), 10);
                }
            },
            ArrayDimensions: {
                finish() {
                    // xx  this.parent.argument.arrayDimensions =[];
                }
            },
            Description: {
                ...localized_text_parser_1.localizedText_parser.LocalizedText,
                finish() {
                    this.parent.argument.description = (0, node_opcua_data_model_1.coerceLocalizedText)({ ...this.localizedText });
                }
            }
        },
        finish() {
            // xx this.argument = new Argument(this.argument);
        }
    }
});
const Range_parser = {
    Range: {
        init() {
            this.range = new node_opcua_data_access_1.Range({});
        },
        parser: {
            Low: {
                finish() {
                    this.parent.range.low = parseFloat(this.text);
                }
            },
            High: {
                finish() {
                    this.parent.range.high = parseFloat(this.text);
                }
            }
        }
    }
};
const EUInformation_parser = {
    EUInformation: {
        init(name, attrs, parent, engine) {
            this.euInformation = new node_opcua_types_1.EUInformation({});
        },
        parser: {
            NamespaceUri: {
                finish() {
                    this.parent.euInformation.namespaceUri = this.text;
                }
            },
            UnitId: {
                finish() {
                    this.parent.euInformation.unitId = parseInt(this.text, 10);
                }
            },
            DisplayName: {
                ...localized_text_parser_1.localizedText_parser.LocalizedText,
                finish() {
                    this.parent.euInformation.displayName = (0, node_opcua_data_model_1.coerceLocalizedText)({ ...this.localizedText });
                }
            },
            Description: {
                ...localized_text_parser_1.localizedText_parser.LocalizedText,
                finish() {
                    this.parent.euInformation.description = (0, node_opcua_data_model_1.coerceLocalizedText)({ ...this.localizedText });
                }
            }
        },
        finish() {
            this.euInformation = new node_opcua_types_1.EUInformation(this.euInformation);
        }
    }
};
const enumValueType_parser = {
    EnumValueType: {
        init() {
            this.enumValueType = new node_opcua_types_1.EnumValueType({
                description: undefined,
                displayName: undefined,
                value: (0, node_opcua_basic_types_1.coerceInt64)(-1)
            });
        },
        parser: {
            Value: {
                // note: Value may be missing, in this case value shall be set to -1(64)
                finish() {
                    // Low part
                    this.parent.enumValueType.value = (0, node_opcua_basic_types_1.coerceInt64)(parseInt(this.text, 10));
                }
            },
            DisplayName: {
                ...localized_text_parser_1.localizedText_parser.LocalizedText,
                finish() {
                    this.parent.enumValueType.displayName = new node_opcua_data_model_1.LocalizedText({ ...this.localizedText });
                }
            },
            Description: {
                ...localized_text_parser_1.localizedText_parser.LocalizedText,
                finish() {
                    this.parent.enumValueType.description = new node_opcua_data_model_1.LocalizedText({ ...this.localizedText });
                }
            }
        },
        finish() {
            this.enumValueType = new node_opcua_types_1.EnumValueType(this.enumValueType);
        }
    }
};
function makeExtensionObjectInnerParser(translateNodeId, setExtensionObject, setExtensionObjectPojo) {
    const a = makeExtensionObjectParser(translateNodeId, setExtensionObject, setExtensionObjectPojo);
    return a.ExtensionObject;
}
;
function makeExtensionObjectParser(translateNodeId, setExtensionObject, setExtensionObjectPojo) {
    return {
        ExtensionObject: {
            init() {
                this.typeDefinitionId = node_opcua_nodeid_1.NodeId.nullNodeId;
            },
            parser: {
                TypeId: {
                    parser: {
                        Identifier: {
                            finish() {
                                const typeDefinitionId = this.text.trim();
                                this.parent.parent.typeDefinitionId = translateNodeId(typeDefinitionId);
                            }
                        }
                    }
                },
                // Body2: new FragmentClonerParser(),
                Body: {
                    parser: {
                        Argument: makeArgumentParser(translateNodeId).Argument,
                        EUInformation: EUInformation_parser.EUInformation,
                        EnumValueType: enumValueType_parser.EnumValueType,
                        Range: Range_parser.Range
                    },
                    startElement(elementName, attrs) {
                        if (!Object.prototype.hasOwnProperty.call(this.parser, elementName)) {
                            // treat it as a opaque XML bloc for the time being
                            // until we find the definition of this object, so we know how to interpret the fields
                            this._cloneFragment = new node_opcua_xml2json_1.InternalFragmentClonerReaderState();
                            this.engine._promote(this._cloneFragment, this.engine.currentLevel, elementName, attrs);
                        }
                    },
                    finish() {
                        const data = this.parent.parent;
                        const typeDefinitionId = this.parent.typeDefinitionId;
                        // typeDefinitionId is also the "Default XML" encoding nodeId !
                        switch (typeDefinitionId.toString()) {
                            case "i=7616": // EnumValueType
                            case "ns=0;i=7616":
                                const extensionObject = this.parser.EnumValueType.enumValueType;
                                (0, assert_1.default)(extensionObject !== null && typeof extensionObject === "object");
                                (0, assert_1.default)(extensionObject instanceof node_opcua_extension_object_1.ExtensionObject);
                                setExtensionObject(extensionObject, data);
                                break;
                            case "i=297": // Arguments
                            case "ns=0;i=297":
                                {
                                    const extensionObject = this.parser.Argument.argument;
                                    (0, assert_1.default)(extensionObject !== null && typeof extensionObject === "object");
                                    (0, assert_1.default)(extensionObject instanceof node_opcua_extension_object_1.ExtensionObject);
                                    setExtensionObject(extensionObject, data);
                                }
                                break;
                            case "i=888": // EUInformation
                            case "ns=0;i=888":
                                {
                                    const extensionObject = this.parser.EUInformation.euInformation;
                                    (0, assert_1.default)(extensionObject !== null && typeof extensionObject === "object");
                                    (0, assert_1.default)(extensionObject instanceof node_opcua_extension_object_1.ExtensionObject);
                                    setExtensionObject(extensionObject, data);
                                }
                                break;
                            case "i=885": // Range
                            case "ns=0;i=885":
                                {
                                    const extensionObject = this.parser.Range.range;
                                    (0, assert_1.default)(extensionObject !== null && typeof extensionObject === "object");
                                    (0, assert_1.default)(extensionObject instanceof node_opcua_extension_object_1.ExtensionObject);
                                    setExtensionObject(extensionObject, data);
                                }
                                break;
                            default: {
                                // istanbul ignore next
                                if (!this._cloneFragment) {
                                    // the XML file is probably not exposing standard UA extension object correctly.
                                    // this has been seen in some generated xml files using the dataType nodeId instead of the default encoding
                                    // nodeid
                                    errorLog("[NODE-OPCUA-E12] standard OPCUA Extension object from (namespace=0) has a invalid TypeId", typeDefinitionId.toString());
                                    break;
                                }
                                const bodyXML = this._cloneFragment.value;
                                this._cloneFragment.value = null;
                                // the "Default Xml" encoding  nodeId
                                const xmlEncodingNodeId = typeDefinitionId;
                                if (typeDefinitionId.isEmpty()) {
                                    debugLog("xmlEncodingNodeId is empty for " + typeDefinitionId.toString());
                                    break;
                                }
                                setExtensionObjectPojo(xmlEncodingNodeId, bodyXML, data);
                                break;
                            }
                        }
                    }
                }
            },
            finish() {
                /* empty */
            }
        }
    };
}
;
function createXMLExtensionObjectDecodingTask(translateNodeId, xmlEncodingNodeId, bodyXML, withDecoded) {
    const capturedXmlBody = bodyXML;
    const task = async (addressSpace2) => {
        const extensionObject = (0, decode_xml_extension_object_1.decodeXmlExtensionObject)(addressSpace2, xmlEncodingNodeId, capturedXmlBody, translateNodeId);
        withDecoded(extensionObject);
    };
    return task;
}
//# sourceMappingURL=extension_object_parser.js.map