"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.makeXmlExtensionObjectReader = makeXmlExtensionObjectReader;
const node_opcua_basic_types_1 = require("node-opcua-basic-types");
const node_opcua_debug_1 = require("node-opcua-debug");
const node_opcua_nodeid_1 = require("node-opcua-nodeid");
const node_opcua_types_1 = require("node-opcua-types");
const node_opcua_utils_1 = require("node-opcua-utils");
const node_opcua_variant_1 = require("node-opcua-variant");
const node_opcua_xml2json_1 = require("node-opcua-xml2json");
const localized_text_parser_1 = require("./parsers/localized_text_parser");
const qualified_name_parser_1 = require("./parsers/qualified_name_parser");
const variant_parser_1 = require("./parsers/variant_parser");
const warningLog = (0, node_opcua_debug_1.make_warningLog)(__filename);
const debugLog = (0, node_opcua_debug_1.make_debugLog)(__filename);
function clamp(value, minValue, maxValue) {
    if (value < minValue) {
        warningLog(`invalid value range : ${value} < ${minValue} but should be [${minValue} , ${maxValue}]`);
        return minValue;
    }
    if (value > maxValue) {
        warningLog(`invalid value range : ${value} > ${maxValue} but should be [${minValue} , ${maxValue}]`);
        return maxValue;
    }
    return value;
}
const partials = {
    LocalizedText: localized_text_parser_1.localizedText_parser.LocalizedText,
    QualifiedName: (0, qualified_name_parser_1.makeQualifiedNameParser)((nodeId) => (0, node_opcua_nodeid_1.coerceNodeId)(nodeId)).QualifiedName,
    String: {
        finish() {
            this.value = this.text;
        }
    },
    Guid: {
        parser: {
            String: {
                finish() {
                    this.parent.value = this.text;
                }
            }
        }
    },
    Boolean: {
        finish() {
            this.value = this.text.toLowerCase() === "true" ? true : false;
        }
    },
    ByteString: {
        init(name, attrs, parent, engine) {
            this.value = null;
        },
        finish() {
            const base64text = this.text;
            const byteString = Buffer.from(base64text, "base64");
            this.value = byteString;
        }
    },
    Float: {
        finish() {
            this.value = parseFloat(this.text);
        }
    },
    Double: {
        finish() {
            this.value = parseFloat(this.text);
        }
    },
    Byte: {
        finish() {
            this.value = clamp(parseInt(this.text, 10), 0, 255);
        }
    },
    SByte: {
        finish() {
            this.value = clamp(parseInt(this.text, 10), -128, 127);
        }
    },
    Int8: {
        finish() {
            this.value = clamp(parseInt(this.text, 10), -128, 127);
        }
    },
    Int16: {
        finish() {
            this.value = clamp(parseInt(this.text, 10), -32768, 32767);
        }
    },
    Int32: {
        finish() {
            this.value = clamp(parseInt(this.text, 10), -2147483648, 2147483647);
        }
    },
    Int64: {
        finish() {
            this.value = (0, node_opcua_basic_types_1.coerceInt64)(parseInt(this.text, 10));
        }
    },
    UInt8: {
        finish() {
            this.value = clamp(parseInt(this.text, 10), 0, 255);
        }
    },
    UInt16: {
        finish() {
            this.value = clamp(parseInt(this.text, 10), 0, 65535);
        }
    },
    UInt32: {
        finish() {
            this.value = clamp(parseInt(this.text, 10), 0, 4294967295);
        }
    },
    UInt64: {
        finish() {
            this.value = (0, node_opcua_basic_types_1.coerceUInt64)(parseInt(this.text, 10));
        }
    },
    DateTime: {
        finish() {
            // to do check Local or GMT
            this.value = new Date(this.text);
        }
    },
    Variant: {
        finish() {
            /** to do */
            warningLog(" Missing  Implemntation contact sterfive.com!");
        }
    },
    NodeId: {
        finish() {
            // to do check Local or GMT
            this.value = (0, node_opcua_nodeid_1.coerceNodeId)(this.text);
        }
    }
};
function _clone(a) {
    if (typeof a === "string" || typeof a === "number" || typeof a === "boolean") {
        return a;
    }
    if (a instanceof Buffer) {
        return Buffer.from(a);
    }
    if (a instanceof Date) {
        return new Date(a);
    }
    if (a instanceof Array) {
        return a.map((x) => _clone(x));
    }
    return { ...a };
}
function _makeTypeReader(dataTypeNodeId1, definitionMap, readerMap, translateNodeId) {
    const n = dataTypeNodeId1;
    if (n.identifierType === node_opcua_nodeid_1.NodeIdType.NUMERIC && n.namespace === 0 && n.value === 0) {
        // a generic Extension Object
        return { name: "Variant", reader: partials["Variant"] };
    }
    if (n.namespace === 0 &&
        n.identifierType === node_opcua_nodeid_1.NodeIdType.NUMERIC &&
        n.value < node_opcua_variant_1.DataType.ExtensionObject) {
        const name = node_opcua_variant_1.DataType[n.value];
        const reader = partials[name];
        return { name, reader };
    }
    const { name, definition } = definitionMap.findDefinition(n);
    const dataTypeName = name;
    let reader = readerMap.get(dataTypeName);
    if (reader) {
        return { name, reader: reader.parser };
    }
    reader = {
        finish() {
            /** empty  */
        },
        parser: {
        /** empty  */
        }
    };
    if (definition instanceof node_opcua_types_1.StructureDefinition) {
        for (const field of definition.fields || []) {
            const typeReader = _makeTypeReader(field.dataType, definitionMap, readerMap, translateNodeId);
            const fieldParser = typeReader.reader;
            const fieldTypename = typeReader.name;
            // istanbul ignore next
            if (!fieldParser) {
                throw new Error(" Cannot find reader for dataType " + field.dataType + " fieldTypename=" + fieldTypename);
            }
            if (field.valueRank === undefined || field.valueRank === -1) {
                // scalar
                const parser = fieldParser;
                if (!parser) {
                    throw new Error("??? " + field.dataType + "  " + field.name);
                }
                reader.parser[field.name] = {
                    parser: fieldParser.parser,
                    // endElement: fieldReader.endElement,
                    finish() {
                        const elName = (0, node_opcua_utils_1.lowerFirstLetter)(field.name);
                        if (fieldParser.finish) {
                            fieldParser.finish.call(this);
                        }
                        else {
                            debugLog("xxx check " + fieldTypename);
                        }
                        this.parent.value = this.parent.value || Object.create(null);
                        this.parent.value[elName] = _clone(this.value);
                    }
                };
            }
            else if (field.valueRank === 1) {
                const listReader = {
                    init() {
                        this.value = [];
                    },
                    parser: { /** empty */},
                    finish() {
                        const elName = (0, node_opcua_utils_1.lowerFirstLetter)(this.name);
                        this.parent.value = this.parent.value || Object.create(null);
                        this.parent.value[elName] = this.value;
                        this.value = undefined;
                    },
                    startElement(name, attrs) {
                        // empty
                    },
                    endElement(element) {
                        this.value.push(_clone(this.parser[element].value));
                    }
                };
                listReader.parser[fieldTypename] = fieldParser;
                reader.parser[field.name] = listReader;
            }
            else {
                throw new Error("Unsupported ValueRank !");
            }
        }
        readerMap.set(dataTypeName, reader);
        return { name, reader };
    }
    else if (definition instanceof node_opcua_types_1.EnumDefinition) {
        const turnToInt = (value) => {
            // Green_100
            return parseInt(value.split("_")[1], 10);
        };
        return {
            name,
            reader: {
                finish() {
                    this.value = turnToInt(this.text);
                }
            }
        };
    }
    else if (definition?.dataType == node_opcua_variant_1.DataType.Variant) {
        let variantOptions = Object.create(null);
        const variantReader = (0, variant_parser_1.makeVariantReader)((self, data) => variantOptions = data, 
        /*setDeferredValue: */ (self, data, deferedTask) => {
            // to do
        }, 
        /* postExtensionObjectDecoding:*/ (task) => {
            // to do
        }, translateNodeId);
        return {
            name,
            reader: {
                init(name, attrs, parent, engine) {
                    this.obj = {};
                },
                ...variantReader,
                finish() {
                    this.value = new node_opcua_variant_1.Variant(variantOptions);
                }
            }
        };
    }
    else {
        // basic datatype
        const typeName = node_opcua_variant_1.DataType[definition.dataType];
        const reader = partials[typeName];
        // istanbul ignore next
        if (!reader) {
            throw new Error("missing parse for " + typeName);
        }
        return { name, reader };
    }
}
function makeXmlExtensionObjectReader(dataTypeNodeId, definitionMap, readerMap, translateNodeId) {
    const { name, definition } = definitionMap.findDefinition(dataTypeNodeId);
    // istanbul ignore next
    if (!(definition instanceof node_opcua_types_1.StructureDefinition)) {
        throw new Error("Expecting StructureDefinition");
    }
    //
    const reader1 = {
        parser: {},
        endElement() {
            this._pojo = this.parser[name].value;
        }
    };
    const { reader } = _makeTypeReader(dataTypeNodeId, definitionMap, readerMap, translateNodeId);
    reader1.parser[name] = reader;
    return new node_opcua_xml2json_1.ReaderState(reader1);
}
//# sourceMappingURL=make_xml_extension_object_parser.js.map