"use strict";
var __importDefault = (this && this.__importDefault) || function (mod) {
    return (mod && mod.__esModule) ? mod : { "default": mod };
};
Object.defineProperty(exports, "__esModule", { value: true });
exports.NodeSetLoader = void 0;
/* eslint-disable max-statements */
/**
 * @module node-opcua-address-space
 */
const util_1 = require("util");
const chalk_1 = __importDefault(require("chalk"));
const node_opcua_basic_types_1 = require("node-opcua-basic-types");
const node_opcua_assert_1 = require("node-opcua-assert");
const node_opcua_basic_types_2 = require("node-opcua-basic-types");
const node_opcua_data_model_1 = require("node-opcua-data-model");
const node_opcua_debug_1 = require("node-opcua-debug");
const node_opcua_factory_1 = require("node-opcua-factory");
const node_opcua_nodeid_1 = require("node-opcua-nodeid");
const node_opcua_variant_1 = require("node-opcua-variant");
const node_opcua_xml2json_1 = require("node-opcua-xml2json");
const semver_1 = __importDefault(require("semver"));
const namespace_post_step_1 = require("./namespace_post_step");
const ensure_datatype_extracted_1 = require("./ensure_datatype_extracted");
const make_semver_compatible_1 = require("./make_semver_compatible");
const variant_parser_1 = require("./parsers/variant_parser");
const doDebug = (0, node_opcua_debug_1.checkDebugFlag)(__filename);
const debugLog = (0, node_opcua_debug_1.make_debugLog)(__filename);
const errorLog = (0, node_opcua_debug_1.make_errorLog)(__filename);
function __make_back_references(namespace) {
    const namespaceP = namespace;
    for (const node of namespaceP.nodeIterator()) {
        node.propagate_back_references();
    }
    for (const node of namespaceP.nodeIterator()) {
        node.install_extra_properties();
    }
}
function make_back_references(addressSpace) {
    const addressSpacePrivate = addressSpace;
    addressSpacePrivate.suspendBackReference = false;
    addressSpace.getNamespaceArray().map(__make_back_references);
}
function stringToUInt32Array(str) {
    const array = str ? str.split(",").map((value) => parseInt(value, 10)) : null;
    return array;
}
function convertAccessLevel(accessLevel) {
    const accessLevelN = parseInt(accessLevel || "1", 10); // CurrentRead if not specified
    return (0, node_opcua_data_model_1.makeAccessLevelFlag)(accessLevelN);
}
function makeDefaultVariant(addressSpace, dataTypeNode, valueRank, arrayDimensions) {
    let variant = { dataType: node_opcua_variant_1.DataType.Null };
    const nodeDataType = addressSpace.findNode(dataTypeNode);
    if (nodeDataType && nodeDataType.basicDataType) {
        const basicDataType = nodeDataType.basicDataType;
        if (basicDataType === node_opcua_variant_1.DataType.Variant) {
            /// we don't now what is the variant
            return undefined;
        }
        if (basicDataType === node_opcua_variant_1.DataType.ExtensionObject) {
            return { dataType: node_opcua_variant_1.DataType.ExtensionObject, value: null };
        }
        const builtInType = (0, node_opcua_factory_1.getBuiltInType)(node_opcua_variant_1.DataType[basicDataType]);
        if (builtInType === undefined || builtInType === null) {
            errorLog("Cannot find builtInType for ", basicDataType);
            return { dataType: node_opcua_variant_1.DataType.Null };
        }
        const dv = builtInType.defaultValue;
        const value = typeof dv === "function" ? dv() : dv;
        let arrayType;
        /*
         *  * n > 1                     : the Value is an array with the specified number of dimensions.
         *  * OneDimension (1):           The value is an array with one dimension.
         *  * OneOrMoreDimensions (0):    The value is an array with one or more dimensions.
         *  * Scalar (-1):                The value is not an array.
         *  * Any (-2):                   The value can be a scalar or an array with any number of dimensions.
         *  * ScalarOrOneDimension (-3):  The value can be a scalar or a one dimensional array.
         */
        switch (valueRank) {
            case -3: //  ScalarOrOneDimension (-3):
            case -2: // any
            case -1:
                arrayType = node_opcua_variant_1.VariantArrayType.Scalar;
                variant = { dataType: basicDataType, value, arrayType };
                break;
            case 0: // one or more dimension
            case 1: // one dimension
                arrayType = node_opcua_variant_1.VariantArrayType.Array;
                variant = { dataType: basicDataType, value: [], arrayType };
                break;
            default:
                arrayType = node_opcua_variant_1.VariantArrayType.Matrix;
                variant = { dataType: basicDataType, value: [], arrayType, dimensions: arrayDimensions };
                break;
        }
    }
    return variant;
}
function makeNodeSetParserEngine(addressSpace, options) {
    const addressSpace1 = addressSpace;
    addressSpace1.suspendBackReference = true;
    options.loadDeprecatedNodes = options.loadDeprecatedNodes === undefined ? true : options.loadDeprecatedNodes;
    options.loadDraftNodes = options.loadDraftNodes || false;
    const postTasks = [];
    const postTasks0_InitializeVariable = [];
    const postTasks0_DecodePojoString = [];
    const postTasks1_InitializeVariable = [];
    const postTasks2_AssignedExtensionObjectToDataValue = [];
    let aliasMap = new Map();
    /**
     * @param aliasName
     */
    function addAlias(aliasName, nodeIdInXmlContext) {
        (0, node_opcua_assert_1.assert)(typeof nodeIdInXmlContext === "string");
        const nodeId = _translateNodeId(nodeIdInXmlContext);
        (0, node_opcua_assert_1.assert)(nodeId instanceof node_opcua_nodeid_1.NodeId);
        aliasMap.set(aliasName, nodeId);
        addressSpace1.getNamespace(nodeId.namespace).addAlias(aliasName, nodeId);
    }
    let namespaceUriTranslationMap = new Map();
    let namespaceCounter = 0;
    let foundNamespaceMap = new Map();
    let models = [];
    let performedCalled = false;
    function _reset_namespace_translation() {
        debugLog("_reset_namespace_translation");
        namespaceUriTranslationMap.clear();
        foundNamespaceMap.clear();
        namespaceCounter = 0;
        aliasMap.clear();
        models = [];
        performedCalled = false;
    }
    function _translateNamespaceIndex(innerIndex) {
        const namespaceIndex = namespaceUriTranslationMap.get(innerIndex);
        // istanbul ignore next
        if (namespaceIndex === undefined) {
            errorLog("Error; namespace_uri_translation", namespaceUriTranslationMap.entries());
            throw new Error("_translateNamespaceIndex() ! Cannot find namespace definition for index " + innerIndex);
        }
        return namespaceIndex;
    }
    function _internal_addReferenceType(params) {
        // istanbul ignore next
        if (!(params.nodeId instanceof node_opcua_nodeid_1.NodeId)) {
            throw new Error("invalid param");
        } // already translated
        const namespace = addressSpace1.getNamespace(params.nodeId.namespace);
        namespace.addReferenceType(params);
    }
    function _internal_createNode(params) {
        // istanbul ignore next
        if (!(params.nodeId instanceof node_opcua_nodeid_1.NodeId)) {
            throw new Error("invalid param expecting a valid nodeId");
        } // already translated
        const namespace = addressSpace1.getNamespace(params.nodeId.namespace);
        return namespace.internalCreateNode(params);
    }
    function _register_namespace_uri_in_translation_table(namespaceUri) {
        if (foundNamespaceMap.has(namespaceUri)) {
            return;
        }
        const namespace = addressSpace1.getNamespace(namespaceUri);
        // istanbul ignore next
        if (!namespace) {
            throw new Error("cannot find namespace for " +
                namespaceUri +
                "\nplease make sure to initialize your address space with the corresponding nodeset files");
        }
        foundNamespaceMap.set(namespaceUri, namespace);
        const index_in_xml = namespaceCounter;
        namespaceCounter++;
        namespaceUriTranslationMap.set(index_in_xml, namespace.index);
        doDebug &&
            debugLog(" _register_namespace_uri = ", namespaceUri, "index in Xml=", index_in_xml, " index in addressSpace", namespace.index);
    }
    function _add_namespace(model) {
        if (model.requiredModels.length > 0) {
            // check that required models exist already in the address space
            for (const requiredModel of model.requiredModels) {
                const existingNamespace = addressSpace1.getNamespace(requiredModel.modelUri);
                // istanbul ignore next
                if (!existingNamespace) {
                    errorLog("Please ensure that the required namespace", requiredModel.modelUri, "is loaded first when loading", model.modelUri);
                    throw new Error("LoadNodeSet : Cannot find namespace for " + requiredModel.modelUri);
                }
                /**
                 *  from https://reference.opcfoundation.org/Core/docs/Part6/F.2/
                 *  The version of the model defined in the UANodeSet.
                 *  This is a human-readable string and not intended for programmatic comparisons.
                 */
                const isLowerVersion = (existingVersion, requiredVersion) => {
                    const existingSemver = (0, make_semver_compatible_1.makeSemverCompatible)(existingVersion);
                    const requiredSemver = (0, make_semver_compatible_1.makeSemverCompatible)(requiredVersion);
                    return semver_1.default.lt(existingSemver, requiredSemver);
                };
                if (isLowerVersion(existingNamespace.version, requiredModel.version)) {
                    errorLog("Expecting ", requiredModel.modelUri, " with version to be at least", requiredModel.version, " but namespace version is ", existingNamespace.version);
                }
                if (existingNamespace.publicationDate.getTime() < requiredModel.publicationDate.getTime()) {
                    errorLog("Expecting ", requiredModel.modelUri, " with publicationDate at least ", requiredModel.publicationDate.toUTCString(), " but namespace publicationDate is ", existingNamespace.publicationDate.toUTCString());
                }
            }
        }
        let namespace;
        // Model must not be already registered
        const existingNamespace = addressSpace1.getNamespace(model.modelUri);
        if (existingNamespace) {
            namespace = existingNamespace;
        }
        else {
            namespace = addressSpace1.registerNamespace(model.modelUri);
            namespace.setRequiredModels(model.requiredModels);
        }
        namespace.version = model.version;
        namespace.publicationDate = model.publicationDate || namespace.publicationDate;
        return namespace;
    }
    const reg = /ns=([0-9]+);(.*)/;
    function _translateNodeId(nodeId) {
        if (aliasMap.has(nodeId)) {
            // note alias are already translated to the right namespaces
            const aliasedNodeId = aliasMap.get(nodeId);
            return aliasedNodeId;
        }
        const m = nodeId.match(reg);
        if (m) {
            const namespaceIndex = _translateNamespaceIndex(parseInt(m[1], 10));
            nodeId = "ns=" + namespaceIndex + ";" + m[2];
        }
        return (0, node_opcua_nodeid_1.resolveNodeId)(nodeId);
    }
    function _translateReferenceType(refType) {
        return _translateNodeId(refType);
    }
    /**
     * convert a nodeId
     *
     *
     * @example
     *    convertToNodeId("String") => resolve alias
     *    convertToNodeId("i=58")   => resolve to nodeId in namespace 0
     *    convertToNodeId("ns=1;i=100") => convert namespace from xml namespace
     *                                      table to corresponding namespace in addressSpace
     */
    function convertToNodeId(nodeIdLike) {
        // treat alias
        if (!nodeIdLike) {
            return null;
        }
        const nodeId = _translateNodeId(nodeIdLike);
        return addressSpace1.resolveNodeId(nodeId);
    }
    function convertQualifiedName(qualifiedName) {
        const qn = (0, node_opcua_data_model_1.stringToQualifiedName)(qualifiedName);
        // Xx if (qn.namespaceIndex > 0) {
        qn.namespaceIndex = _translateNamespaceIndex(qn.namespaceIndex);
        // Xx }
        return qn;
    }
    const state_Alias = {
        finish() {
            addAlias(this.attrs.Alias, this.text);
        }
    };
    const references_parser = {
        init() {
            this.parent.obj.references = [];
            this.array = this.parent.obj.references;
        },
        parser: {
            Reference: {
                finish() {
                    this.parent.array.push({
                        isForward: this.attrs.IsForward === undefined ? true : this.attrs.IsForward === "false" ? false : true,
                        nodeId: convertToNodeId(this.text),
                        referenceType: _translateReferenceType(this.attrs.ReferenceType)
                    });
                }
            }
        }
    };
    // #region UAObject
    const state_UAObject = {
        init(name, attrs) {
            _perform();
            this.obj = {
                nodeClass: node_opcua_data_model_1.NodeClass.Object,
                isAbstract: (0, node_opcua_basic_types_1.coerceBoolean)(attrs.IsAbstract),
                nodeId: convertToNodeId(attrs.NodeId) || null,
                browseName: convertQualifiedName(attrs.BrowseName),
                eventNotifier: (0, node_opcua_basic_types_1.coerceByte)(attrs.EventNotifier) || 0,
                symbolicName: attrs.SymbolicName || null
            };
            this.isDraft = attrs.ReleaseStatus === "Draft";
            this.isDeprecated = attrs.ReleaseStatus === "Deprecated";
        },
        finish() {
            if (canIngore({ isDraft: this.isDraft, isDeprecated: this.isDeprecated }, this.obj)) {
                return;
            }
            _internal_createNode(this.obj);
        },
        parser: {
            DisplayName: {
                finish() {
                    this.parent.obj.displayName = this.text;
                }
            },
            Description: {
                finish() {
                    this.parent.obj.description = this.text;
                }
            },
            References: references_parser
        }
    };
    // #endregion
    // #region UAObjectType
    const state_UAObjectType = {
        init(name, attrs) {
            _perform();
            this.obj = {
                nodeClass: node_opcua_data_model_1.NodeClass.ObjectType,
                isAbstract: (0, node_opcua_basic_types_1.coerceBoolean)(attrs.IsAbstract),
                nodeId: convertToNodeId(attrs.NodeId) || null,
                browseName: convertQualifiedName(attrs.BrowseName),
                eventNotifier: (0, node_opcua_basic_types_1.coerceByte)(attrs.EventNotifier) || 0
            };
        },
        finish() {
            _internal_createNode(this.obj);
        },
        parser: {
            DisplayName: {
                finish() {
                    this.parent.obj.displayName = this.text;
                }
            },
            Description: {
                finish() {
                    this.parent.obj.description = this.text;
                }
            },
            References: references_parser
        }
    };
    // #endregion
    // #region UAReferenceType
    const state_UAReferenceType = {
        init(name, attrs) {
            _perform();
            this.obj = {
                nodeClass: node_opcua_data_model_1.NodeClass.ReferenceType,
                isAbstract: (0, node_opcua_basic_types_1.coerceBoolean)(attrs.IsAbstract),
                nodeId: convertToNodeId(attrs.NodeId) || null,
                browseName: convertQualifiedName(attrs.BrowseName)
            };
        },
        finish() {
            _internal_addReferenceType(this.obj);
        },
        parser: {
            DisplayName: {
                finish() {
                    this.parent.obj.displayName = this.text;
                }
            },
            Description: {
                finish() {
                    this.parent.obj.description = this.text;
                }
            },
            InverseName: {
                finish() {
                    this.parent.obj.inverseName = this.text;
                }
            },
            References: references_parser
        }
    };
    // #endregion
    // #region UADataType
    const pendingSimpleTypeToRegister = [];
    const state_UADataType = {
        init(name, attrs) {
            _perform();
            this.obj = {
                nodeClass: node_opcua_data_model_1.NodeClass.DataType,
                isAbstract: (0, node_opcua_basic_types_1.coerceBoolean)(attrs.IsAbstract) || false,
                nodeId: convertToNodeId(attrs.NodeId) || null,
                browseName: convertQualifiedName(attrs.BrowseName),
                displayName: "",
                description: "",
                symbolicName: attrs.SymbolicName
            };
            this.isDraft = attrs.ReleaseStatus === "Draft";
            this.isDeprecated = attrs.ReleaseStatus === "Deprecated";
            this.definitionFields = [];
        },
        finish() {
            if (canIngore({ isDraft: this.isDraft, isDeprecated: this.isDeprecated }, this.obj)) {
                return;
            }
            const definitionFields = this.definitionFields;
            // replace DataType with nodeId, and description to LocalizedText
            definitionFields.map((x) => {
                if (x.description) {
                    x.description = { text: x.description };
                }
                if (x.displayName) {
                    x.displayName = { text: x.displayName };
                }
                if (x.dataType) {
                    x.dataType = convertToNodeId(x.dataType);
                }
                return x;
            });
            this.obj.partialDefinition = definitionFields;
            let capturedDataTypeNode = _internal_createNode(this.obj);
            const processBasicDataType = async (addressSpace2) => {
                const definitionName = capturedDataTypeNode.browseName.name;
                const isStructure = capturedDataTypeNode.isStructure();
                const isEnumeration = capturedDataTypeNode.isEnumeration();
                if (!isEnumeration && !isStructure && capturedDataTypeNode.nodeId.namespace !== 0) {
                    // add a custom basic type that is not a structure nor a enumeration
                    pendingSimpleTypeToRegister.push({ name: definitionName, dataTypeNodeId: capturedDataTypeNode.nodeId });
                }
                capturedDataTypeNode = undefined;
            };
            postTasks.push(processBasicDataType);
        },
        parser: {
            DisplayName: {
                finish() {
                    this.parent.obj.displayName = this.text;
                }
            },
            Description: {
                finish() {
                    this.parent.obj.description = this.text;
                }
            },
            References: references_parser,
            Definition: node_opcua_xml2json_1._definitionParser
        }
    };
    // #endregion
    const canIngore = ({ isDraft, isDeprecated }, node) => {
        if (isDraft && !options.loadDraftNodes) {
            debugLog("Ignoring Draft            =", node_opcua_data_model_1.NodeClass[node.nodeClass], node.browseName.toString());
            return true;
        }
        if (isDeprecated && !options.loadDeprecatedNodes) {
            debugLog("Ignoring Deprecate        =", node_opcua_data_model_1.NodeClass[node.nodeClass], node.browseName.toString());
            return true;
        }
        return false;
    };
    function fixExtensionObjectAndArray(obj, deferred) {
        // let's create the mechanism that postpone the assignment of the extension object
        let capturedNode = obj;
        const task = async (addressSpace2) => {
            const extensionObjOrArray = deferred();
            let nodeId = capturedNode.nodeId;
            (0, node_opcua_assert_1.assert)(nodeId, "expecting a nodeid");
            const node = addressSpace2.findNode(nodeId);
            if (node.nodeClass === node_opcua_data_model_1.NodeClass.Variable) {
                const v = node;
                (0, node_opcua_assert_1.assert)(v.getBasicDataType() === node_opcua_variant_1.DataType.ExtensionObject, "expecting an extension object");
                if (extensionObjOrArray != null) {
                    v.bindExtensionObject(extensionObjOrArray, { createMissingProp: false });
                }
            }
            else if (node.nodeClass === node_opcua_data_model_1.NodeClass.VariableType) {
                const v = node;
                v /*fix me*/.value.value = extensionObjOrArray;
            }
        };
        postTasks2_AssignedExtensionObjectToDataValue.push(task);
    }
    const state_UAVariable = {
        init(name, attrs) {
            _perform();
            const valueRank = attrs.ValueRank === undefined ? -1 : (0, node_opcua_basic_types_1.coerceInt32)(attrs.ValueRank);
            const accessLevel = convertAccessLevel(attrs.AccessLevel);
            const nodeId = convertToNodeId(attrs.NodeId);
            this.obj = {
                value: undefined, //  { dataType: DataType.Null },
                nodeClass: node_opcua_data_model_1.NodeClass.Variable,
                browseName: convertQualifiedName(attrs.BrowseName),
                parentNodeId: convertToNodeId(attrs.ParentNodeId),
                dataType: convertToNodeId(attrs.DataType),
                //description: null,
                // displayName: coerceLocalizedText(attrs.BrowseName)!,
                valueRank,
                arrayDimensions: valueRank <= 0 ? null : stringToUInt32Array(attrs.ArrayDimensions),
                minimumSamplingInterval: attrs.MinimumSamplingInterval ? parseInt(attrs.MinimumSamplingInterval, 10) : 0,
                historizing: false,
                nodeId,
                accessLevel: accessLevel,
                userAccessLevel: accessLevel // convertAccessLevel(attrs.UserAccessLevel || attrs.AccessLevel);
            };
            this.isDraft = attrs.ReleaseStatus === "Draft" || false;
            this.isDeprecated = attrs.ReleaseStatus === "Deprecated" || false;
        },
        finish() {
            if (canIngore({ isDraft: this.isDraft, isDeprecated: this.isDeprecated }, this.obj)) {
                return;
            }
            /*
            // set default value based on obj data Type
            if (this.obj.value === undefined) {
                const dataTypeNode = this.obj.dataType;
                const valueRank = this.obj.valueRank;
                this.obj.value = makeDefaultVariant(addressSpace, dataTypeNode, valueRank);
            }
            */
            // eslint-disable-next-line prefer-const
            let capturedVariable;
            if (this.obj.value && this.obj.value.dataType !== node_opcua_variant_1.DataType.Null) {
                let capturedValue = this.obj.value;
                const task = async (addressSpace2) => {
                    if (false && doDebug) {
                        debugLog("1 setting value to ", capturedVariable.nodeId.toString(), new node_opcua_variant_1.Variant(capturedValue).toString());
                    }
                    capturedVariable.setValueFromSource(capturedValue);
                    capturedValue = undefined;
                    capturedVariable = undefined;
                };
                if (capturedValue.dataType !== node_opcua_variant_1.DataType.ExtensionObject) {
                    postTasks0_InitializeVariable.push(task);
                }
                else {
                    // do them later when every thing is created
                    postTasks1_InitializeVariable.push(task);
                }
            }
            else {
                const task = async (addressSpace2) => {
                    const dataTypeNode = capturedVariable.dataType;
                    const valueRank = capturedVariable.valueRank;
                    const arrayDimensions = capturedVariable.arrayDimensions;
                    const value = makeDefaultVariant(addressSpace, dataTypeNode, valueRank, arrayDimensions);
                    if (value) {
                        if (false && doDebug) {
                            debugLog("2 setting value to ", capturedVariable.nodeId.toString(), value);
                        }
                        if (value.dataType === node_opcua_variant_1.DataType.Null) {
                            capturedVariable.setValueFromSource(value, node_opcua_basic_types_2.StatusCodes.BadWaitingForInitialData);
                        }
                        else {
                            capturedVariable.setValueFromSource(value, node_opcua_basic_types_2.StatusCodes.Good);
                        }
                    }
                    capturedVariable = undefined;
                };
                postTasks0_InitializeVariable.push(task);
            }
            this.obj.value = Object.create(null);
            capturedVariable = _internal_createNode(this.obj);
        },
        parser: {
            DisplayName: {
                finish() {
                    this.parent.obj.displayName = (0, node_opcua_data_model_1.coerceLocalizedText)(this.text);
                }
            },
            Description: {
                finish() {
                    this.parent.obj.description = (0, node_opcua_data_model_1.coerceLocalizedText)(this.text);
                }
            },
            References: references_parser,
            Value: (0, variant_parser_1.makeVariantReader)((self, data) => {
                self.parent.obj.value = data;
            }, (self, data, deferredTask) => {
                self.parent.obj.value = data;
                const capturedVariable = { nodeId: self.parent.obj.nodeId };
                fixExtensionObjectAndArray(capturedVariable, deferredTask);
            }, (task) => {
                postTasks0_DecodePojoString.push(task);
            }, _translateNodeId)
        }
    };
    const state_UAVariableType = {
        init(name, attrs) {
            _perform();
            const valueRank = (0, node_opcua_basic_types_1.coerceInt32)(attrs.ValueRank) || -1;
            this.obj = {
                value: undefined, // { dataType: DataType.Null },
                isAbstract: (0, node_opcua_basic_types_1.coerceBoolean)(attrs.IsAbstract),
                nodeClass: node_opcua_data_model_1.NodeClass.VariableType,
                browseName: convertQualifiedName(attrs.BrowseName),
                parentNodeId: convertToNodeId(attrs.ParentNodeId) || null,
                dataType: convertToNodeId(attrs.DataType) || null,
                valueRank,
                arrayDimensions: valueRank <= 0 ? null : stringToUInt32Array(attrs.ArrayDimensions),
                minimumSamplingInterval: attrs.MinimumSamplingInterval ? parseInt(attrs.MinimumSamplingInterval, 10) : 0,
                historizing: false,
                nodeId: convertToNodeId(attrs.NodeId) || null
            };
            this.isDraft = attrs.ReleaseStatus === "Draft";
            this.isDeprecated = attrs.ReleaseStatus === "Deprecated";
        },
        finish() {
            if (canIngore({ isDraft: this.isDraft, isDeprecated: this.isDeprecated }, this.obj)) {
                return;
            }
            _internal_createNode(this.obj);
        },
        parser: {
            DisplayName: {
                finish() {
                    this.parent.obj.displayName = (0, node_opcua_data_model_1.coerceLocalizedText)(this.text || "");
                }
            },
            Description: {
                finish() {
                    this.parent.obj.description = (0, node_opcua_data_model_1.coerceLocalizedText)(this.text || "");
                }
            },
            References: references_parser,
            Value: (0, variant_parser_1.makeVariantReader)((self, data) => {
                self.parent.obj.value = data;
            }, (self, data, deferredTask) => {
                self.parent.obj.value = data;
                const capturedVariable = { nodeId: self.parent.obj.nodeId };
                fixExtensionObjectAndArray(capturedVariable, deferredTask);
            }, (task) => {
                postTasks0_DecodePojoString.push(task);
            }, _translateNodeId)
        }
    };
    // #endregion
    // #region UAMethod
    const state_UAMethod = {
        init(name, attrs) {
            _perform();
            this.obj = {
                nodeClass: node_opcua_data_model_1.NodeClass.Method,
                // MethodDeclarationId
                // ParentNodeId
                browseName: convertQualifiedName(attrs.BrowseName),
                parentNodeId: attrs.ParentNodeId || null,
                nodeId: convertToNodeId(attrs.NodeId) || null,
                methodDeclarationId: attrs.MethodDeclarationId ? _translateNodeId(attrs.MethodDeclarationId) : null
            };
            this.isDraft = attrs.ReleaseStatus === "Draft";
            this.isDeprecated = attrs.ReleaseStatus === "Deprecated";
        },
        finish() {
            if (canIngore({ isDraft: this.isDraft, isDeprecated: this.isDeprecated }, this.obj)) {
                return;
            }
            _internal_createNode(this.obj);
        },
        parser: {
            DisplayName: {
                finish() {
                    this.parent.obj.displayName = this.text;
                }
            },
            References: references_parser
        }
    };
    const state_ModelTableEntry = new node_opcua_xml2json_1.ReaderState({
        // ModelTableEntry
        init() {
            this._requiredModels = [];
        },
        parser: {
            RequiredModel: {
                init(name, attrs) {
                    const modelUri = attrs.ModelUri;
                    const version = attrs.Version;
                    const publicationDate = new Date(Date.parse(attrs.PublicationDate));
                    this.parent._requiredModels.push({ modelUri, version, publicationDate });
                },
                finish() {
                    /** */
                }
            }
        },
        finish() {
            const modelUri = this.attrs.ModelUri; // //"http://opcfoundation.org/UA/"
            const version = this.attrs.Version; // 1.04
            const publicationDate = this.attrs.PublicationDate ? new Date(Date.parse(this.attrs.PublicationDate)) : undefined; // "2018-05-15T00:00:00Z" "
            // optional,
            const symbolicName = this.attrs.SymbolicName;
            const accessRestrictions = this.attrs.AccessRestrictions;
            const model = {
                accessRestrictions,
                modelUri,
                publicationDate,
                requiredModels: this._requiredModels,
                symbolicName,
                version
            };
            const namespace = _add_namespace(model);
            models.push(model);
        }
    });
    // #endregion
    function _updateTranslationTable() {
        _register_namespace_uri_in_translation_table("http://opcfoundation.org/UA/");
        for (const namespaceUri of _namespaceUris) {
            _register_namespace_uri_in_translation_table(namespaceUri);
        }
    }
    function _perform() {
        if (performedCalled)
            return;
        performedCalled = true;
        /**special case for old nodeset file version 1.02 where no models exists */
        if (models.length === 0) {
            for (const namespaceuri of _namespaceUris) {
                const existingNamespace = addressSpace1.getNamespace(namespaceuri);
                if (existingNamespace) {
                    continue;
                }
                _add_namespace({
                    modelUri: namespaceuri,
                    version: "1.0.0",
                    requiredModels: []
                });
            }
        }
        _updateTranslationTable();
    }
    // state_ModelTableEntry.parser["RequiredModel"] = state_ModelTableEntry;
    let _namespaceUris = [];
    const state_0 = {
        parser: {
            Aliases: {
                init() {
                    _perform();
                },
                parser: { Alias: state_Alias }
            },
            NamespaceUris: {
                init() {
                    //
                    _namespaceUris = [];
                },
                parser: {
                    Uri: {
                        finish() {
                            _namespaceUris.push(this.text);
                        }
                    }
                },
                finish() {
                    // verify that requested namespaces are already loaded or abort with a message
                }
            },
            Models: {
                // ModelTable
                init(name, attrs) {
                    /* */
                },
                parser: {
                    Model: state_ModelTableEntry
                },
                finish() {
                    /** */
                }
            },
            UADataType: state_UADataType,
            UAMethod: state_UAMethod,
            UAObject: state_UAObject,
            UAObjectType: state_UAObjectType,
            UAReferenceType: state_UAReferenceType,
            UAVariable: state_UAVariable,
            UAVariableType: state_UAVariableType
        }
    };
    const parser = new node_opcua_xml2json_1.Xml2Json(state_0);
    async function terminate() {
        make_back_references(addressSpace1);
        // setting up Server_NamespaceArray
        if (addressSpace1.rootFolder?.objects?.server?.namespaceArray) {
            addressSpace1.rootFolder.objects.server.namespaceArray.setValueFromSource({
                arrayType: node_opcua_variant_1.VariantArrayType.Array,
                dataType: node_opcua_variant_1.DataType.String,
                value: addressSpace1.getNamespaceArray().map((ns) => ns.namespaceUri)
            });
            // istanbul ignore next
            if (doDebug) {
                debugLog("addressSpace NS = ", addressSpace1.rootFolder.objects.server.namespaceArray.readValue().value.value.join(" "));
            }
        }
        doDebug &&
            debugLog(chalk_1.default.bgGreenBright("Performing post loading tasks -------------------------------------------") +
                chalk_1.default.green("DONE"));
        async function performPostLoadingTasks(tasks) {
            for (const task of tasks) {
                try {
                    await task(addressSpace1);
                }
                catch (err) {
                    // istanbul ignore next
                    // tslint:disable:no-console
                    if (util_1.types.isNativeError(err)) {
                        errorLog(" performPostLoadingTasks Err  => ", err.message, "\n", err);
                    }
                    await task(addressSpace1);
                }
            }
            tasks.splice(0);
        }
        async function finalSteps() {
            /// ----------------------------------------------------------------------------------------
            // perform post task
            doDebug && debugLog(chalk_1.default.bgGreenBright("Performing post loading tasks -------------------------------------------"));
            await performPostLoadingTasks(postTasks);
            doDebug &&
                debugLog(chalk_1.default.bgGreenBright("Performing post loading task: Initializing Simple Variables ---------------------"));
            await performPostLoadingTasks(postTasks0_InitializeVariable);
            doDebug && debugLog(chalk_1.default.bgGreenBright("Performing DataType extraction -------------------------------------------"));
            (0, node_opcua_assert_1.assert)(!addressSpace1.suspendBackReference);
            await (0, ensure_datatype_extracted_1.ensureDatatypeExtracted)(addressSpace);
            const dataTypeManager = addressSpace.getDataTypeManager();
            /// ----------------------------------------------------------------------------------------
            doDebug && debugLog(chalk_1.default.bgGreenBright("DataType extraction done ") + chalk_1.default.green("DONE"));
            for (const { name, dataTypeNodeId } of pendingSimpleTypeToRegister) {
                if (dataTypeNodeId.namespace === 0) {
                    continue;
                }
                const dataTypeFactory = dataTypeManager.getDataTypeFactoryForNamespace(dataTypeNodeId.namespace);
            }
            pendingSimpleTypeToRegister.splice(0);
            doDebug && debugLog(chalk_1.default.bgGreenBright("Performing post loading task: Decoding Pojo String (parsing XML objects) -"));
            await performPostLoadingTasks(postTasks0_DecodePojoString);
            doDebug &&
                debugLog(chalk_1.default.bgGreenBright("Performing post loading task: Initializing Complex Variables ---------------------"));
            await performPostLoadingTasks(postTasks1_InitializeVariable);
            doDebug && debugLog(chalk_1.default.bgGreenBright("Performing post loading tasks: (assigning Extension Object to Variables) -"));
            await performPostLoadingTasks(postTasks2_AssignedExtensionObjectToDataValue);
            doDebug && debugLog(chalk_1.default.bgGreenBright("Performing post variable initialization ---------------------"));
            (0, namespace_post_step_1.promoteObjectsAndVariables)(addressSpace);
        }
        try {
            await finalSteps();
        }
        catch (err) {
            (0, node_opcua_assert_1.renderError)(err);
        }
    }
    async function addNodeSet(xmlData) {
        _reset_namespace_translation();
        parser.parseString(xmlData);
    }
    return {
        addNodeSet,
        terminate
    };
}
class NodeSetLoader {
    options;
    _s;
    constructor(addressSpace, options) {
        this.options = options;
        this._s = makeNodeSetParserEngine(addressSpace, options || {});
    }
    async addNodeSetAsync(xmlData) {
        return await this._s.addNodeSet(xmlData);
    }
    async terminate() {
        await this._s.terminate();
    }
}
exports.NodeSetLoader = NodeSetLoader;
//# sourceMappingURL=load_nodeset2.js.map