"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.preLoad = preLoad;
exports.findOrder = findOrder;
exports.generateAddressSpaceRaw = generateAddressSpaceRaw;
const node_opcua_debug_1 = require("node-opcua-debug");
const node_opcua_xml2json_1 = require("node-opcua-xml2json");
const node_opcua_date_time_1 = require("node-opcua-date-time");
const adjust_namespace_array_1 = require("../../src/nodeset_tools/adjust_namespace_array");
const load_nodeset2_1 = require("./load_nodeset2");
const doDebug = (0, node_opcua_debug_1.checkDebugFlag)(__filename);
const debugLog = (0, node_opcua_debug_1.make_debugLog)(__filename);
const errorLog = (0, node_opcua_debug_1.make_errorLog)(__filename);
async function parseDependencies(xmlData) {
    const namespaceUris = [];
    const models = [];
    let currentModel = undefined;
    const state0 = {
        parser: {
            UANodeSet: {
                parser: {
                    NamespaceUris: {
                        parser: {
                            Uri: {
                                finish() {
                                    namespaceUris.push(this.text);
                                }
                            }
                        }
                    },
                    Models: {
                        parser: {
                            Model: {
                                init(elementName, attrs) {
                                    const modelUri = attrs.ModelUri;
                                    const version = attrs.Version;
                                    const publicationDate = new Date(Date.parse(attrs.PublicationDate));
                                    currentModel = {
                                        modelUri,
                                        version,
                                        publicationDate,
                                        requiredModel: []
                                    };
                                    doDebug && console.log(`currentModel = ${JSON.stringify(currentModel)}`);
                                    models.push(currentModel);
                                },
                                parser: {
                                    RequiredModel: {
                                        init(elementName, attrs) {
                                            const modelUri = attrs.ModelUri;
                                            const version = attrs.Version;
                                            const publicationDate = new Date(Date.parse(attrs.PublicationDate));
                                            if (!currentModel) {
                                                throw new Error("Internal Error");
                                            }
                                            currentModel.requiredModel.push({
                                                modelUri,
                                                version,
                                                publicationDate
                                            });
                                        }
                                    }
                                }
                            }
                        }
                    }
                }
            }
        }
    };
    const parser = new node_opcua_xml2json_1.Xml2Json(state0);
    parser.parseString(xmlData);
    if (models.length === 0 && namespaceUris.length >= 1) {
        models.push({
            modelUri: namespaceUris[0],
            version: "1",
            publicationDate: (0, node_opcua_date_time_1.getMinOPCUADate)(),
            requiredModel: []
        });
    }
    return { models, namespaceUris: namespaceUris };
}
/**
 * Detect order of namespace loading
 */
async function preLoad(xmlFiles, xmlLoader) {
    // a nodeset2 file may define multiple namespaces
    const namespaceDesc = [];
    for (let index = 0; index < xmlFiles.length; index++) {
        doDebug && console.log("---------------------------------------------", xmlFiles[index]);
        const xmlData = await xmlLoader(xmlFiles[index]);
        const indexStart = xmlData.match(/<UANodeSet/m)?.index;
        const i1 = (xmlData.match(/<\/Models>/m)?.index || 0) + "</Models>".length;
        const i2 = (xmlData.match(/<\/NamespaceUris>/m)?.index || 0) + "</NamespaceUris>".length;
        const indexEnd = Math.max(i1, i2);
        if (indexStart === undefined || indexEnd === undefined) {
            throw new Error("Internal Error");
        }
        const xmlData2 = xmlData.substring(indexStart, indexEnd);
        doDebug &&
            console.log(xmlData2
                .split("\n")
                .splice(0, 46)
                .map((x, i) => `${i + 0} ${x}`)
                .join("\n"));
        const namespaceModel = await parseDependencies(xmlData2);
        namespaceDesc.push({ xmlData, namespaceModel, index });
    }
    return namespaceDesc;
}
function findOrder(nodesetDescs) {
    // compute the order of loading of the namespaces
    const order = [];
    const visited = new Set();
    const findNodesetIndex = (namespaceUri) => {
        const index = nodesetDescs.findIndex((x) => x.namespaceModel.models.findIndex((e) => e.modelUri === namespaceUri) !== -1);
        return index;
    };
    const visit = (model) => {
        const key = model.modelUri;
        if (visited.has(key)) {
            return;
        }
        visited.add(key);
        for (const requiredModel of model.requiredModel) {
            const requiredModelIndex = findNodesetIndex(requiredModel.modelUri);
            if (requiredModelIndex === -1) {
                throw new Error("Cannot find namespace for " + requiredModel.modelUri);
            }
            const nd = nodesetDescs[requiredModelIndex];
            for (const n of nd.namespaceModel.models) {
                visit(n);
            }
        }
        const nodesetIndex = findNodesetIndex(model.modelUri);
        const alreadyIn = order.findIndex((x) => x === nodesetIndex) !== -1;
        if (!alreadyIn)
            order.push(nodesetIndex);
    };
    const visit2 = (nodesetDesc) => {
        for (const model of nodesetDesc.namespaceModel.models.values()) {
            visit(model);
        }
    };
    for (let index = 0; index < nodesetDescs.length; index++) {
        const nodesetDesc = nodesetDescs[index];
        visit2(nodesetDesc);
    }
    return order;
}
/**
 * @param addressSpace the addressSpace to populate
 * @xmlFiles: a lis of xml files
 * @param xmlLoader - a helper function to return the content of the xml file
 */
async function generateAddressSpaceRaw(addressSpace, xmlFiles, xmlLoader, options) {
    const nodesetLoader = new load_nodeset2_1.NodeSetLoader(addressSpace, options);
    if (!Array.isArray(xmlFiles)) {
        xmlFiles = [xmlFiles];
    }
    const nodesetDesc = await preLoad(xmlFiles, xmlLoader);
    const order = findOrder(nodesetDesc);
    // register namespace in the same order as specified in the xmlFiles array
    for (let index = 0; index < order.length; index++) {
        const n = nodesetDesc[index];
        for (const model of n.namespaceModel.models) {
            const ns = addressSpace.registerNamespace(model.modelUri);
            ns.setRequiredModels(model.requiredModel);
        }
    }
    for (let index = 0; index < order.length; index++) {
        const nodesetIndex = order[index];
        const nodeset = nodesetDesc[nodesetIndex];
        debugLog(" loading ", nodesetIndex, nodeset.xmlData.length);
        try {
            await nodesetLoader.addNodeSetAsync(nodeset.xmlData);
        }
        catch (err) {
            errorLog("generateAddressSpace:  Loading xml file ", xmlFiles[index], " failed with error ", err.message);
            throw err;
        }
    }
    await nodesetLoader.terminate();
    (0, adjust_namespace_array_1.adjustNamespaceArray)(addressSpace);
    // however process them in series
}
//# sourceMappingURL=generateAddressSpaceRaw.js.map