"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.callMethodHelper = callMethodHelper;
/**
 * @module node-opcua-address-space
 */
const node_opcua_assert_1 = require("node-opcua-assert");
const node_opcua_data_model_1 = require("node-opcua-data-model");
const node_opcua_nodeid_1 = require("node-opcua-nodeid");
const node_opcua_status_code_1 = require("node-opcua-status-code");
const argument_list_1 = require("./argument_list");
const resolve_opaque_on_address_space_1 = require("./resolve_opaque_on_address_space");
// Symbolic Id                   Description
// ----------------------------  -----------------------------------------------------------------------------
// BadNodeIdInvalid              Used to indicate that the specified object is not valid.
//
// BadNodeIdUnknown             Used to indicate that the specified object is not valid.
//
// BadArgumentsMissing          The client did not specify all of the input arguments for the method.
// BadUserAccessDenied
//
// BadMethodInvalid             The method id does not refer to a method for the specified object.
// BadOutOfRange                Used to indicate that an input argument is outside the acceptable range.
// BadTypeMismatch              Used to indicate that an input argument does not have the correct data type.
//                               A ByteString is structurally the same as a one dimensional array of Byte.
//                               A server shall accept a ByteString if an array of Byte is expected.
// BadNoCommunication
async function callMethodHelper(context, addressSpace, callMethodRequest) {
    const objectId = callMethodRequest.objectId;
    const methodId = callMethodRequest.methodId;
    const inputArguments = callMethodRequest.inputArguments || [];
    (0, node_opcua_assert_1.assert)(objectId instanceof node_opcua_nodeid_1.NodeId);
    (0, node_opcua_assert_1.assert)(methodId instanceof node_opcua_nodeid_1.NodeId);
    const object = addressSpace.findNode(objectId);
    if (!object) {
        return { statusCode: node_opcua_status_code_1.StatusCodes.BadNodeIdUnknown };
    }
    if (object.nodeClass !== node_opcua_data_model_1.NodeClass.Object && object.nodeClass !== node_opcua_data_model_1.NodeClass.ObjectType) {
        return { statusCode: node_opcua_status_code_1.StatusCodes.BadNodeIdInvalid };
    }
    const methodObj = addressSpace.findNode(methodId);
    if (!methodObj) {
        return { statusCode: node_opcua_status_code_1.StatusCodes.BadMethodInvalid };
    }
    if (methodObj.nodeClass !== node_opcua_data_model_1.NodeClass.Method) {
        return { statusCode: node_opcua_status_code_1.StatusCodes.BadMethodInvalid };
    }
    const response1 = (0, argument_list_1.getMethodDeclaration_ArgumentList)(addressSpace, objectId, methodId);
    if (response1.statusCode.isNotGood()) {
        return { statusCode: response1.statusCode };
    }
    const methodDeclaration = response1.methodDeclaration;
    // verify input Parameters
    const methodInputArguments = methodDeclaration.getInputArguments();
    const response = (0, argument_list_1.verifyArguments_ArgumentList)(addressSpace, methodInputArguments, inputArguments);
    if (response.statusCode.isNotGood()) {
        return response;
    }
    try {
        await (0, resolve_opaque_on_address_space_1.resolveOpaqueOnAddressSpace)(addressSpace, inputArguments);
        const callMethodResponse = await methodObj.execute(object, inputArguments, context);
        callMethodResponse.inputArgumentResults =
            callMethodResponse.inputArgumentResults || response.inputArgumentResults || [];
        (0, node_opcua_assert_1.assert)(callMethodResponse.statusCode);
        if (callMethodResponse.statusCode?.isGood()) {
            (0, node_opcua_assert_1.assert)(Array.isArray(callMethodResponse.outputArguments));
        }
        (0, node_opcua_assert_1.assert)(Array.isArray(callMethodResponse.inputArgumentResults));
        (0, node_opcua_assert_1.assert)(callMethodResponse.inputArgumentResults.length === methodInputArguments.length);
        const outputArguments = callMethodResponse.outputArguments || [];
        await (0, resolve_opaque_on_address_space_1.resolveOpaqueOnAddressSpace)(addressSpace, outputArguments);
        return callMethodResponse;
    }
    catch (err) {
        return { statusCode: node_opcua_status_code_1.StatusCodes.BadInternalError };
    }
}
//# sourceMappingURL=call_helpers.js.map