"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.CloneHelper = void 0;
exports.fullPath = fullPath;
exports.fullPath2 = fullPath2;
exports.exploreNode = exploreNode;
exports.reconstructNonHierarchicalReferences = reconstructNonHierarchicalReferences;
exports.reconstructFunctionalGroupType = reconstructFunctionalGroupType;
const node_opcua_assert_1 = require("node-opcua-assert");
const node_opcua_debug_1 = require("node-opcua-debug");
const node_opcua_data_model_1 = require("node-opcua-data-model");
const node_opcua_nodeid_1 = require("node-opcua-nodeid");
const node_opcua_constants_1 = require("node-opcua-constants");
const warningLog = (0, node_opcua_debug_1.make_warningLog)("CLONE");
const errorLog = (0, node_opcua_debug_1.make_errorLog)(__filename);
const doTrace = (0, node_opcua_debug_1.checkDebugFlag)("INSTANTIATE");
const traceLog = errorLog;
// istanbul ignore next
/** @private */
function fullPath(node) {
    const browseName = node.browseName.toString();
    const parent = node.findReferencesExAsObject("Aggregates", node_opcua_data_model_1.BrowseDirection.Inverse)[0];
    if (parent) {
        return fullPath(parent) + "/" + browseName;
    }
    const containingFolder = node.findReferencesExAsObject("Organizes", node_opcua_data_model_1.BrowseDirection.Inverse)[0];
    if (containingFolder) {
        return fullPath(containingFolder) + "@" + browseName;
    }
    return browseName;
}
// istanbul ignore next
/** @private */
function fullPath2(node) {
    return fullPath(node) + " (" + node.nodeId.toString() + ")";
}
// istanbul ignore next
/** @private */
function exploreNode(node) {
    const f = (n) => {
        return `${n.browseName.toString()} (${n.nodeId.toString()})${n.modellingRule ? " - " + n.modellingRule : ""}`;
    };
    const r = (r) => {
        const ref = node.addressSpace.findNode(r);
        if (!ref)
            return `${r.nodeId.toString()} (unknown)`;
        return ref?.browseName.toString() + " " + ref.nodeId.toString();
    };
    const map = new Set();
    const _explore = (node, pad) => {
        const a = node.findReferencesEx("Aggregates", node_opcua_data_model_1.BrowseDirection.Forward);
        const b = node.findReferencesEx("Organizes", node_opcua_data_model_1.BrowseDirection.Forward);
        for (const ref of [...a, ...b]) {
            const alreadyVisited = map.has(ref.nodeId.toString());
            traceLog(pad, "  +-- ", r(ref.referenceType).padEnd(20), "-->", f(ref.node).padEnd(10), alreadyVisited ? " (already visited)" : "");
            if (alreadyVisited) {
                continue;
            }
            map.add(ref.nodeId.toString());
            _explore(ref.node, pad + "   ");
        }
    };
    traceLog("exploring ", f(node));
    _explore(node, "   ");
}
//
//  case 1:
//   /-----------------------------\
//   | AcknowledgeableConditionType |
//   \-----------------------------/
//              ^        |
//              |        +---------------------|- (EnabledState)   (shadow element)
//              |
//   /-----------------------------\
//   |        AlarmConditionType   |
//   \-----------------------------/
//              |
//              +-------------------------------|- EnabledState    <
//
// find also child object with the same browse name that are
// overridden in the SuperType
// case 2:
//
//   /-----------------------------\
//   | MyDeviceType               |
//   \-----------------------------/
//              ^        |
//              |        |       +----------+
//              |        +-------| Folder1  |
//              |                +----------+
//              |                     |
//              |                     +--------------|- (EnabledState)   (shadow element)
//              |
//   /-----------------------------\
//   | MyDeriveDeviceType   |
//   \-----------------------------/
//              |
//              |        |       +----------+
//              |        +-------| Folder1  |
//              |                +----------+
//              |                     |
//              |                     +--------------|- (EnabledState)
//
// find also child object with the same browse name that are
function _get_parent_type_and_path(originalObject) {
    if (originalObject.nodeClass === node_opcua_data_model_1.NodeClass.Method) {
        return { parentType: null, path: [] };
    }
    const addressSpace = originalObject.addressSpace;
    const parents = originalObject.findReferencesEx("HasChild", node_opcua_data_model_1.BrowseDirection.Inverse);
    // istanbul ignore next
    if (parents.length > 1) {
        warningLog(" object ", originalObject.browseName.toString(), " has more than one parent !");
        warningLog(originalObject.toString());
        warningLog(" parents : ");
        for (const parent of parents) {
            warningLog("     ", parent.toString(), addressSpace.findNode(parent.nodeId).browseName.toString());
        }
        return { parentType: null, path: [] };
    }
    (0, node_opcua_assert_1.assert)(parents.length === 0 || parents.length === 1);
    if (parents.length === 0) {
        return { parentType: null, path: [] };
    }
    const theParent = addressSpace.findNode(parents[0].nodeId);
    if (theParent && (theParent.nodeClass === node_opcua_data_model_1.NodeClass.VariableType || theParent.nodeClass === node_opcua_data_model_1.NodeClass.ObjectType)) {
        return { parentType: theParent, path: [originalObject.browseName] };
    }
    // walk up
    const { parentType, path } = _get_parent_type_and_path(theParent);
    return { parentType, path: [...path, originalObject.browseName] };
}
function followPath(node, path) {
    let current = node;
    for (const qn of path) {
        const ref = current
            .findReferencesExAsObject("HierarchicalReferences", node_opcua_data_model_1.BrowseDirection.Forward)
            .find((r) => r.browseName.toString() === qn.toString());
        if (!ref) {
            return null;
        }
        current = ref;
    }
    return current;
}
class CloneHelper {
    level = 0;
    _context = null;
    _contextStack = [];
    mapTypeInstanceChildren = new Map();
    pad() {
        return " ".padEnd(this.level * 2, " ");
    }
    getClonedArray() {
        const result = [];
        for (const map of this.mapTypeInstanceChildren.values()) {
            for (const cloneInfo of map.values()) {
                result.push(cloneInfo);
            }
        }
        return result;
    }
    pushContext({ clonedParent, originalParent }) {
        // istanbul ignore next
        doTrace &&
            traceLog("push context: ", "original parent = ", fullPath2(originalParent), "cloned parent =", fullPath2(clonedParent));
        const typeInstance = originalParent.nodeId.toString() + clonedParent.nodeId.toString();
        // istanbul ignore next
        doTrace && traceLog("typeInstance (1) = ", typeInstance, fullPath2(originalParent), fullPath2(clonedParent));
        let a = this.mapTypeInstanceChildren.get(typeInstance);
        if (a) {
            throw new Error("Internal Error");
        }
        a = new Map();
        this.mapTypeInstanceChildren.set(typeInstance, a);
        if (this._context) {
            this._contextStack.push(this._context);
        }
        this._context = a;
    }
    popContext() {
        (0, node_opcua_assert_1.assert)(this._contextStack.length > 0);
        this._context = this._contextStack.pop();
    }
    registerClonedObject({ clonedNode, originalNode }) {
        if (!this._context) {
            this.pushContext({ clonedParent: clonedNode, originalParent: originalNode });
        }
        // istanbul ignore next
        doTrace &&
            traceLog("registerClonedObject", "originalNode = ", fullPath2(originalNode), "clonedNode =", fullPath2(clonedNode));
        const insertShadow = (map) => {
            const { parentType, path } = _get_parent_type_and_path(originalNode);
            if (parentType) {
                let base = parentType.subtypeOfObj;
                while (base) {
                    const shadowChild = followPath(base, path);
                    if (shadowChild) {
                        // istanbul ignore next
                        doTrace && traceLog("shadowChild = ", fullPath2(shadowChild));
                        map.set(shadowChild.nodeId.toString(), {
                            cloned: clonedNode,
                            original: shadowChild
                        });
                    }
                    base = base.subtypeOfObj;
                }
            }
        };
        // find subTypeOf
        if (!this._context) {
            throw new Error("internal error: Cannot find context");
        }
        // also create  [Type+Instance] map
        // to do so we need to find the TypeDefinition of the originalNode
        this._context.set(originalNode.nodeId.toString(), {
            cloned: clonedNode,
            original: originalNode
        });
        insertShadow(this._context);
    }
    getCloned({ originalParent, clonedParent, originalNode }) {
        //
        //  Type                                                 Instance
        //    +-> Folder (A)                                         +-> Folder (A')
        //    |      |                                               |     |
        //    |      +- Component (B)                                |     +- Component (B')
        //    °-> Folder (C)       [parentNode]                      +-> Folder (C')          <= [clonedParent]
        //          |                                                      |
        //          +- Component (B again !)  [originalNode]               +- Component (B again !)
        // istanbul ignore next
        doTrace &&
            traceLog("typeInstance (3) = originalParent", fullPath2(originalParent), "originalNode=", fullPath2(originalNode), "clonedParent", fullPath2(clonedParent));
        const info = this._context.get(originalNode.nodeId.toString());
        if (info) {
            return info.cloned;
        }
        return null;
    }
}
exports.CloneHelper = CloneHelper;
const hasTypeDefinitionNodeId = (0, node_opcua_nodeid_1.makeNodeId)(node_opcua_constants_1.ReferenceTypeIds.HasTypeDefinition);
const hasModellingRuleNodeId = (0, node_opcua_nodeid_1.makeNodeId)(node_opcua_constants_1.ReferenceTypeIds.HasModellingRule);
/**
 * remove unwanted reference such as HasTypeDefinition and HasModellingRule
 * from the list
 */
function _remove_unwanted_ref(references) {
    // filter out HasTypeDefinition (i=40) , HasModellingRule (i=37);
    references = references.filter((reference) => !(0, node_opcua_nodeid_1.sameNodeId)(reference.referenceType, hasTypeDefinitionNodeId) &&
        !(0, node_opcua_nodeid_1.sameNodeId)(reference.referenceType, hasModellingRuleNodeId));
    return references;
}
/**
 *
 */
function findNonHierarchicalReferences(originalObject) {
    // todo: MEMOIZE this method
    const addressSpace = originalObject.addressSpace;
    // we need to explore the non hierarchical references backwards
    let references = originalObject.findReferencesEx("NonHierarchicalReferences", node_opcua_data_model_1.BrowseDirection.Inverse);
    references = [].concat(references, originalObject.findReferencesEx("HasEventSource", node_opcua_data_model_1.BrowseDirection.Inverse));
    const { parentType, path } = _get_parent_type_and_path(originalObject);
    if (parentType && parentType.subtypeOfObj) {
        // parent is a ObjectType or VariableType and is not a root type
        (0, node_opcua_assert_1.assert)(parentType.nodeClass === node_opcua_data_model_1.NodeClass.VariableType || parentType.nodeClass === node_opcua_data_model_1.NodeClass.ObjectType);
        // let investigate the same child base child
        const child = followPath(parentType.subtypeOfObj, path);
        if (child) {
            const baseRef = findNonHierarchicalReferences(child);
            references = [].concat(references, baseRef);
        }
    }
    // perform some cleanup
    references = _remove_unwanted_ref(references);
    return references;
}
const findImplementedObject = (cloneInfoArray, ref) => {
    const a = cloneInfoArray.filter((x) => x.original.nodeId.toString() === ref.nodeId.toString());
    if (a.length === 0)
        return null;
    const info = a[0];
    return info;
};
function reconstructNonHierarchicalReferences(extraInfo) {
    const cloneInfoArray = extraInfo.getClonedArray();
    // navigate through original objects to find those that are being references by node that
    // have been cloned .
    // this could be node organized by some FunctionalGroup
    // istanbul ignore next
    doTrace && traceLog("reconstructNonHierarchicalReferences");
    for (const { original, cloned } of cloneInfoArray) {
        apply(original, cloned);
    }
    function apply(original, cloned) {
        const addressSpace = original.addressSpace;
        // find NonHierarchical References on original object
        const originalNonHierarchical = findNonHierarchicalReferences(original);
        if (originalNonHierarchical.length === 0) {
            return;
        }
        // istanbul ignore next
        doTrace && traceLog(" investigation ", "original", fullPath2(original), node_opcua_data_model_1.NodeClass[cloned.nodeClass], fullPath2(cloned));
        for (const ref of originalNonHierarchical) {
            const info = findImplementedObject(cloneInfoArray, ref);
            if (!info)
                continue;
            // if the object pointed by this reference is also cloned ...
            const originalDest = info.original;
            const cloneDest = info.cloned;
            // istanbul ignore next
            doTrace &&
                traceLog("   adding reference ", fullPath2(addressSpace.findNode(ref.referenceType)), " from cloned ", fullPath2(cloned), " to cloned ", fullPath2(cloneDest));
            // restore reference
            cloned.addReference({
                isForward: false,
                nodeId: cloneDest.nodeId,
                referenceType: ref.referenceType
            });
        }
    }
}
/**
 * recreate functional group types according to type definition
 *

 * @param baseType
 */
/* @example:
 *
 *    MyDeviceType
 *        |
 *        +----------|- ParameterSet(BaseObjectType)
 *        |                   |
 *        |                   +-----------------|- Parameter1
 *        |                                             ^
 *        +----------|- Config(FunctionalGroupType)     |
 *                                |                     |
 *                                +-------- Organizes---+
 */
function reconstructFunctionalGroupType(extraInfo) {
    const cloneInfoArray = extraInfo.getClonedArray();
    // navigate through original objects to find those that are being organized by some FunctionalGroup
    for (const { original, cloned } of cloneInfoArray) {
        const organizedByArray = original.findReferencesEx("Organizes", node_opcua_data_model_1.BrowseDirection.Inverse);
        for (const ref of organizedByArray) {
            const info = findImplementedObject(cloneInfoArray, ref);
            if (!info)
                continue;
            const folder = info.original;
            if (folder.nodeClass !== node_opcua_data_model_1.NodeClass.Object)
                continue;
            if (!folder.typeDefinitionObj)
                continue;
            if (folder.typeDefinitionObj.browseName.name.toString() !== "FunctionalGroupType") {
                continue;
            }
            // now create the same reference with the instantiated function group
            const destFolder = info.cloned;
            (0, node_opcua_assert_1.assert)(ref.referenceType);
            // may be we should check that the referenceType is a subtype of Organizes
            const alreadyExist = destFolder
                .findReferences(ref.referenceType, !ref.isForward)
                .find((r) => r.nodeId === cloned.nodeId);
            if (alreadyExist) {
                continue;
            }
            destFolder.addReference({
                isForward: !ref.isForward,
                nodeId: cloned.nodeId,
                referenceType: ref.referenceType
            });
        }
    }
}
//# sourceMappingURL=clone_helper.js.map