import { toMessageDict } from "./messages.js";
import { StreamError } from "./errors.js";
export class StreamManager {
    constructor(messages) {
        Object.defineProperty(this, "abortRef", {
            enumerable: true,
            configurable: true,
            writable: true,
            value: new AbortController()
        });
        Object.defineProperty(this, "messages", {
            enumerable: true,
            configurable: true,
            writable: true,
            value: void 0
        });
        Object.defineProperty(this, "listeners", {
            enumerable: true,
            configurable: true,
            writable: true,
            value: new Set()
        });
        Object.defineProperty(this, "state", {
            enumerable: true,
            configurable: true,
            writable: true,
            value: void 0
        });
        Object.defineProperty(this, "setState", {
            enumerable: true,
            configurable: true,
            writable: true,
            value: (newState) => {
                this.state = { ...this.state, ...newState };
                this.notifyListeners();
            }
        });
        Object.defineProperty(this, "notifyListeners", {
            enumerable: true,
            configurable: true,
            writable: true,
            value: () => {
                this.listeners.forEach((listener) => listener());
            }
        });
        Object.defineProperty(this, "subscribe", {
            enumerable: true,
            configurable: true,
            writable: true,
            value: (listener) => {
                this.listeners.add(listener);
                return () => this.listeners.delete(listener);
            }
        });
        Object.defineProperty(this, "getSnapshot", {
            enumerable: true,
            configurable: true,
            writable: true,
            value: () => this.state
        });
        Object.defineProperty(this, "setStreamValues", {
            enumerable: true,
            configurable: true,
            writable: true,
            value: (values, kind = "stream") => {
                if (typeof values === "function") {
                    const [prevValues, prevKind] = this.state.values ?? [null, "stream"];
                    const nextValues = values(prevValues, prevKind);
                    this.setState({ values: nextValues != null ? [nextValues, kind] : null });
                }
                else {
                    const nextValues = values != null ? [values, kind] : null;
                    this.setState({ values: nextValues });
                }
            }
        });
        Object.defineProperty(this, "getMutateFn", {
            enumerable: true,
            configurable: true,
            writable: true,
            value: (kind, historyValues) => {
                return (update) => {
                    const prev = {
                        ...historyValues,
                        ...(this.state.values ?? [null, "stream"])[0],
                    };
                    const next = typeof update === "function" ? update(prev) : update;
                    this.setStreamValues({ ...prev, ...next }, kind);
                };
            }
        });
        Object.defineProperty(this, "matchEventType", {
            enumerable: true,
            configurable: true,
            writable: true,
            value: (expected, actual, _data) => {
                return expected === actual || actual.startsWith(`${expected}|`);
            }
        });
        Object.defineProperty(this, "start", {
            enumerable: true,
            configurable: true,
            writable: true,
            value: async (action, options) => {
                if (this.state.isLoading)
                    return;
                try {
                    this.setState({ isLoading: true, error: undefined });
                    this.abortRef = new AbortController();
                    const run = await action(this.abortRef.signal);
                    let streamError;
                    for await (const { event, data } of run) {
                        if (event === "error") {
                            streamError = new StreamError(data);
                            break;
                        }
                        const namespace = event.includes("|")
                            ? event.split("|").slice(1)
                            : undefined;
                        const mutate = this.getMutateFn("stream", options.initialValues);
                        if (event === "metadata")
                            options.callbacks.onMetadataEvent?.(data);
                        if (event === "events")
                            options.callbacks.onLangChainEvent?.(data);
                        if (this.matchEventType("updates", event, data)) {
                            options.callbacks.onUpdateEvent?.(data, { namespace, mutate });
                        }
                        if (this.matchEventType("custom", event, data)) {
                            options.callbacks.onCustomEvent?.(data, { namespace, mutate });
                        }
                        if (this.matchEventType("checkpoints", event, data)) {
                            options.callbacks.onCheckpointEvent?.(data, { namespace });
                        }
                        if (this.matchEventType("tasks", event, data)) {
                            options.callbacks.onTaskEvent?.(data, { namespace });
                        }
                        if (this.matchEventType("debug", event, data)) {
                            options.callbacks.onDebugEvent?.(data, { namespace });
                        }
                        if (event === "values") {
                            if ("__interrupt__" in data) {
                                this.setStreamValues((prev) => ({ ...prev, ...data }));
                            }
                            else {
                                this.setStreamValues(data);
                            }
                        }
                        if (this.matchEventType("messages", event, data)) {
                            const [serialized, metadata] = data;
                            const messageId = this.messages.add(serialized, metadata);
                            if (!messageId) {
                                console.warn("Failed to add message to manager, no message ID found");
                                continue;
                            }
                            this.setStreamValues((streamValues) => {
                                const values = { ...options.initialValues, ...streamValues };
                                // Assumption: we're concatenating the message
                                const messages = options.getMessages(values).slice();
                                const { chunk, index } = this.messages.get(messageId, messages.length) ?? {};
                                if (!chunk || index == null)
                                    return values;
                                if (chunk.getType() === "remove") {
                                    messages.splice(index, 1);
                                }
                                else {
                                    messages[index] = toMessageDict(chunk);
                                }
                                return options.setMessages(values, messages);
                            });
                        }
                    }
                    if (streamError != null)
                        throw streamError;
                    const values = await options.onSuccess?.();
                    if (typeof values !== "undefined")
                        this.setStreamValues(values);
                }
                catch (error) {
                    if (!(error instanceof Error && // eslint-disable-line no-instanceof/no-instanceof
                        (error.name === "AbortError" || error.name === "TimeoutError"))) {
                        console.error(error);
                        this.setState({ error });
                        await options.onError?.(error);
                    }
                }
                finally {
                    this.setState({ isLoading: false });
                    this.abortRef = new AbortController();
                    options.onFinish?.();
                }
            }
        });
        Object.defineProperty(this, "stop", {
            enumerable: true,
            configurable: true,
            writable: true,
            value: async (historyValues, options) => {
                this.abortRef.abort();
                this.abortRef = new AbortController();
                options.onStop?.({ mutate: this.getMutateFn("stop", historyValues) });
            }
        });
        Object.defineProperty(this, "clear", {
            enumerable: true,
            configurable: true,
            writable: true,
            value: () => {
                // Cancel any running streams
                this.abortRef.abort();
                this.abortRef = new AbortController();
                // Set the stream state to null
                this.setState({ error: undefined, values: null });
                // Clear any pending messages
                this.messages.clear();
            }
        });
        this.messages = messages;
        this.state = { isLoading: false, values: null, error: undefined };
    }
    get isLoading() {
        return this.state.isLoading;
    }
    get values() {
        return this.state.values?.[0] ?? null;
    }
    get error() {
        return this.state.error;
    }
}
