/* __LC_ALLOW_ENTRYPOINT_SIDE_EFFECTS__ */
"use client";
import { useEffect, useRef, useState, useSyncExternalStore } from "react";
import { StreamManager } from "../ui/manager.js";
import { MessageTupleManager } from "../ui/messages.js";
import { BytesLineDecoder, SSEDecoder } from "../utils/sse.js";
import { IterableReadableStream } from "../utils/stream.js";
import { useControllableThreadId } from "./thread.js";
export class FetchStreamTransport {
    constructor(options) {
        Object.defineProperty(this, "options", {
            enumerable: true,
            configurable: true,
            writable: true,
            value: options
        });
    }
    async stream(payload) {
        const { signal, ...body } = payload;
        let requestInit = {
            method: "POST",
            headers: {
                "Content-Type": "application/json",
                ...this.options.defaultHeaders,
            },
            body: JSON.stringify(body),
            signal,
        };
        if (this.options.onRequest) {
            requestInit = await this.options.onRequest(this.options.apiUrl, requestInit);
        }
        const fetchFn = this.options.fetch ?? fetch;
        const response = await fetchFn(this.options.apiUrl, requestInit);
        if (!response.ok) {
            throw new Error(`Failed to stream: ${response.statusText}`);
        }
        const stream = (response.body || new ReadableStream({ start: (ctrl) => ctrl.close() }))
            .pipeThrough(BytesLineDecoder())
            .pipeThrough(SSEDecoder());
        return IterableReadableStream.fromReadableStream(stream);
    }
}
export function useStreamCustom(options) {
    const [messageManager] = useState(() => new MessageTupleManager());
    const [stream] = useState(() => new StreamManager(messageManager));
    useSyncExternalStore(stream.subscribe, stream.getSnapshot, stream.getSnapshot);
    const [threadId, onThreadId] = useControllableThreadId(options);
    const threadIdRef = useRef(threadId);
    // Cancel the stream if thread ID has changed
    useEffect(() => {
        if (threadIdRef.current !== threadId) {
            threadIdRef.current = threadId;
            stream.clear();
        }
    }, [threadId, stream]);
    const getMessages = (value) => {
        const messagesKey = options.messagesKey ?? "messages";
        return Array.isArray(value[messagesKey]) ? value[messagesKey] : [];
    };
    const setMessages = (current, messages) => {
        const messagesKey = options.messagesKey ?? "messages";
        return { ...current, [messagesKey]: messages };
    };
    const historyValues = options.initialValues ?? {};
    const stop = () => stream.stop(historyValues, { onStop: options.onStop });
    const submit = async (values, submitOptions) => {
        let callbackMeta;
        let usableThreadId = threadId;
        stream.setStreamValues(() => {
            if (submitOptions?.optimisticValues != null) {
                return {
                    ...historyValues,
                    ...(typeof submitOptions.optimisticValues === "function"
                        ? submitOptions.optimisticValues(historyValues)
                        : submitOptions.optimisticValues),
                };
            }
            return { ...historyValues };
        });
        await stream.start(async (signal) => {
            if (!usableThreadId) {
                // generate random thread id
                usableThreadId = crypto.randomUUID();
                threadIdRef.current = usableThreadId;
                onThreadId(usableThreadId);
            }
            if (!usableThreadId) {
                throw new Error("Failed to obtain valid thread ID.");
            }
            return options.transport.stream({
                input: values,
                context: submitOptions?.context,
                command: submitOptions?.command,
                signal,
                config: {
                    ...submitOptions?.config,
                    configurable: {
                        thread_id: usableThreadId,
                        ...submitOptions?.config?.configurable,
                    },
                },
            });
        }, {
            getMessages,
            setMessages,
            initialValues: {},
            callbacks: options,
            onSuccess: () => undefined,
            onError(error) {
                options.onError?.(error, callbackMeta);
            },
        });
    };
    return {
        get values() {
            return stream.values ?? {};
        },
        error: stream.error,
        isLoading: stream.isLoading,
        stop,
        submit,
        get interrupt() {
            if (stream.values != null &&
                "__interrupt__" in stream.values &&
                Array.isArray(stream.values.__interrupt__)) {
                const valueInterrupts = stream.values.__interrupt__;
                if (valueInterrupts.length === 0)
                    return { when: "breakpoint" };
                if (valueInterrupts.length === 1)
                    return valueInterrupts[0];
                // TODO: fix the typing of interrupts if multiple interrupts are returned
                return valueInterrupts;
            }
            return undefined;
        },
        get messages() {
            if (!stream.values)
                return [];
            return getMessages(stream.values);
        },
    };
}
