"use strict";
var __importDefault = (this && this.__importDefault) || function (mod) {
    return (mod && mod.__esModule) ? mod : { "default": mod };
};
Object.defineProperty(exports, "__esModule", { value: true });
exports.BaseModel = void 0;
const langgraph_1 = require("@langchain/langgraph");
const promptSynonyms_1 = require("./promptSynonyms");
//@ts-nocheck
const zod_1 = require("zod");
const scadaHelpers_1 = __importDefault(require("./scadaHelpers"));
const tools_1 = require("../../tools");
// --------------------
// Base Tool
// --------------------
class StructuredTool {
    // Add a generic invoke helper
    async invoke(args) {
        const validated = this.schema.parse(args);
        return this._call(validated);
    }
    // Runtime validation helper
    validate(args) {
        return this.schema.parse(args);
    }
}
// --------------------
// DocsTool
// --------------------
const DocsToolSchema = zod_1.z.object({
    query: zod_1.z.string().describe("The documentation search query in natural language"),
    topK: zod_1.z.number().optional().default(5).describe("Maximum number of passages to retrieve (default 5)"),
});
class DocsTool extends StructuredTool {
    constructor(scadaHelper) {
        super();
        this.name = "retrieve_documents";
        this.description = "Search documentation (manuals, guides, configuration reference).";
        this.schema = DocsToolSchema;
        this.scadaHelper = scadaHelper;
    }
    async _call({ query, topK }) {
        const [docsString, docs] = await this.scadaHelper.searchDocs(query, topK ?? 5);
        return [docsString, docs];
    }
}
DocsTool.schema = DocsToolSchema;
// --------------------
// LiveTool
// --------------------
const LiveToolSchema = zod_1.z.object({
    tags: zod_1.z.array(zod_1.z.string()).nullable().optional().describe('Optional list of tag names to read live values for. Must be a JSON array of strings (e.g., ["Motor1", "Motor2"])'),
});
class LiveTool extends StructuredTool {
    constructor(scadaHelper) {
        super();
        this.name = "read_live_tags";
        this.description = 'Read live SCADA tag values. Always provide tags as a JSON array of strings (e.g., ["Motor1", "Motor2"]).';
        this.schema = LiveToolSchema;
        this.scadaHelper = scadaHelper;
    }
    async _call(tags) {
        const [data, conf] = await this.scadaHelper.getLiveTags(tags);
        return [data, conf];
    }
}
LiveTool.schema = LiveToolSchema;
// --------------------
// HistoryTool
// --------------------
const HistoryToolSchema = zod_1.z.object({
    tags: zod_1.z.array(zod_1.z.string()).nullable().optional().describe('List of tag names to query. Must be a non-empty array of strings (e.g., ["Motor1", "Motor2"]).'),
    from: zod_1.z.string().optional().nullable().describe("Optional start timestamp in ISO 8601 format (e.g., 2025-08-23T12:00:00Z)"),
    to: zod_1.z.string().optional().nullable().describe("Optional end timestamp in ISO 8601 format (e.g., 2025-08-23T13:00:00Z)"),
});
class HistoryTool extends StructuredTool {
    constructor(scadaHelper) {
        super();
        this.name = "query_historical_values";
        this.description = 'Query historical SCADA process values (numeric/analog tag values over time). Always provide tags as a JSON array of strings (e.g., ["Motor1", "Motor2"]).';
        this.schema = HistoryToolSchema;
        this.scadaHelper = scadaHelper;
    }
    async _call(input) {
        // Fallback: convert string to array if needed
        let t = input.tags;
        if (t && typeof t === "string") {
            input.tags = t.split(",").map((t) => t.trim());
        }
        const [data, conf] = await this.scadaHelper.getHistory(input);
        return [data, conf];
    }
}
HistoryTool.schema = HistoryToolSchema;
// --------------------
// LiveAlarmsTool
// --------------------
const LiveAlarmsToolSchema = zod_1.z.object({
    alarms: zod_1.z.array(zod_1.z.string()).optional().nullable().describe('List of alarms to query. Must be a non-empty array of strings (e.g., ["Motor1", "Motor2"]).'),
    limit: zod_1.z.number().int().min(1).max(1000).optional().default(100).describe("Maximum number of results to return (1–1000)"),
});
class LiveAlarmsTool extends StructuredTool {
    constructor(scadaHelper) {
        super();
        this.name = "get_live_alarms";
        this.description = "Fetch currently active/live alarms (e.g., unacknowledged alarms).";
        this.schema = LiveAlarmsToolSchema;
        this.scadaHelper = scadaHelper;
    }
    async _call(input) {
        const [data, conf] = await this.scadaHelper.getAlarms(input, false);
        return [data, conf];
    }
}
LiveAlarmsTool.schema = LiveAlarmsToolSchema;
class CurrentDateTool extends StructuredTool {
    constructor() {
        super(...arguments);
        this.name = "get_current_date";
        this.description = "Get the current date and time in ISO 8601 format.";
        this.schema = CurrentDateTool.schema;
    }
    async _call() {
        const now = new Date().toISOString();
        return [now, {}];
    }
}
CurrentDateTool.schema = zod_1.z.object({}); // no input needed
class SystemStatusTool extends StructuredTool {
    constructor(scadaHelper) {
        super();
        this.name = "get_system_status";
        this.description = `Provides a high-level overview of the SCADA system status.
Use this for questions like:
- "What is the system status?"
- "Is everything working fine?"
- "Any problems right now?"
This tool collects active alarms and critical tag statuses to produce a summarized report.`;
        this.schema = SystemStatusTool.schema;
        this.scadaHelper = scadaHelper;
    }
    async _call() {
        let [data, conf] = await this.scadaHelper.getAlarms({}, true);
        return [data, conf];
    }
}
SystemStatusTool.schema = zod_1.z.object({}); // no input needed
// --------------------
// AlarmHistoryTool
// --------------------
const AlarmHistoryToolSchema = zod_1.z.object({
    alarms: zod_1.z.array(zod_1.z.string()).optional().nullable().describe('List of alarms to query. Must be a non-empty array of strings (e.g., ["Motor1", "Motor2"]).'),
    from: zod_1.z.string().optional().nullable().describe("Optional start timestamp in ISO 8601 format (e.g., 2025-08-23T12:00:00Z)"),
    to: zod_1.z.string().optional().nullable().describe("Optional end timestamp in ISO 8601 format (e.g., 2025-08-23T13:00:00Z)"),
    limit: zod_1.z.number().int().min(1).max(10000).optional().default(50).describe("Maximum number of results to return (1–10000)"),
});
class AlarmHistoryTool extends StructuredTool {
    constructor(scadaHelper) {
        super();
        this.name = "get_history_of_alarms";
        this.description = "Fetch historical alarm events (past alarms, alarm logs, alarm history).";
        this.schema = AlarmHistoryToolSchema;
        this.scadaHelper = scadaHelper;
    }
    async _call(input) {
        const [data, conf] = await this.scadaHelper.getHistoricalAlarms(input);
        return [data, conf];
    }
}
AlarmHistoryTool.schema = AlarmHistoryToolSchema;
// ---------------- Graph State ----------------
const GraphState = langgraph_1.Annotation.Root({
    query: (0, langgraph_1.Annotation)({ value: (_, v) => v, default: () => "" }),
    messages: (0, langgraph_1.Annotation)({ value: (_, v) => v ?? [], default: () => [] }),
    toolHits: (0, langgraph_1.Annotation)({ value: (_, v) => v ?? [], default: () => [] }),
    intent: (0, langgraph_1.Annotation)({
        value: (_, v) => v ?? { kind: "unknown", needs: [], target: undefined },
        default: () => ({ kind: "unknown", needs: [], target: undefined }),
    }),
    answer: (0, langgraph_1.Annotation)({ value: (_, v) => v ?? "", default: () => "" }),
    range: (0, langgraph_1.Annotation)({
        value: (_, v) => v ?? { from: "", to: "" },
        default: () => ({ from: "", to: "" }),
    }),
    docs: (0, langgraph_1.Annotation)({
        value: (_, v) => v ?? [],
        default: () => [],
    }),
    toolDetails: (0, langgraph_1.Annotation)({
        value: (_, v) => v ?? {},
        default: () => ({}),
    }),
});
class BaseModel {
    static setDefaultTemperature(temperature) {
        BaseModel.defaultTemperature = temperature;
    }
    constructor(temperature) {
        this.checkpointer = new langgraph_1.MemorySaver();
        this.lang = "en";
        this.chatHistory = null;
        this._temperature = temperature;
    }
    get temperature() {
        return this._temperature ?? BaseModel.defaultTemperature;
    }
    // eslint-disable-next-line @typescript-eslint/no-empty-function
    async init() { }
    async build(configObj, tenantId, embeddingModel, vectorDB) {
        this.configObj = configObj;
        this.tenantId = tenantId;
        this.embeddingModel = embeddingModel;
        this.vectorDB = vectorDB;
        this.scadaHelper = new scadaHelpers_1.default(configObj, this.tenantId, this.embeddingModel, this.llm, this.vectorDB);
        // Example: mapping between classification "needs" and actual tool instances
        const toolMap = {
            docs: new DocsTool(this.scadaHelper),
            live: new LiveTool(this.scadaHelper),
            history: new HistoryTool(this.scadaHelper),
            live_alarms: new LiveAlarmsTool(this.scadaHelper),
            history_alarms: new AlarmHistoryTool(this.scadaHelper),
            datetime: new CurrentDateTool(),
            system_status: new SystemStatusTool(this.scadaHelper),
        };
        // --- Tools ---
        const tools = Object.values(toolMap);
        this.modelWithTools = this.llm.bindTools(tools);
        // ---------------- Nodes ----------------
        const ingest = async (state) => ({
            ...state,
            messages: [...(state.messages ?? []), { role: "user", content: state.query }],
        });
        const route = async (state) => {
            if (this.wsCallback) {
                this.wsCallback({ command: "status", message: "thinking" });
            }
            const history = state.messages.slice(-6);
            const user = state.query;
            let langSynonyms = promptSynonyms_1.promptSynonyms[this.lang];
            let s_docs = "";
            let s_history = "";
            let s_history_alarms = "";
            let s_live = "";
            let s_live_alarms = "";
            let s_datetime = "";
            let s_system_status = "";
            if (langSynonyms && this.lang !== "en") {
                s_docs = JSON.stringify(langSynonyms.docs).substring(1).slice(0, -1);
                s_history = JSON.stringify(langSynonyms.history).substring(1).slice(0, -1);
                s_history_alarms = JSON.stringify(langSynonyms.history_alarms).substring(1).slice(0, -1);
                s_live = JSON.stringify(langSynonyms.live).substring(1).slice(0, -1);
                s_live_alarms = JSON.stringify(langSynonyms.live_alarms).substring(1).slice(0, -1);
                s_datetime = JSON.stringify(langSynonyms.datetime).substring(1).slice(0, -1);
                s_system_status = JSON.stringify(langSynonyms.system_status).substring(1).slice(0, -1);
            }
            const prompt = [
                {
                    role: "system",
                    content: `Classify user intent for SCADA assistant.

Return JSON:
{
  "kind": "classification label for intent",
  "needs": ["docs" | "live" | "history" | "live_alarms" | "history_alarms" | "datetime" | "system_status"],
  "target"?: string // optional, e.g., a specific tag or alarm type
}

Rules:
- "docs" → ONLY for questions about SCADA documentation (manuals, guides, troubleshooting, configuration reference). Never use "docs" if the user asks to show, fetch, or list real data (tags, alarms, logs, history).
- "live" → for current real-time tag values.
- "history" → ONLY for historical process tag values (numeric/analog values over time). Do NOT use this for alarms. If the word "alarm" appears, always use "history_alarms".
- "live_alarms" → for active/live alarms (currently in alarm state).
- "history_alarms" → for historical alarm events (past alarms, alarm logs).
- "datetime" → when user refers to "today", "now", "yesterday".
- "system_status" → for user questions about overall system health, status summary, or general operational state.

Resolve ambiguity using conversation history if needed.
If both could apply, always prefer the more specific alarm category:
- "history_alarms" over "history"
- "live_alarms" over "live"

Synonyms mapping (multilingual):
- "docs": ["manual", "guide", "instructions", "explain", "how to", "configuration", 
           ${s_docs}] // ${this.lang}

- "live": ["current value", "live value", "real-time", "status", 
          ${s_live}] // ${this.lang}

- "history": ["history", "trends", "graphs", "charts", 
              ${s_history}] // ${this.lang}

- "live_alarms": ["alarms", "warnings", "faults", "errors", "alerts", 
             ${s_live_alarms}] // ${this.lang}

- "history_alarms": [
    "historical alarms", "past alarms", "alarm logs", "alarm history",
    ${s_history_alarms}] // ${this.lang}

- "datetime": ["today", "now", "yesterday", "last week", "last hour",
               ${s_datetime}] // ${this.lang}

- "system_status": ["system status", "health", "problems", "working fine", 
                    ${s_system_status}] // ${this.lang}

Examples:
User: "Show me live tags Motor1 and Motor2"
→ { "kind": "query live tags", "needs": ["live"], "target": ["Motor1", "Motor2"] }

User: "Open the documentation for alarm handling"
→ { "kind": "docs search", "needs": ["docs"], "target": "alarm handling" }

User: "Show me warnings"
→ { "kind": "active alarms", "needs": ["live_alarms"], "target": ["alarms"] }

User: "Show me alarms for Motor1 and Motor2 from yesterday"
→ { "kind": "historical alarms", "needs": ["history_alarms"], "target": ["Motor1", "Motor2"] }

User: "Give me the history of all alarms"
→ { "kind": "historical alarms", "needs": ["history_alarms"], "target": ["alarms"] }

User: "Explain alarm history for turbine"
→ { "kind": "historical alarms", "needs": ["history_alarms"], "target": ["turbine"] }

User: "Give me temperature trend last week"
→ { "kind": "historical values", "needs": ["history"], "target": ["Temperature"] }

User: "How does redundancy work?"
→ { "kind": "docs search", "needs": ["docs"], "target": "redundancy" }

User: "What is the system status?"
→ { "kind": "system status", "needs": ["system_status"], "target": "overall system" }

User: "Is everything working fine?"
→ { "kind": "system status", "needs": ["system_status"], "target": "overall system" }

User: "Any problems right now?"
→ { "kind": "system status", "needs": ["system_status"], "target": "overall system" }
`,
                },
                { role: "system", content: "History:\n" + JSON.stringify(history) },
                { role: "user", content: user },
            ];
            try {
                const res = await this.llm.invoke(prompt);
                const txt = res.content ?? "";
                const match = txt.match(/\{[\s\S]*\}/);
                if (match)
                    return { ...state, intent: JSON.parse(match[0]) };
            }
            catch (e) {
                console.error("Router failed", e);
            }
            return { ...state, intent: { kind: "fallback", needs: ["docs"] } };
        };
        const planAndCall = async (state) => {
            const results = [];
            let datetimeResult = null;
            let dateParsingRun = false;
            for (const need of state.intent.needs) {
                const tool = toolMap[need];
                if (!tool)
                    continue;
                console.log("NEED:" + need);
                if (this.wsCallback) {
                    if (need === "live") {
                        this.wsCallback({ command: "status", message: "reading live values" });
                    }
                    if (need === "live_alarms") {
                        this.wsCallback({ command: "status", message: "reading live alarms" });
                    }
                    if (need === "history_alarms") {
                        this.wsCallback({ command: "status", message: "reading historical alarms" });
                    }
                    if (need === "docs") {
                        this.wsCallback({ command: "status", message: "searching documentation" });
                    }
                    if (need === "history") {
                        this.wsCallback({ command: "status", message: "reading tag history" });
                    }
                    if (need === "system_status") {
                        this.wsCallback({ command: "status", message: "retrieving system status" });
                    }
                }
                // Base input
                let input = { query: state.query };
                // If this is a history-type tool, merge datetime ranges
                if ((need === "history" || need === "history_alarms") && !dateParsingRun) {
                    dateParsingRun = true;
                    datetimeResult = await this.scadaHelper.resolveRelativeRangeWithLLM(state.query);
                    state.range.from = datetimeResult.from;
                    state.range.to = datetimeResult.to;
                    //pokud neni datum musime prohledat historii otazek
                    if (!datetimeResult.from || !datetimeResult.to) {
                        for (let j = this.chatHistory.chat.length - 1; j >= 0; j--) {
                            let range = this.chatHistory.chat[j].sources.range;
                            if (range && range.from && range.to) {
                                state.range.from = range.from;
                                state.range.to = range.to;
                            }
                        }
                    }
                }
                if (need === "history" || need === "history_alarms") {
                    input = { ...input, ...datetimeResult };
                }
                if (need === "history" || need === "history_alarms" || need === "live" || need === "live_alarms") {
                    let tags = state.intent.target;
                    if (tags && typeof tags === "string") {
                        tags = [tags];
                    }
                    if (need === "history" || need === "live") {
                        input.tags = tags;
                    }
                    if (need === "history_alarms" || need === "live_alarms") {
                        input.alarms = tags;
                    }
                }
                const [output, data] = await tool.invoke(input);
                if (need === "docs") {
                    state.docs = data;
                }
                if (need === "history" || need === "live" || need === "history_alarms" || need === "live_alarms") {
                    state.toolDetails[need] = data;
                }
                results.push({ source: need, data: output });
            }
            return { ...state, toolHits: results };
        };
        const synthesize = async (state) => {
            const history = state.messages.slice(-6);
            const context = JSON.stringify(state.toolHits ?? [], null, 2);
            //const messages = [{ role: "system", content: "You are a SCADA assistant." }, ...(state.messages ?? []), { role: "system", content: `Context:\n${context}` }];
            const messages = [
                {
                    role: "system",
                    content: `You are a SCADA assistant. Speak directly to the operator.
Never show tool calls, internal thoughts, or reasoning.
Only return the operator-facing answer.`,
                },
                { role: "system", content: `Context:\n${context}` },
                ...history,
            ];
            if (this.wsCallback) {
                this.wsCallback({ command: "status", message: "generating response" });
            }
            const stream = await this.llm.stream(messages);
            //const stream = await this.modelWithTools.stream(messages);
            const { think, reply } = await (0, tools_1.separateThinkWithCallback)(stream, (chunk) => {
                //process.stdout.write(chunk); // live streaming to console
                if (this.callback)
                    this.callback(chunk);
            });
            console.log("\n--- Hidden reasoning ---");
            console.log(think);
            console.log("\n--- Final reply ---");
            console.log(reply);
            let res = reply;
            //const answer = (res as any)?.content ?? "";
            const answer = res;
            return {
                ...state,
                messages: [...(state.messages ?? []), { role: "assistant", content: answer }],
                answer,
            };
        };
        const finalize = async (state) => state;
        // ---------------- Graph ----------------
        this.graph = new langgraph_1.StateGraph(GraphState)
            .addNode("ingest", ingest)
            .addNode("route", route)
            .addNode("planAndCall", planAndCall)
            .addNode("synthesize", synthesize)
            .addNode("finalize", finalize)
            .addEdge(langgraph_1.START, "ingest")
            .addEdge("ingest", "route")
            .addEdge("route", "planAndCall")
            .addEdge("planAndCall", "synthesize")
            .addEdge("synthesize", "finalize")
            .addEdge("finalize", langgraph_1.END)
            .compile({ checkpointer: this.checkpointer });
        return this.graph;
    }
    // ---------------- Run ----------------
    async ask(lang, sessionId, query, messages, chatHistory, callback, wsStatus) {
        this.callback = callback;
        this.wsCallback = wsStatus;
        this.lang = lang;
        this.chatHistory = chatHistory;
        const initial = {
            query,
            messages: messages,
            toolHits: [],
            intent: { kind: "unknown", needs: [], target: "" },
            answer: "",
            range: { from: "", to: "" },
            docs: [],
            toolDetails: {},
        };
        this.scadaHelper.setLang(lang);
        return await this.graph.invoke(initial, {
            configurable: { thread_id: sessionId },
        });
    }
    async query(messages, callback) {
        // Run LLM implementation in subclass
        const response = await this.runQuery(messages, callback);
        return response;
    }
}
exports.BaseModel = BaseModel;
