"use strict";
var __importDefault = (this && this.__importDefault) || function (mod) {
    return (mod && mod.__esModule) ? mod : { "default": mod };
};
Object.defineProperty(exports, "__esModule", { value: true });
exports.LibSqlDb = void 0;
const debug_1 = __importDefault(require("debug"));
const client_1 = require("@libsql/client");
const rag_tools_1 = require("../../rag-tools");
class LibSqlDb {
    constructor({ path, tableName }) {
        this.debug = (0, debug_1.default)("embedjs:vector:LibSqlDb");
        this.tableName = tableName ?? "vectors";
        this.client = (0, client_1.createClient)({
            url: `file:${path}`,
        });
    }
    async init({ dimensions }) {
        await this.client.execute(`CREATE TABLE IF NOT EXISTS ${this.tableName} (
            id              TEXT PRIMARY KEY,
            pageContent     TEXT UNIQUE,
            uniqueLoaderId  TEXT NOT NULL,
            source          TEXT NOT NULL,
            vector          F32_BLOB(${dimensions}),
            type        TEXT
        );`);
    }
    async insertChunks(type, chunks) {
        const batch = chunks.map((chunk) => {
            return {
                sql: `INSERT OR IGNORE INTO ${this.tableName} (id, pageContent, uniqueLoaderId, source, vector, type)
            VALUES (?, ?, ?, ?, vector32('[${chunk.vector.join(",")}]'), ?);`,
                args: [chunk.metadata.id, chunk.pageContent, chunk.metadata.uniqueLoaderId, chunk.metadata.source, type],
            };
        });
        this.debug(`Executing batch - ${(0, rag_tools_1.truncateCenterString)(JSON.stringify(batch), 1000)}`);
        const result = await this.client.batch(batch, "write");
        return result.reduce((a, b) => a + b.rowsAffected, 0);
    }
    async getEmbeddingsUniqueIDs(type, returnType) {
        let statement = `SELECT uniqueLoaderId, source 
            FROM ${this.tableName}
            where type = '${type}'`;
        const results = await this.client.execute(statement);
        let result = results.rows;
        if (returnType === "source") {
            let res = {};
            for (let r of result) {
                if (!res[r.source]) {
                    res[r.source] = r.uniqueLoaderId;
                }
            }
            return res;
        }
        else if (returnType === "list") {
            let ids = [];
            let sources = [];
            for (let r of result) {
                if (!ids.includes(r.uniqueLoaderId)) {
                    sources.push(r.source);
                    ids.push(r.uniqueLoaderId);
                }
            }
            return [ids, sources];
        }
        else {
            return null;
        }
        return [];
    }
    async similaritySearch(query, k, docLimit, type) {
        let where = "";
        if (type) {
            where = " WHERE type='" + type + "' ";
        }
        let statement = `SELECT id, pageContent, uniqueLoaderId, source, type,
                vector_distance_cos(vector, vector32('[${query.join(",")}]')) as distance
            FROM ${this.tableName} ${where}
            ORDER BY vector_distance_cos(vector, vector32('[${query.join(",")}]')) ASC
            LIMIT ${k};`;
        if (docLimit) {
            if (type) {
                where = " AND type='" + type + "' ";
            }
            statement = `
    WITH ranked_results AS (
        SELECT id, pageContent, uniqueLoaderId, source, type,
               vector_distance_cos(vector, vector32('[${query.join(",")}]')) as distance,
               (SELECT COUNT(*) 
                FROM ${this.tableName} AS sub 
                WHERE sub.uniqueLoaderId = main.uniqueLoaderId 
                AND sub.id <= main.id) AS row_num
        FROM ${this.tableName} AS main 
        ORDER BY distance ASC
    )
    SELECT id, pageContent, uniqueLoaderId, source, type, distance
    FROM ranked_results
    WHERE row_num <= ${docLimit} ${where}
    ORDER BY distance ASC
    LIMIT ${k};
`;
        }
        this.debug(`Executing statement - ${(0, rag_tools_1.truncateCenterString)(statement, 700)}`);
        const results = await this.client.execute(statement);
        return results.rows.map((row) => {
            //const metadata = JSON.parse(result.metadata.toString());
            return {
                score: 1 - row.distance,
                pageContent: row.pageContent,
                metadata: {
                    source: row.source,
                    id: row.id,
                    uniqueLoaderId: row.uniqueLoaderId,
                    type: row.type,
                },
            };
        });
    }
    async getVectorCount() {
        const statement = `SELECT count(id) as count FROM ${this.tableName};`;
        this.debug(`Executing statement - ${statement}`);
        const results = await this.client.execute(statement);
        return Number.parseInt(results.rows[0].count.toString());
    }
    async deleteKeys(uniqueLoaderId) {
        await this.client.execute(`DELETE FROM ${this.tableName} WHERE
           uniqueLoaderId = '${uniqueLoaderId}';`);
        return true;
    }
    async reset() {
        await this.client.execute(`DELETE FROM ${this.tableName};`);
    }
}
exports.LibSqlDb = LibSqlDb;
