"use strict";
var __importDefault = (this && this.__importDefault) || function (mod) {
    return (mod && mod.__esModule) ? mod : { "default": mod };
};
Object.defineProperty(exports, "__esModule", { value: true });
exports.LibMongoDb = void 0;
//import { Collection, MongoClient } from "MongoDb";
const mongodb_1 = require("mongodb");
const debug_1 = __importDefault(require("debug"));
class LibMongoDb {
    constructor({ connectionString, certFile, dbName, collectionName }) {
        this.debug = (0, debug_1.default)("embedjs:vector:LibMongoDb");
        this.collectionName = collectionName ?? LibMongoDb.COLLECTION_NAME;
        this.dbName = dbName ?? LibMongoDb.DEFAULT_DB_NAME;
        let mongoConf = {
            serverSelectionTimeoutMS: 5000,
            tlsCertificateKeyFile: certFile,
            tlsAllowInvalidCertificates: true,
        };
        if (process.env.debug) {
            mongoConf.directConnection = true;
        }
        this.client = new mongodb_1.MongoClient(connectionString, mongoConf);
    }
    //private getIndexName(indexName: string) {
    //  return LibMongoDb.INDEX_PREFIX + indexName;
    //}
    hasIndex(indexes, name) {
        for (let index of indexes) {
            if (index.name.startsWith(name)) {
                return true;
            }
        }
        return false;
    }
    async init({ dimensions }) {
        this.debug("Connecting to database");
        await this.client.connect();
        const database = this.client.db(this.dbName);
        this.debug("Connected");
        const collections = await database.collections({ nameOnly: true, authorizedCollections: true });
        if (!collections.some((x) => x.collectionName === this.collectionName)) {
            this.debug(`Creating collection '${this.collectionName}'`);
            await database.createCollection(this.collectionName);
        }
        this.collection = database.collection(this.collectionName);
        this.debug("Collection reference obtained");
        let indexes = await this.collection.listSearchIndexes(LibMongoDb.VECTOR_FIELD_NAME).toArray();
        if (!this.hasIndex(indexes, LibMongoDb.VECTOR_FIELD_NAME)) {
            this.debug(`Creating vector search index '${LibMongoDb.VECTOR_FIELD_NAME}'`);
            await this.collection.createSearchIndex({
                name: LibMongoDb.VECTOR_FIELD_NAME,
                type: "vectorSearch",
                definition: {
                    fields: [
                        {
                            type: "vector",
                            numDimensions: dimensions,
                            path: LibMongoDb.VECTOR_FIELD_NAME,
                            similarity: "cosine",
                        },
                    ],
                },
            });
        }
        if (!this.hasIndex(indexes, LibMongoDb.LOADER_FIELD_NAME)) {
            this.debug(`Creating index '${LibMongoDb.LOADER_FIELD_NAME}'`);
            await this.collection.createIndex({ [LibMongoDb.LOADER_FIELD_NAME]: 1 });
        }
        if (!this.hasIndex(indexes, LibMongoDb.UNIQUE_FIELD_NAME)) {
            this.debug(`Creating index '${LibMongoDb.UNIQUE_FIELD_NAME}'`);
            await this.collection.createIndex({ [LibMongoDb.UNIQUE_FIELD_NAME]: 1 });
        }
        if (!this.hasIndex(indexes, LibMongoDb.TYPE_FIELD_NAME)) {
            this.debug(`Creating index '${LibMongoDb.TYPE_FIELD_NAME}'`);
            await this.collection.createIndex({ [LibMongoDb.TYPE_FIELD_NAME]: 1 });
        }
        this.debug("All indexes created / exist already");
    }
    async insertChunks(type, chunks) {
        this.debug(`Inserting ${chunks.length} chunks`);
        const insertResult = await this.collection.insertMany(chunks.map((chunk) => {
            let metadata = chunk.metadata;
            const uniqueLoaderId = metadata.uniqueLoaderId;
            //delete metadata.uniqueLoaderId;
            const source = metadata.source;
            //delete metadata.source;
            const id = metadata.id;
            //delete metadata.id;
            return {
                [LibMongoDb.UNIQUE_FIELD_NAME]: id,
                [LibMongoDb.VECTOR_FIELD_NAME]: chunk.vector,
                [LibMongoDb.LOADER_FIELD_NAME]: uniqueLoaderId,
                [LibMongoDb.TYPE_FIELD_NAME]: type,
                pageContent: chunk.pageContent,
                source: source,
                //metadata,
            };
        }));
        return insertResult.insertedCount;
    }
    async getEmbeddingsUniqueIDs(type, returnType) {
        if (returnType === "source") {
            let result = await this.collection.find({ type: type }, { projection: { _id: 0, [LibMongoDb.LOADER_FIELD_NAME]: 1, source: 1 } }).toArray();
            let res = {};
            for (let r of result) {
                if (!res[r.source]) {
                    res[r.source] = r[LibMongoDb.LOADER_FIELD_NAME];
                }
            }
            return res;
        }
        else if (returnType === "list") {
            let result = await this.collection.find({ type: type }, { projection: { _id: 0, [LibMongoDb.LOADER_FIELD_NAME]: 1, source: 1 } }).toArray();
            let ids = [];
            let sources = [];
            for (let r of result) {
                if (!ids.includes(r[LibMongoDb.LOADER_FIELD_NAME])) {
                    sources.push(r.source);
                    ids.push(r[LibMongoDb.LOADER_FIELD_NAME]);
                }
            }
            return [ids, sources];
        }
        else {
            return null;
        }
    }
    async similaritySearch(query, k, limitsPerDoc, type) {
        this.debug(`Searching with query dimension ${query.length}`);
        let searchArray = [];
        if (type) {
            searchArray.push({
                [LibMongoDb.TYPE_FIELD_NAME]: type,
            });
        }
        searchArray.push({
            $vectorSearch: {
                index: LibMongoDb.VECTOR_FIELD_NAME,
                path: LibMongoDb.VECTOR_FIELD_NAME,
                numCandidates: k, //25 * k,
                queryVector: query,
                limit: k,
            },
        });
        searchArray.push({
            $project: {
                _id: 0,
                source: 1,
                metadata: 1,
                pageContent: 1,
                [LibMongoDb.TYPE_FIELD_NAME]: 1,
                [LibMongoDb.UNIQUE_FIELD_NAME]: 1,
                [LibMongoDb.LOADER_FIELD_NAME]: 1,
                score: {
                    $meta: "vectorSearchScore",
                },
            },
        });
        let result = await this.collection.aggregate(searchArray).toArray();
        let data = result.map((row) => {
            return {
                score: row.score,
                pageContent: row.pageContent,
                metadata: {
                    source: row.source,
                    id: row[LibMongoDb.UNIQUE_FIELD_NAME],
                    uniqueLoaderId: row[LibMongoDb.LOADER_FIELD_NAME],
                    type: row[LibMongoDb.TYPE_FIELD_NAME],
                },
            };
        });
        return data;
    }
    async getVectorCount() {
        return this.collection.countDocuments();
    }
    async deleteKeys(uniqueLoaderId) {
        this.debug(`Deleting keys tied to loader '${uniqueLoaderId}'`);
        const result = await this.collection.deleteMany({ [LibMongoDb.LOADER_FIELD_NAME]: uniqueLoaderId });
        return !!result.deletedCount;
    }
    async reset() {
        await this.collection.deleteMany({});
    }
}
exports.LibMongoDb = LibMongoDb;
LibMongoDb.DEFAULT_DB_NAME = "embedjs";
LibMongoDb.COLLECTION_NAME = "vectors";
LibMongoDb.VECTOR_FIELD_NAME = "vectors";
LibMongoDb.LOADER_FIELD_NAME = "loaderId";
LibMongoDb.UNIQUE_FIELD_NAME = "uniqueId";
LibMongoDb.TYPE_FIELD_NAME = "type";
